# Archivo: CotizFormQueryIng.py
# Ruta: src\App\Operaciones_Module\Ingenieria\Cotiz\CotizFormQueryIng.py
# Descripción: Módulo de logica principal para consultar formularios de cotización en el módulo de Ingeniería.
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from flask import jsonify, render_template, request, send_file, session, after_this_request
from flask_mail import Mail
from Consultas_SQL.Operaciones.Ingenieria.Cotiz.CotizFormQueryIngSQL import get_tasks_list

from App.Operaciones_Module.Ingenieria.Cotiz.FormularioPDFService import FormularioPDFService

import os
def funcion_CotizFormQueryIng(app, mail=None):
    """
    Agrupa las rutas para el módulo de consulta de Tareas de Cotización.
    (Se añade 'mail=None' para mantener consistencia con otros módulos
    como AccessDistributors, aunque no se use aquí).
    """
    @app.route("/Operaciones_Module/Ingenieria/Cotiz/list", methods=["GET"])
    def list_cotiz_tasks():
        """
        SIRVE LOS DATOS A JAVASCRIPT (JSON).
        Esta es la ruta que el JavaScript llama (fetch) para llenar la tabla.
        Debe coincidir exactamente con la ruta en CotizFormQueryIng.js.
        """
        try:
            # 1. Llama a la función del archivo SQL
            tasks = get_tasks_list()
            
            # 2. Devuelve los datos en formato JSON
            return jsonify(tasks), 200
        except Exception as e:
            print(f"[Error en ruta /Operaciones_Module/Ingenieria/Cotiz/list] {e}")
            # Devolvemos un error en formato JSON
            return jsonify({"error": str(e), "data": []}), 500
        
    
    # ============================================
    # 3. NUEVO ENDPOINT: GENERAR PDF CON ADJUNTOS
    # ============================================
    
    @app.route('/Operaciones_Module/Ingenieria/Cotiz/generar-pdf', methods=['GET'])
    def generar_pdf_formulario():
        """
        Genera y descarga PDF de formulario con archivos adjuntos.
        
        ¿Por qué este endpoint?
        - Centraliza la lógica de generación de PDFs
        - Permite parámetros configurables (con/sin adjuntos, con/sin ZIP)
        - Retorna el archivo directamente para descarga inmediata
        
        Parámetros URL (query params):
            - form_id: ID del formulario (REQUERIDO) - Ejemplo: '1002-1-FormGral'
            - adjuntos: 'true'|'false' - Incluir archivos adjuntos (default: true)
            - zip: 'true'|'false' - Crear archivo ZIP (default: true)
        
        Ejemplos de uso:
            GET /Operaciones_Module/Ingenieria/Cotiz/generar-pdf?form_id=1002-1-FormGral
            GET /Operaciones_Module/Ingenieria/Cotiz/generar-pdf?form_id=1002-1-FormGral&adjuntos=false
            GET /Operaciones_Module/Ingenieria/Cotiz/generar-pdf?form_id=1002-1-FormGral&zip=false
        
        Returns:
            File: PDF o ZIP para descarga directa
            HTML: Página de error si algo falla
        """
        
        # ========================================
        # PASO 1: OBTENER Y VALIDAR PARÁMETROS
        # ========================================
        
        # Obtener form_id de los parámetros de la URL
        form_id = request.args.get('form_id')
        
        # Validar que form_id existe
        if not form_id:
            return """
            <html>
                <body style="font-family: Arial; padding: 40px; text-align: center;">
                    <h1 style="color: #e74c3c;">⚠️ Parámetro Faltante</h1>
                    <p>Debe proporcionar el parámetro <code>form_id</code></p>
                    <p style="color: #666;">
                        Ejemplo: <code>/generar-pdf?form_id=1002-1-FormGral</code>
                    </p>
                    <hr style="margin: 30px 0;">
                    <button onclick="history.back()" style="padding: 10px 20px; font-size: 16px;">
                        ← Regresar
                    </button>
                </body>
            </html>
            """, 400  # 400 = Bad Request
        
        # Obtener parámetros opcionales (con valores por defecto)
        incluir_adjuntos = request.args.get('adjuntos', 'true').lower() == 'true'
        crear_zip = request.args.get('zip', 'true').lower() == 'true'
        
        
        # ========================================
        # PASO 2: LOGGING PARA DEBUGGING
        # ========================================
        
        print("=" * 80)
        print("🚀 ENDPOINT: generar_pdf_formulario")
        print("=" * 80)
        print(f"📋 FormID solicitado: {form_id}")
        print(f"📎 Incluir adjuntos: {incluir_adjuntos}")
        print(f"📦 Crear ZIP: {crear_zip}")
        print(f"👤 UserID: {session.get('user_id', 'No autenticado')}")
        print("=" * 80)
        
        
        # ========================================
        # PASO 3: GENERAR PDF CON EL SERVICIO
        # ========================================
        
        try:
            # Llamar al servicio que hace todo el trabajo pesado
            resultado = FormularioPDFService.generar_pdf_con_adjuntos(
                form_id=form_id,
                nombre_archivo=None,  # Se genera automáticamente
                incluir_adjuntos=incluir_adjuntos,
                crear_zip=crear_zip
            )
            
            # ========================================
            # PASO 4: VALIDAR RESULTADO
            # ========================================
            
            if not resultado:
                # Si resultado es None, el FormID no existe
                print(f"❌ FormID no encontrado: {form_id}")
                
                return f"""
                <html>
                    <body style="font-family: Arial; padding: 40px; text-align: center;">
                        <h1 style="color: #e74c3c;">❌ FormID No Encontrado</h1>
                        <p>No se encontró el FormID: <strong>{form_id}</strong></p>
                        <p style="color: #666;">
                            El FormID no existe en la base de datos o no tiene datos asociados.
                        </p>
                        <hr style="margin: 30px 0;">
                        <h3>💡 Sugerencias:</h3>
                        <ul style="text-align: left; max-width: 600px; margin: 0 auto;">
                            <li>Verifica que el FormID sea correcto</li>
                            <li>Revisa la consola del servidor para más detalles</li>
                            <li>Contacta al administrador del sistema</li>
                        </ul>
                        <br>
                        <button onclick="history.back()" style="padding: 10px 20px; font-size: 16px;">
                            ← Regresar
                        </button>
                    </body>
                </html>
                """, 404  # 404 = Not Found
            
            
            # ========================================
            # PASO 5: DECIDIR QUÉ ARCHIVO ENVIAR
            # ========================================
            
            # Prioridad: Si hay ZIP, enviar ZIP. Si no, enviar solo PDF.
            if resultado['zip'] and os.path.exists(resultado['zip']):
                # Caso 1: Hay ZIP (PDF + adjuntos empaquetados)
                archivo_enviar = resultado['zip']
                nombre_descarga = os.path.basename(archivo_enviar)
                directorio_eliminar = os.path.dirname(archivo_enviar)
                mimetype = 'application/zip'
                
                print(f"📦 Enviando ZIP: {nombre_descarga}")
                
            else:
                # Caso 2: Solo PDF (sin adjuntos o sin ZIP)
                archivo_enviar = resultado['pdf']
                directorio_eliminar = resultado['directorio']
                nombre_descarga = f'formulario_{form_id.replace("-", "_")}.pdf'
                mimetype = 'application/pdf'
                
                print(f"📄 Enviando PDF: {nombre_descarga}")
                
                
                    # ✅ ELIMINAR DESPUÉS DE ENVIAR
            @after_this_request
            def eliminar_archivos_temporales(response):
                """
                Se ejecuta DESPUÉS de enviar el archivo al cliente
                Elimina la carpeta temporal con todos los archivos
                """
                try:
                    import shutil
                    
                    if os.path.exists(directorio_eliminar):
                        shutil.rmtree(directorio_eliminar)
                        print(f"🗑️  Carpeta eliminada: {directorio_eliminar}")
                        
                except Exception as e:
                    print(f"⚠️  No se pudo eliminar carpeta temporal: {e}")
                
                return response
            
            # ========================================
            # PASO 6: ENVIAR ARCHIVO AL CLIENTE
            # ========================================
            
            # ✅ Leer el archivo completamente a memoria antes de enviarlo
            with open(archivo_enviar, 'rb') as f:
                archivo_bytes = f.read()

            from io import BytesIO
            archivo_buffer = BytesIO(archivo_bytes)
            archivo_buffer.seek(0)

            return send_file(
                archivo_buffer,
                mimetype=mimetype,
                as_attachment=True,
                download_name=nombre_descarga
            )
            
        
        # ========================================
        # PASO 7: MANEJO DE ERRORES
        # ========================================
        
        except Exception as e:
            import traceback
            error_detail = traceback.format_exc()
            
            print()
            print("=" * 80)
            print("❌ ERROR CRÍTICO EN ENDPOINT")
            print("=" * 80)
            print(f"FormID: {form_id}")
            print(f"Error: {e}")
            print()
            print("Stack trace completo:")
            print(error_detail)
            print("=" * 80)
            
            # Retornar página HTML de error (más amigable que JSON)
            return f"""
            <html>
                <head>
                    <title>Error - Sistema IGSA</title>
                    <style>
                        body {{
                            font-family: 'Courier New', monospace;
                            padding: 40px;
                            background: #1e1e1e;
                            color: #d4d4d4;
                        }}
                        .error-box {{
                            background: #252526;
                            padding: 20px;
                            border-radius: 5px;
                            border-left: 4px solid #f48771;
                            margin: 20px 0;
                        }}
                        .error-title {{
                            color: #f48771;
                            font-size: 24px;
                            margin-bottom: 20px;
                        }}
                        .error-details {{
                            background: #1e1e1e;
                            padding: 15px;
                            border-radius: 3px;
                            overflow-x: auto;
                            margin-top: 15px;
                        }}
                        button {{
                            background: #0e639c;
                            color: white;
                            border: none;
                            padding: 10px 20px;
                            font-size: 16px;
                            cursor: pointer;
                            border-radius: 3px;
                            margin-top: 20px;
                        }}
                        button:hover {{
                            background: #1177bb;
                        }}
                    </style>
                </head>
                <body>
                    <div class="error-box">
                        <div class="error-title">❌ Error del Sistema</div>
                        <p><strong>FormID solicitado:</strong> {form_id}</p>
                        <p><strong>Tipo de error:</strong> {type(e).__name__}</p>
                        <p><strong>Mensaje:</strong> {str(e)}</p>
                        
                        <div class="error-details">
                            <strong>Stack Trace:</strong>
                            <pre>{error_detail}</pre>
                        </div>
                        
                        <button onclick="history.back()">← Regresar</button>
                    </div>
                </body>
            </html>
            """, 500  # 500 = Internal Server Error

