# Archivo: DMAccessEmployee.py
# Ruta: src\App\SupyCtrol_Module\DataMaster\DMAccessEmployee.py
# Lenguaje: Python con Flask

from flask import jsonify, request, render_template
from Consultas_SQL.SupYCtrol.DataMaster.DMAccessEmployeeSQL import (
    get_AccessEmployees, get_all_request_types, get_roles_by_filters, 
    get_all_companies, get_divisions_by_company, get_departments_by_division, 
    get_info, check_if_email_exists, update_info, update_user_request_rejected
)
from flask_mail import Message
import datetime
import pytz
from App.Security_Module.UserPassword import send_activation_email

def funcion_AccessEmployees(app, mail):
    """
    Agrupamos todas las funciones que importamos en la aplicación Flask.
    """
    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/list", methods=["GET"])
    def list_AccessEmployees():
        """Retorna la lista de empleados pendientes de aceptación."""
        try:
            AccessEmployees = get_AccessEmployees()
            return jsonify(AccessEmployees), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/request-types", methods=["GET"])
    def get_request_types():
        """Retorna la lista de tipos de formulario disponibles."""
        try:
            request_types = get_all_request_types()
            return jsonify(request_types), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500
        

    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/companies", methods=["GET"])
    def get_companies():
        """Retorna la lista de compañías disponibles."""
        try:
            companies = get_all_companies()
            return jsonify(companies), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/divisions", methods=["GET"])
    def get_divisions():
        """Retorna las divisiones disponibles para una compañía específica."""
        try:
            company_id = request.args.get("companyId")
            if not company_id:
                return jsonify({"error": "CompanyID es requerido"}), 400
                
            divisions = get_divisions_by_company(company_id)
            return jsonify(divisions), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/departments", methods=["GET"])
    def get_departments():
        """Retorna los departamentos disponibles para una compañía y división específicas."""
        try:
            company_id = request.args.get("companyId")
            division_id = request.args.get("divisionId")
            
            if not company_id or not division_id:
                return jsonify({"error": "CompanyID y DivisionID son requeridos"}), 400
                
            departments = get_departments_by_division(company_id, division_id)
            return jsonify(departments), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500        
        
    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/filtered-roles", methods=["GET"])
    def get_filtered_roles():
        """Retorna la lista de roles disponibles filtrados por compañía, división y departamento."""
        try:
            company_id = request.args.get("companyId", "")
            division_id = request.args.get("divisionId", "")
            departament_id = request.args.get("departamentId", "")
            
            roles = get_roles_by_filters(company_id, division_id, departament_id)
            return jsonify(roles), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500    
        
    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/getinfo", methods=["GET"])
    def getinfo():
        """Obtiene la información detallada de un empleado por su RequestID."""
        try:
            request_id = request.args.get("RequestID")
            
            print('ID del usuario: ', request_id)
            
            if not request_id:
                return jsonify({"error": "RequestID es requerido"}), 400

            InfoEmployee = get_info(request_id)
            
            print ("Información del empleado: ", InfoEmployee)
            
            return jsonify(InfoEmployee), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500
        
    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/check-email", methods=["GET"])
    def check_email_exists():
        """Verifica si un correo electrónico ya existe en la tabla Users."""
        try:
            email = request.args.get("email")
            
            if not email:
                return jsonify({"error": "Email es requerido"}), 400

            exists = check_if_email_exists(email)
            return jsonify({"exists": exists}), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/update", methods=["POST"])
    def updateinfo():
        """Actualiza la información de un empleado."""
        try:
            data = request.get_json()
            
            # print('Datos recibidos: ', data)
            
            if not data:
                return jsonify({"error": "No se recibieron datos"}), 400

            request_id = data.get("RequestID")
            if not request_id:
                return jsonify({"error": "RequestID es requerido"}), 400

            update_result = update_info(data)
            
            print('Resultado de la actualización: ', update_result)
            
            if not update_result or "user_id" not in update_result:
                return jsonify({"error": "No se pudo obtener el ID del usuario actualizado"}), 500
                
            print('ID del usuario actualizado: ', update_result["user_id"])
            
            # Llamar a send_activation_email para enviar email de activación
            token_result = send_activation_email(app, mail, update_result["user_id"])
            # token_result = "Mensaje enviado" or "Error al crear token/enviar emanil
            """
            if token_result == 'Error en la generacion de token':
                messagemail ="Hubo un error al enviar el correo contacte a soporte" #concatenar con userid
            else:
                messagemail = "Alta de usuario exitoso y Correo enviado"
            """
            # Si token_result es una tupla, significa que hubo un error
            if isinstance(token_result, tuple):
                # Obtener el objeto jsonify y el código de estado
                json_response, status_code = token_result
                # Extraer el contenido JSON del objeto jsonify
                error_data = json_response.get_json()
                # Devolver el error manteniendo el código de estado original
                return jsonify({
                    "message": "Hubo un error al enviar el correo de activación contacte a soporte carlos.velazquezm@igsa.com.mx ",
                    "error": error_data.get('error', 'Error desconocido')# Se obtiene la variable error del json
                }), status_code
            # Si no es una tupla, es una respuesta exitosa
            return jsonify({
                "message": "Información actualizada correctamente",
                "token_info": token_result.json if hasattr(token_result, 'json') else "Email enviado correctamente"
            }), 200

        except Exception as e:
            import traceback
            print(f"Error en al intentar resgistrar usuario: {e}")
            print(traceback.format_exc())
            return jsonify({"error": str(e)}), 500
            
            
    @app.route("/SupyCtrol_Module/DataMaster/DMAccessEmployee/reject", methods=["POST"])
    def reject_employee():
        """Rechaza una solicitud de empleado y envía notificación por correo."""
        try:
            data = request.get_json()
            employee_id = data.get("employeeId")
            comentario = data.get("comentario")  # Opcional
            
            # Validar datos
            if not employee_id:
                return jsonify({"error": "ID de empleado requerido"}), 400
            
            # Obtener información del empleado
            employee_info = get_info(employee_id)
            if not employee_info or "error" in employee_info:
                return jsonify({"error": "No se pudo obtener la información del empleado"}), 500
            
            # Actualizar estado de la solicitud en UserRequests
            timezone = pytz.timezone("America/Mexico_City")
            current_date = datetime.datetime.now(timezone).strftime('%Y-%m-%d %H:%M:%S')
            update_result = update_user_request_rejected(employee_id, current_date, comentario)
            if "error" in update_result:
                return jsonify(update_result), 500
            
            # Enviar correo al empleado
            email_result = enviar_correo_rechazo(employee_info, comentario)
            if "error" in email_result:
                # El correo falló, pero la solicitud fue rechazada correctamente
                return jsonify({
                    "message": "Solicitud rechazada, pero hubo un problema al enviar el correo electrónico",
                    "error_mail": email_result["error"]
                }), 200
            
            return jsonify({"message": "Solicitud rechazada correctamente y notificación enviada"}), 200
            
        except Exception as e:
            print(f"[Error en reject_employee] {e}")
            return jsonify({"error": str(e)}), 500

    def enviar_correo_rechazo(employee_info, comentario=None):
        """
        Envía un correo de notificación de rechazo al empleado.
        """
        try:
            # Obtener el correo del empleado
            destinatario = employee_info["CorporateEmail"]
            if not destinatario:
                return {"error": "No se encontró el correo del empleado"}
                
            # Asunto del correo
            asunto = "Respuesta a su solicitud de acceso como empleado"
            
            # Preparar datos para la plantilla
            timezone = pytz.timezone("America/Mexico_City")
            current_date = datetime.datetime.now(timezone)
            data = {
                "FirstName": employee_info["FirstName"],
                "LastName": employee_info["LastName"],
                "Year": datetime.datetime.now().year,
                "Comentario": comentario
            }
            
            # Renderizar el cuerpo del correo utilizando la plantilla HTML
            cuerpo_html = render_template('Security/UserPasswordemail.html', data=data)
            # Crear el mensaje
            msg = Message(
                asunto,
                sender=app.config['MAIL_USERNAME'],
                recipients=[destinatario]
            )
            
            # Asignar el cuerpo HTML
            msg.html = cuerpo_html
            
            # Enviar el correo
            with app.app_context():
                mail.send(msg)
                
            return {"success": "Correo enviado correctamente"}
        except Exception as e:
            print(f"Error al enviar correo de rechazo: {e}")
            return {"error": str(e)}

    # Retornar las demás rutas existentes...
    return app