# Archivo: GSYCManufacBOMIng.py
# Ruta: src\App\SupyCtrol_Module\GerenteSyC\UtilityMaterials\GSYCManufacBOMIng.py
# Lenguaje: Python con Flask

from flask import jsonify, request
from Consultas_SQL.SupYCtrol.GerenteSyC.UtilityMaterials.GSYCManufacBOMIngSQL import (
    BOMExploder, get_bom_data, search_parts, get_status_manual, calculate_total_costs, get_total_cost_for_parent
)


def funciones_BomPartIng(app):
    """
    Registra las rutas para la funciones_BomPartIng directamente en la aplicación Flask.
    """

    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/explode", methods=["POST"])
    def explode_bom():
        """Explota una BOM multinivel"""
        try:
            data = request.json
            part_num = data.get('partNum')
            revision = data.get('revision')
            
            if not part_num or not revision:
                return jsonify({
                    'success': False,
                    'error': 'PartNum y Revision son requeridos'
                }), 400
            
            exploder = BOMExploder()
            result = exploder.explode_bom(part_num, revision)
            
            return jsonify(result)
            
        except Exception as e:
            return jsonify({
                'success': False,
                'error': f'Error interno: {str(e)}'
            }), 500

    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/get-bom/<process_id>", methods=["GET"])
    def get_bom(process_id):
        """Obtiene datos de una BOM explodida"""
        try:
            bom_data = get_bom_data(process_id)
            
            if 'error' in bom_data:
                return jsonify({
                    'success': False,
                    'error': bom_data['error']
                }), 500
            
            return jsonify({
                'success': True,
                'data': bom_data
            })
            
        except Exception as e:
            return jsonify({
                'success': False,
                'error': f'Error interno: {str(e)}'
            }), 500

    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/search-parts", methods=["GET"])
    def search_parts_endpoint():
        """Busca partes manufacturadas en el ERP"""
        try:
            query = request.args.get('q', '')
            
            if len(query) < 2:
                return jsonify({
                    'success': False,
                    'error': 'La búsqueda debe tener al menos 2 caracteres'
                }), 400
            
            parts = search_parts(query)
            
            return jsonify({
                'success': True,
                'parts': parts
            })
            
        except Exception as e:
            return jsonify({
                'success': False,
                'error': f'Error interno: {str(e)}'
            }), 500
    

    """
    Endpoint para actualizar el costo externo y estado de un material en la BOM (Lista de Materiales).
    Recibe: processId, partNum (número de parte) y newExtCost (nuevo costo externo).
    Retorna: Resultado de la operación con éxito/error y mensaje correspondiente.
    """
    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/update-cost", methods=["POST"])
    def update_cost():
        try:
            data = request.get_json()
            
            # Validaciones básicas
            if not data:
                return jsonify({'success': False, 'error': 'Datos no proporcionados'}), 400
                
            required_fields = ['newExtCost', 'processId', 'partNum']
            if not all(field in data for field in required_fields):
                return jsonify({
                    'success': False,
                    'error': f'Faltan campos requeridos: {required_fields}'
                }), 400
            
            # Procesar la actualización
            result = get_status_manual(
                new_ext_cost=data['newExtCost'],
                process_id=data['processId'],
                part_num=data['partNum']
            )
            
            if not result.get('success'):
                return jsonify(result), 400
                
            return jsonify(result)
            
        except Exception as e:
            return jsonify({
                'success': False,
                'error': f'Error inesperado: {str(e)}'
            }), 500
        


    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/calculate-costs/<process_id>", methods=["POST"])
    def calculate_costs(process_id):
        """Endpoint para calcular y actualizar costos totales"""
        try:
            # Validar process_id
            if not process_id or len(process_id) != 36:  # UUID tiene 36 caracteres
                return jsonify({
                    'success': False,
                    'error': "ProcessID inválido"
                }), 400

            app.logger.info(f"Iniciando cálculo de costos para process_id: {process_id}")
            result = calculate_total_costs(process_id)
            
            if not result.get('success'):
                app.logger.error(f"Error calculando costos: {result.get('error')}")
                return jsonify(result), 400
                
            # Registrar éxito con los datos disponibles
            app.logger.info(f"""
                Costos calculados para {process_id}.
                Materiales actualizados: {len(result['updatedMaterials'])}
                Costo total calculado: {result['totalCalculatedCost']}
            """)
            
            return jsonify({
                'success': True,
                'message': 'Costos calculados exitosamente',
                'updatedMaterials': result['updatedMaterials'],
                'totalCalculatedCost': result['totalCalculatedCost'],
                'processId': process_id
            })
            
        except Exception as error:
            app.logger.error(f"Error inesperado: {str(error)}")
            return jsonify({
                'success': False,
                'error': f"Error interno del servidor: {str(error)}"
            }), 500

        
    @app.route("/SyC/GerenteSyC/UtilityMaterials/GSYCManufacBOMIng/get-total-cost", methods=["GET"])
    def get_total_cost():
        """Endpoint con verificación de consistencia"""
        process_id = request.args.get('processId')
        part_num = request.args.get('partNum')
        revision = request.args.get('revision')
        
        result = get_total_cost_for_parent(process_id, part_num, revision)
        
        if result['success']:
            # Registrar diferencia si hay menos registros que los esperados
            expected_materials = request.args.get('expectedMaterials')
            expected_operations = request.args.get('expectedOperations')
            
            if expected_materials and result['materialCount'] != int(expected_materials):
                app.logger.warning(f"Diferencia en materiales: Esperados={expected_materials}, Obtenidos={result['materialCount']}")
            
            if expected_operations and result['operationCount'] != int(expected_operations):
                app.logger.warning(f"Diferencia en operaciones: Esperados={expected_operations}, Obtenidos={result['operationCount']}")
        
        return jsonify(result)
        

        
        