# Archivo: AccessDistributors.py
# Ruta: src\App\Ventas_Module\VentasEUA\AccessDistributors.py
# Lenguaje: Python con Flask

from venv import logger
from flask import jsonify, request, render_template
# from flask_mail import Mail, Message
from flask_mail import Message
from Consultas_SQL.Ventas.VentasEUA.AccessDistributorsSQL import (
    get_AccessDistributors, get_SellerforDistributors, get_distributorSelectedinfo, 
    check_existing_email, get_request_type_info, insert_new_user, insert_user_profile, 
    update_user_request_status, insert_user_role, update_user_request_rejected
)
import datetime
import pytz
import os
from App.Security_Module.UserPassword import send_activation_email

def funcion_AccessDistributors(app, mail):
    """
    Agrupamos todas las funciones que importamos app en la aplicación Flask.
    """

    @app.route("/Ventas_Module/VentasEUA/AccessDistributors/list", methods=["GET"])
    def list_AccessDistributors():
        """Retorna la lista de Proveedores pendites de aceptacion."""
        try:
            AccessDistributors = get_AccessDistributors()

            return jsonify(AccessDistributors), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/Ventas_Module/VentasEUA/AccessDistributors/get_sellers", methods=["GET"])
    def get_sellers():
        try:
            sellers = get_SellerforDistributors()
            return jsonify(sellers), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500
        
    @app.route("/Ventas_Module/VentasEUA/AccessDistributors/get_comments/<distributor_id>", methods=["GET"])
    def get_distributor_comments(distributor_id):
        try:
            comments = get_distributorSelectedinfo(distributor_id)
            return jsonify(comments), 200
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route("/Ventas_Module/VentasEUA/AccessDistributors/assign_seller", methods=["POST"])
    def assign_seller():
        try:
            data = request.get_json()
            distributor_id = data.get("distributorId")
            seller_id = data.get("sellerId")
            discount = data.get("discount")

            print(f"Aprobando distribuidor {distributor_id} y asignando vendedor {seller_id} con descuento {discount}")
            
            # Validación de datos
            if not distributor_id or not seller_id or discount is None:
                return jsonify({"error": "Datos incompletos"}), 400
            
            # Conversión de descuento a dos decimales
            try:
                discount = round(float(discount), 2)
            except ValueError:
                return jsonify({"error": "Formato de descuento inválido"}), 400
            
            # Obtener información del distribuidor
            distributor_info = get_distributorSelectedinfo(distributor_id)
            if not distributor_info or "error" in distributor_info:
                return jsonify({"error": "No se pudo obtener la información del distribuidor"}), 500
            
            # Verificar que el correo no exista ya en la tabla Users
            email_exists = check_existing_email(distributor_info["ContactEmail"])
            if email_exists:
                return jsonify({"error": "El correo electrónico ya está registrado en el sistema"}), 400
            
            # Actualizar estado de la solicitud en UserRequests
            timezone = pytz.timezone("America/Mexico_City")
            current_date = datetime.datetime.now(timezone).strftime('%Y-%m-%d %H:%M:%S')
            update_result = update_user_request_status(distributor_id, current_date)
            if "error" in update_result:
                return jsonify(update_result), 500
            
            # Insertar nuevo usuario en la tabla Users y obtener UserID
            user_result = insert_new_user(distributor_info["RequestTypeID"], distributor_info["ContactEmail"])
            if "error" in user_result:
                return jsonify(user_result), 500
            
            # Obtener CompanyID, DivisionID y DepartamentID basado en RequestTypeID
            type_info = get_request_type_info(distributor_info["RequestTypeID"])
            if "error" in type_info:
                return jsonify(type_info), 500
            
            # Insertar perfil de usuario con los datos del distribuidor, incluyendo el vendedor y descuento
            profile_result = insert_user_profile(
                user_id=user_result["UserID"],
                request_id=distributor_info["RequestID"],
                request_type_id=distributor_info["RequestTypeID"],
                business_name=distributor_info["BusinessName"],
                legal_representative=distributor_info["LegalRepresentative"],
                primary_contact=distributor_info["PrimaryContact"],
                fiscal_address=distributor_info["FiscalAddress"],
                postal_code=distributor_info["PostalCode"],
                state=distributor_info["State"],
                city=distributor_info["City"],
                discount=discount,
                seller_user_id=seller_id,
                email=distributor_info["ContactEmail"],
                contact_phone=distributor_info["ContactPhone"],
                company_id=type_info["CompanyID"],
                division_id=type_info["DivisionID"],
                departament_id=type_info["DepartamentID"]
            )
            
            if "error" in profile_result:
                return jsonify(profile_result), 500
            
            # Asignar rol al usuario (V-P-DIEU para distribuidores)
            role_result = insert_user_role(user_result["UserID"])
            if "error" in role_result:
                return jsonify(role_result), 500
            
            token_result = send_activation_email(app, mail, user_result["UserID"])
            
            # token_result = "Mensaje enviado" or "Error al crear token/enviar emanil
            """
            if token_result == 'Error en la generacion de token':
                messagemail ="Hubo un error al enviar el correo contacte a soporte" #concatenar con userid
            else:
                messagemail = "Alta de usuario exitoso y Correo enviado"
            """
            # Si token_result es una tupla, significa que hubo un error
            if isinstance(token_result, tuple):
                # Obtener el objeto jsonify y el código de estado
                json_response, status_code = token_result
                # Extraer el contenido JSON del objeto jsonify
                error_data = json_response.get_json()
                # Devolver el error manteniendo el código de estado original
                return jsonify({
                    "message": "Hubo un error al enviar el correo de activación contacte a soporte carlos.velazquezm@igsa.com.mx ",
                    "error": error_data.get('error', 'Error desconocido')# Se obtiene la variable error del json
                }), status_code
            # Si no es una tupla, es una respuesta exitosa
            return jsonify({"message": "Distribuidor aprobado y vendedor asignado correctamente"}), 200
            
        except Exception as e:
            print(f"[Error en assign_seller] {e}")
            return jsonify({"error": str(e)}), 500
        
    @app.route("/Ventas_Module/VentasEUA/AccessDistributors/reject", methods=["POST"])
    def reject_distributor():
        try:
            data = request.get_json()
            distributor_id = data.get("distributorId")
            comentario = data.get("comentario")  # Opcional
            
            # Validar datos
            if not distributor_id:
                return jsonify({"error": "ID de distribuidor requerido"}), 400
            
            # Obtener información del distribuidor
            distributor_info = get_distributorSelectedinfo(distributor_id)
            if not distributor_info or "error" in distributor_info:
                return jsonify({"error": "No se pudo obtener la información del distribuidor"}), 500
            
            # Actualizar estado de la solicitud en UserRequests
            timezone = pytz.timezone("America/Mexico_City")
            current_date = datetime.datetime.now(timezone).strftime('%Y-%m-%d %H:%M:%S')
            update_result = update_user_request_rejected(distributor_id, current_date, comentario)
            if "error" in update_result:
                return jsonify(update_result), 500
            
            # Enviar correo al distribuidor
            email_result = enviar_correo_rechazo(distributor_info, comentario)
            if "error" in email_result:
                # El correo falló, pero la solicitud fue rechazada correctamente
                return jsonify({
                    "message": "Solicitud rechazada, pero hubo un problema al enviar el correo electrónico",
                    "error_mail": email_result["error"]
                }), 200
            
            return jsonify({"message": "Solicitud rechazada correctamente y notificación enviada"}), 200
            
        except Exception as e:
            print(f"[Error en reject_distributor] {e}")
            return jsonify({"error": str(e)}), 500

    def enviar_correo_rechazo(distributor_info, comentario=None):
        """
        Envía un correo de notificación de rechazo al distribuidor.
        """
        try:
            # Obtener el correo del distribuidor
            destinatario = distributor_info["ContactEmail"]
            if not destinatario:
                return {"error": "No se encontró el correo del distribuidor"}
                
            # Asunto del correo
            asunto = "Respuesta a su solicitud de distribuidor"
            
            # En la función enviar_correo_rechazo:
            timezone = pytz.timezone("America/Mexico_City")
            current_date = datetime.datetime.now(timezone)
            data = {
                "PrimaryContact": distributor_info["PrimaryContact"],
                "BusinessName": distributor_info["BusinessName"],
                "Year": datetime.datetime.now().year,
                "Comentario": comentario
            }
            
            # Renderizar el cuerpo del correo utilizando la plantilla HTML
            cuerpo_html = render_template('Ventas/VentasEUA/Mail/AccessDistributorsReject.html', data=data)
            
            # Crear el mensaje
            msg = Message(
                asunto,
                sender=app.config['MAIL_USERNAME'],
                recipients=[destinatario]
            )
            
            # Asignar el cuerpo HTML
            msg.html = cuerpo_html
            
            # Enviar el correo
            with app.app_context():
                mail.send(msg)
                
            return {"success": "Correo enviado correctamente"}
        except Exception as e:
            print(f"Error al enviar correo de rechazo: {e}")
            return {"error": str(e)}