# Archivo: CZE_BreakerProtectionTypeOptimizadoSQL.py
# Ruta: src/Consultas_SQL/Ventas/VentasEUA/DataMaster/CZE_BreakerProtectionTypeOptimizadoSQL.py
# Lenguaje: Python
from Consultas_SQL.conexion import get_connection

# Configuración para la tabla de breakers
TABLE_NAME = "CZE_BreakerProtectionType"
ID_FIELD = "Item_breaker_Protection"

def get_BreakerProtectionType():
    """Obtiene los BreakerProtectionType desde la base de datos."""
    query = f"""
    SELECT *
    FROM {TABLE_NAME}
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        
        # Obtener los nombres de las columnas de la tabla
        column_names = [column[0] for column in cursor.description]
        
        # Obtener los resultados
        results = cursor.fetchall()
        
        # Construir dinámicamente la lista de diccionarios
        items = []
        for row in results:
            item_dict = {}
            for i, column_name in enumerate(column_names):
                item_dict[column_name] = row[i]
            items.append(item_dict)
            
        return items
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo BreakerProtectionType: {e}")
        return []
    finally:
        if conn:
            conn.close()
            
def item_exists(item_id):
    """
    Verifica si un BreakerProtectionType existe en la base de datos.
    
    Args:
        item_id: Valor del campo ID a buscar
        
    Returns:
        tuple: Resultado de la consulta COUNT(*)
    """
    query = f"""
    SELECT COUNT(*) 
    FROM {TABLE_NAME} 
    WHERE {ID_FIELD} = ?
    """
    conn = get_connection()
    try:
        cursor = conn.cursor()
        cursor.execute(query, (item_id,))
        result = cursor.fetchone()
        return result
    finally:
        if conn:
            conn.close()

def insert_BreakerProtectionType(data_dict):
    """
    Inserta un nuevo BreakerProtectionType en la base de datos.
    
    Args:
        data_dict (dict): Diccionario con los campos y valores a insertar
    """
    # Obtener los campos y valores del diccionario
    fields = list(data_dict.keys())
    values = list(data_dict.values())
    
    # Crear los placeholders para la consulta SQL (?, ?, ?)
    placeholders = ", ".join(["?"] * len(fields))
    
    # Construir la consulta SQL
    query = f"""
    INSERT INTO {TABLE_NAME} ({", ".join(fields)})
    VALUES ({placeholders})
    """
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, values)
        conn.commit()
    except Exception as e:
        print(f"[Error inesperado] Error insertando BreakerProtectionType: {e}")
    finally:
        if conn:
            conn.close()            

def update_BreakerProtectionType(data_dict):
    """
    Actualiza un BreakerProtectionType en la base de datos.
    
    Args:
        data_dict (dict): Diccionario con los campos y valores a actualizar.
                         Debe incluir el ID_FIELD para identificar el registro.
    """
    # Verificar que el ID está presente en el diccionario
    if ID_FIELD not in data_dict:
        print(f"[Error] El campo ID '{ID_FIELD}' es obligatorio para actualizar")
        return
    
    # Extraer el ID y eliminarlo del diccionario de datos
    id_value = data_dict.pop(ID_FIELD)
    
    # Si no hay campos para actualizar, salir
    if not data_dict:
        print("[Error] No se proporcionaron campos para actualizar")
        return
    
    # Construir la cláusula SET para la actualización
    set_clause = ", ".join([f"{field} = ?" for field in data_dict.keys()])
    
    # Valores para la consulta (valores para SET + valor para WHERE)
    values = list(data_dict.values())
    values.append(id_value)  # Añadir el ID al final
    
    query = f"""
    UPDATE {TABLE_NAME}
    SET {set_clause}
    WHERE {ID_FIELD} = ?
    """
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, values)
        conn.commit()
    except Exception as e:
        print(f"[Error inesperado] Error actualizando registro en {TABLE_NAME}: {e}")
    finally:
        if conn:
            conn.close()
            
def delete_BreakerProtectionType(item_id):
    """
    Elimina un BreakerProtectionType de la base de datos.
    
    Args:
        item_id: Valor del ID del BreakerProtectionType a eliminar
    """
    query = f"""
    DELETE FROM {TABLE_NAME}
    WHERE {ID_FIELD} = ?
    """
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (item_id,))
        conn.commit()
    except Exception as e:
        print(f"[Error inesperado] Error eliminando BreakerProtectionType: {e}")
    finally:
        if conn:
            conn.close()

def searchitems():
    """
    Obtiene los IDs y hojas técnicas de todos los BreakerProtectionType.
    
    Returns:
        list: Lista de diccionarios con ID y TechnicalSheet
    """
    query = f"""
    SELECT {ID_FIELD}, TechnicalSheet 
    FROM {TABLE_NAME}
    """
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        results = cursor.fetchall()
        
        if not results:
            return []
        
        # Construir lista de diccionarios con campos específicos
        items = []
        for row in results:
            items.append({
                "Item_breaker": row[0],
                "TechnicalSheet": row[1]
            })
            
        return items
    except Exception as e:
        print(f"[Error inesperado] Error consultando BreakerProtectionType: {e}")
        return []
    finally:
        if conn:
            conn.close()

def update_BreakerProtectionType_TechnicalSheet(item, TechnicalSheet):
    """
    Actualiza la hoja técnica de un BreakerProtectionType.
    
    Args:
        item: ID del BreakerProtectionType
        TechnicalSheet: Nueva URL de la hoja técnica
    """
    query = f"""
    UPDATE {TABLE_NAME}
    SET TechnicalSheet = ?
    WHERE {ID_FIELD} = ?
    """
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (TechnicalSheet, item))
        conn.commit()
    except Exception as e:
        print(f"[Error inesperado] Error actualizando hoja técnica: {e}")
    finally:
        if conn:
            conn.close()