# Archivo: MailManagementSQL.py
# Ruta: src\Consultas_SQL\Utilities\MailManagementSQL.py
# Descripción: Módulo para el manejo de funciones SQL usadas en la logica de MailManagement.py
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025



import logging
from Consultas_SQL.conexion import get_connection

# Configurar logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def obtener_destinatarios_mail(mail_list_id):
    """
    Obtiene los destinatarios de correo electrónico desde las tablas de configuración.
    
    Args:
        mail_list_id (int): ID de la lista de correos en Util_MailListHead
        
    Returns:
        dict: Diccionario con listas de destinatarios por tipo (TO, CC, BCC)
            {
                'TO': ['correo1@example.com', 'correo2@example.com'],
                'CC': ['cc1@example.com'],
                'BCC': ['bcc1@example.com'],
                'total_destinatarios': 4,
                'success': True,
                'mensaje': 'Destinatarios obtenidos correctamente'
            }
    """
    
    destinatarios = {
        'TO': [],
        'CC': [],
        'BCC': [],
        'total_destinatarios': 0,
        'success': False,
        'mensaje': '',
        'mail_list_id': mail_list_id
    }
    
    try:
        # Obtener conexión a la base de datos
        connection = get_connection()
        cursor = connection.cursor()
        
        # Consulta para obtener destinatarios activos
        query = """
            SELECT 
                d.Mail,
                d.RecipientType,
                d.MailListLine,
                h.Subject,
                h.ModuleID
            FROM Util_MailListDetail d
            INNER JOIN Util_MailListHead h ON d.MailListID = h.MailListID
            WHERE d.MailListID = ? 
                AND d.ActiveLine = 1 
                AND h.Active = 1
                AND d.Mail IS NOT NULL 
                AND d.Mail != ''
            ORDER BY d.MailListLine
        """
        
        cursor.execute(query, (mail_list_id,))
        resultados = cursor.fetchall()
        
        if not resultados:
            destinatarios['success'] = True
            destinatarios['mensaje'] = f'No se encontraron destinatarios activos para MailListID: {mail_list_id}'
            logger.warning(f"No se encontraron destinatarios para MailListID: {mail_list_id}")
            return destinatarios
        
        # Procesar resultados y agrupar por tipo de destinatario
        for row in resultados:
            mail = row[0].strip() if row[0] else None
            recipient_type = row[1].strip().upper() if row[1] else 'TO'
            
            # Validar que el correo no esté vacío
            if mail and '@' in mail:
                # Normalizar tipo de destinatario
                if recipient_type not in ['TO', 'CC', 'BCC']:
                    recipient_type = 'TO'
                
                # Evitar duplicados
                if mail not in destinatarios[recipient_type]:
                    destinatarios[recipient_type].append(mail)
        
        # Calcular total de destinatarios
        destinatarios['total_destinatarios'] = (
            len(destinatarios['TO']) + 
            len(destinatarios['CC']) + 
            len(destinatarios['BCC'])
        )
        
        # Establecer resultado
        if destinatarios['total_destinatarios'] > 0:
            destinatarios['success'] = True
            destinatarios['mensaje'] = f'Se obtuvieron {destinatarios["total_destinatarios"]} destinatarios correctamente'
            logger.info(f"MailListID {mail_list_id}: {destinatarios['total_destinatarios']} destinatarios obtenidos")
        else:
            destinatarios['success'] = True
            destinatarios['mensaje'] = f'No se encontraron correos válidos para MailListID: {mail_list_id}'
            logger.warning(f"No se encontraron correos válidos para MailListID: {mail_list_id}")
        
        cursor.close()
        connection.close()
        
    except Exception as e:
        destinatarios['success'] = False
        destinatarios['mensaje'] = f'Error al obtener destinatarios: {str(e)}'
        logger.error(f"Error en obtener_destinatarios_mail: {str(e)}")
    
    return destinatarios


def validar_mail_list_existe(mail_list_id):
    """
    Valida si existe el MailListID en la tabla Util_MailListHead.
    
    Args:
        mail_list_id (int): ID de la lista de correos
        
    Returns:
        dict: Resultado de la validación
              {
                  'existe': True/False,
                  'activo': True/False,
                  'subject': 'Asunto del correo',
                  'module_id': 'ID del módulo',
                  'success': True/False,
                  'mensaje': 'Mensaje descriptivo'
              }
    """
    
    resultado = {
        'existe': False,
        'activo': False,
        'subject': None,
        'module_id': None,
        'success': False,
        'mensaje': ''
    }
    
    try:
        connection = get_connection()
        cursor = connection.cursor()
        
        query = """
            SELECT 
                MailListID,
                ModuleID,
                Subject,
                Active,
                CreatedAt,
                CreatedBy
            FROM Util_MailListHead
            WHERE MailListID = ?
        """
        
        cursor.execute(query, (mail_list_id,))
        row = cursor.fetchone()
        
        if row:
            resultado['existe'] = True
            resultado['activo'] = bool(row[3])  # Active
            resultado['subject'] = row[2]       # Subject
            resultado['module_id'] = row[1]     # ModuleID
            resultado['success'] = True
            
            if resultado['activo']:
                resultado['mensaje'] = f'MailListID {mail_list_id} existe y está activo'
            else:
                resultado['mensaje'] = f'MailListID {mail_list_id} existe pero está inactivo'
                
        else:
            resultado['success'] = True
            resultado['mensaje'] = f'MailListID {mail_list_id} no existe en la base de datos'
        
        cursor.close()
        connection.close()
        
    except Exception as e:
        resultado['success'] = False
        resultado['mensaje'] = f'Error al validar MailListID: {str(e)}'
        logger.error(f"Error en validar_mail_list_existe: {str(e)}")
    
    return resultado


def obtener_estadisticas_mail_list(mail_list_id):
    """
    Obtiene estadísticas detalladas de una lista de correos.
    
    Args:
        mail_list_id (int): ID de la lista de correos
        
    Returns:
        dict: Estadísticas de la lista
              {
                  'total_registros': 5,
                  'activos': 3,
                  'inactivos': 2,
                  'por_tipo': {'TO': 2, 'CC': 1, 'BCC': 0},
                  'correos_validos': 3,
                  'correos_invalidos': 0,
                  'success': True,
                  'mensaje': 'Estadísticas obtenidas correctamente'
              }
    """
    
    estadisticas = {
        'total_registros': 0,
        'activos': 0,
        'inactivos': 0,
        'por_tipo': {'TO': 0, 'CC': 0, 'BCC': 0},
        'correos_validos': 0,
        'correos_invalidos': 0,
        'success': False,
        'mensaje': ''
    }
    
    try:
        connection = get_connection()
        cursor = connection.cursor()
        
        query = """
            SELECT 
                Mail,
                RecipientType,
                ActiveLine
            FROM Util_MailListDetail
            WHERE MailListID = ?
        """
        
        cursor.execute(query, (mail_list_id,))
        resultados = cursor.fetchall()
        
        estadisticas['total_registros'] = len(resultados)
        
        for row in resultados:
            mail = row[0]
            recipient_type = row[1].strip().upper() if row[1] else 'TO'
            activo = bool(row[2])
            
            # Contar activos/inactivos
            if activo:
                estadisticas['activos'] += 1
            else:
                estadisticas['inactivos'] += 1
            
            # Contar por tipo (solo activos)
            if activo:
                if recipient_type in ['TO', 'CC', 'BCC']:
                    estadisticas['por_tipo'][recipient_type] += 1
                else:
                    estadisticas['por_tipo']['TO'] += 1
            
            # Validar correos
            if mail and '@' in mail and '.' in mail:
                estadisticas['correos_validos'] += 1
            else:
                estadisticas['correos_invalidos'] += 1
        
        estadisticas['success'] = True
        estadisticas['mensaje'] = f'Estadísticas obtenidas para MailListID {mail_list_id}'
        
        cursor.close()
        connection.close()
        
    except Exception as e:
        estadisticas['success'] = False
        estadisticas['mensaje'] = f'Error al obtener estadísticas: {str(e)}'
        logger.error(f"Error en obtener_estadisticas_mail_list: {str(e)}")
    
    return estadisticas

def guardar_log_envio(
    mail_list_id,
    subject,
    recipients_to,
    recipients_cc,
    recipients_bcc,
    template_used,
    template_data,
    files_sent,
    sent_by,
    status='ENVIADO',
    error_msg=None
):
    """
    Guarda un registro de envío de correo en la tabla Util_MailSentLog.
    """
    try:
        connection = get_connection()
        cursor = connection.cursor()

        query = """
            INSERT INTO Util_MailSentLog
                (MailListID, Subject, RecipientsTO, RecipientsCC, RecipientsBCC,
                TemplateUsed, TemplateData, FilesSent, SentBy, Status, ErrorMsg)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """

        cursor.execute(query, (
            mail_list_id,
            subject,
            recipients_to,
            recipients_cc,
            recipients_bcc,
            template_used,
            template_data,
            files_sent,
            sent_by,
            status,
            error_msg
        ))
        connection.commit()
        cursor.close()
        connection.close()
        return {'success': True, 'mensaje': 'Log de envío registrado'}
    except Exception as e:
        return {'success': False, 'mensaje': f'Error al guardar log: {str(e)}'}

def obtener_subject_de_mail_list(mail_list_id):
    try:
        connection = get_connection()
        cursor = connection.cursor()
        query = "SELECT Subject FROM Util_MailListHead WHERE MailListID = ?"
        cursor.execute(query, (mail_list_id,))
        row = cursor.fetchone()
        cursor.close()
        connection.close()
        return row[0] if row else None
    except Exception as e:
        return None
