# Archivo: AccessDistributorsSQL.py
# Ruta: src\Consultas_SQL\Ventas\VentasEUA\AccessDistributorsSQL.py
# Lenguaje: Python con Flask

from venv import logger
from Consultas_SQL.conexion import get_connection

def get_AccessDistributors():
    """Obtiene las compañías desde la base de datos."""
    query = """
    SELECT
        UserRequests.RequestID,
        UserRequests.RequestTypeID,
        UserRequests.BusinessName,
        UserRequests.LegalRepresentative,
        UserRequests.PrimaryContact,
        UserRequests.ContactEmail,
        UserRequests.FiscalAddress,
        UserRequests.PostalCode,
        UserRequests.State,
        UserRequests.City,
        UserRequests.ContactPhone,
        UserRequests.Comments,
        FORMAT(UserRequests.CreatedAt, 'dd-MM-yyyy HH:mm') AS CreatedAt
    FROM 
        UserRequests
    WHERE
        (RequestTypeID = 'DISCAN' OR RequestTypeID = 'DISEUA')
        AND Status = 'PENDIENTE'
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        results = cursor.fetchall()
        
        return [{
            "RequestID":           row[0], 
            "RequestTypeID":       row[1], 
            "BusinessName":        row[2], 
            "LegalRepresentative": row[3], 
            "PrimaryContact":      row[4], 
            "ContactEmail":        row[5], 
            "FiscalAddress":       row[6], 
            "PostalCode":          row[7], 
            "State":               row[8], 
            "City":                row[9], 
            "ContactPhone":        row[10], 
            "Comments":            row[11], 
            "CreatedAt":           row[12]
        } for row in results]
        
        
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo a los distribuidores: {e}")
        return []
    finally:
        if conn:
            conn.close()

def get_SellerforDistributors():
    """Obtiene la Lista de Vendedores para asignarle al distribuidor."""
    query = """
    SELECT
        UserRoles.UserID,
        Concat (Profiles.FirstName,' ',Profiles.LastName) AS Seller
    FROM 
        Roles
    LEFT JOIN
        UserRoles ON Roles.RoleID = UserRoles.RoleID
    LEFT JOIN
        Profiles ON UserRoles.UserID = Profiles.UserID
    WHERE
        Roles.RoleID = 'V-P-VEEU'
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        results = cursor.fetchall()
        
        return [{
            "UserID":       row[0], 
            "Seller":    row[1]
            } for row in results]
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo a los vendedores: {e}")
        return []
    finally:
        if conn:
            conn.close()
            
def get_distributorSelectedinfo(distributor_id):
    """Obtiene los comentarios de un distribuidor específico por su RequestID."""
    query = """
    SELECT
        UserRequests.RequestID,
        UserRequests.RequestTypeID,
        UserRequests.BusinessName,
        UserRequests.LegalRepresentative,
        UserRequests.PrimaryContact,
        UserRequests.ContactEmail,
        UserRequests.FiscalAddress,
        UserRequests.PostalCode,
        UserRequests.State,
        UserRequests.City,
        UserRequests.ContactPhone,
        UserRequests.Comments,
        FORMAT(UserRequests.CreatedAt, 'dd-MM-yyyy HH:mm') AS CreatedAt
    FROM 
        UserRequests
    WHERE
        RequestID = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (distributor_id,))
        result = cursor.fetchone()
        
        if result:
            return {
                "RequestID":           result[0], 
                "RequestTypeID":       result[1], 
                "BusinessName":        result[2], 
                "LegalRepresentative": result[3], 
                "PrimaryContact":      result[4], 
                "ContactEmail":        result[5], 
                "FiscalAddress":       result[6], 
                "PostalCode":          result[7], 
                "State":               result[8], 
                "City":                result[9], 
                "ContactPhone":        result[10], 
                "Comments":            result[11], 
                "CreatedAt":           result[12]
            }
        else:
            return {"Comments": ""}
            
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo comentarios del distribuidor: {e}")
        return {"Comments": "", "error": str(e)}
    finally:
        if conn:
            conn.close()

def insert_user_role(user_id, role_id="V-P-DIEU"):
    """
    Asigna un rol al usuario en la tabla UserRoles.
    Por defecto, asigna el rol "V-P-DIEU" para distribuidores.
    """
    query = """
    INSERT INTO UserRoles (
        UserID,
        RoleID
    ) VALUES (?, ?)
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (user_id, role_id))
        conn.commit()
        
        return {"message": "Rol asignado correctamente al usuario"}
    except Exception as e:
        print(f"[Error SQL] Error asignando rol al usuario: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()

def check_existing_email(email):
    """
    Valida si el correo ya existe en la tabla Users.
    Retorna True si existe, False si no existe.
    """
    query = """
    SELECT COUNT(*) AS count
    FROM Users
    WHERE Email = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (email,))
        result = cursor.fetchone()
        
        return result[0] > 0  # True si count > 0, False si count = 0
        
    except Exception as e:
        print(f"[Error SQL] Error verificando email existente: {e}")
        # En caso de error, asumimos que el email existe para evitar duplicados
        return True
    finally:
        if conn:
            conn.close()

def update_user_request_status(request_id, review_date):
    """
    Actualiza el estado y fecha de revisión en UserRequests.
    """
    query = """
    UPDATE UserRequests
    SET 
        Status = 'APROBADO',
        ReviewDate = ?
    WHERE 
        RequestID = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (review_date, request_id))
        conn.commit()
        
        return {"message": "Estado de solicitud actualizado correctamente"}
    except Exception as e:
        print(f"[Error SQL] Error actualizando estado de solicitud: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()

def insert_new_user(request_type_id, email):
    """
    Inserta un nuevo usuario en la tabla Users y retorna el UserID generado.
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        
        # Primero: Insertar el usuario
        insert_query = """
        INSERT INTO Users (
            RequestTypeID,
            Email
        ) VALUES (?, ?);
        """
        cursor.execute(insert_query, (request_type_id, email))
        
        # Segundo: Obtener el ID generado
        identity_query = "SELECT @@IDENTITY AS UserID;"
        cursor.execute(identity_query)
        result = cursor.fetchone()
        user_id = result[0] if result else None
        
        conn.commit()
        
        if not user_id:
            return {"error": "No se pudo obtener el UserID generado"}
        
        return {"UserID": user_id, "message": "Usuario creado correctamente"}
    except Exception as e:
        print(f"[Error SQL] Error creando nuevo usuario: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()

def get_request_type_info(request_type_id):
    """
    Obtiene CompanyID, DivisionID y DepartamentID basado en RequestTypeID.
    """
    query = """
    SELECT
        CompanyID,
        DivisionID,
        DepartamentID
    FROM 
        RequestType
    WHERE 
        RequestTypeID = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (request_type_id,))
        result = cursor.fetchone()
        
        if not result:
            return {"error": f"No se encontró información para el RequestTypeID: {request_type_id}"}
        
        return {
            "CompanyID": result[0],
            "DivisionID": result[1],
            "DepartamentID": result[2]
        }
    except Exception as e:
        print(f"[Error SQL] Error obteniendo información de tipo de solicitud: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()

def insert_user_profile(user_id, request_id, request_type_id, business_name, legal_representative, 
                    primary_contact, fiscal_address, postal_code, state, city, discount, 
                    seller_user_id, email, contact_phone, company_id, division_id, departament_id):
    """
    Inserta un nuevo perfil de usuario en la tabla Profiles.
    """
    query = """
    INSERT INTO Profiles (
        UserID, RequestID, RequestTypeID, BusinessName, LegalRepresentative,
        PrimaryContact, FiscalAddress, PostalCode, State, City, Discount,
        SellerUserID, Email, ContactPhone, CompanyID, DivisionID, DepartamentID
    )
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, (
            user_id, request_id, request_type_id, business_name, legal_representative,
            primary_contact, fiscal_address, postal_code, state, city, discount,
            seller_user_id, email, contact_phone, company_id, division_id, departament_id
        ))
        conn.commit()
        
        return {"message": "Perfil de usuario creado correctamente"}
    except Exception as e:
        print(f"[Error SQL] Error creando perfil de usuario: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()
            
# Añadir a AccessDistributorsSQL.py

def update_user_request_rejected(request_id, review_date, comentario=None):
    """
    Actualiza el estado a RECHAZADO y fecha de revisión en UserRequests.
    El comentario (motivo del rechazo) no se guarda en la base de datos,
    solo se usa para el correo electrónico.
    """
    query = """
    UPDATE UserRequests
    SET 
        Status = 'RECHAZADO',
        ReviewDate = ?
    WHERE 
        RequestID = ?
    """
    params = (review_date, request_id)
    
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query, params)
        conn.commit()
        
        return {"message": "Solicitud rechazada correctamente"}
    except Exception as e:
        print(f"[Error SQL] Error actualizando estado de solicitud a RECHAZADO: {e}")
        return {"error": str(e)}
    finally:
        if conn:
            conn.close()

