# Archivo: registration.py
# Ruta: src\Consultas_SQL\registration.py
# Lenguaje: Python con Flask
import pyodbc
from Consultas_SQL.conexion import get_connection

def get_Correo_Solicitud_de_alta(id):
    """
    Obtiene los correos de los empleados.
    """
    query = """
       SELECT Correos FROM CorreosdeModulos WHERE  ID = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        # Ejecutar la consulta con parámetros seguros
        cursor.execute(query,(id) )
        results = cursor.fetchall()
        
        return [{"Correos": row[0]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo correos de empleados: {e}")
        return []
    finally:
        # Asegúrate de cerrar la conexión si está abierta
        if conn:
            conn.close()


def get_companies():
    query = "SELECT CompanyID, Company FROM Company"
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        results = cursor.fetchall()
        
        # Usar los nombres de columna correctos
        return [{"CompanyID": row[0], "Company": row[1]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo compañías: {e}")
        return []
    finally:
        if conn:
            conn.close()

def get_RequestType():
    """
    Obtiene las compañías desde la base de datos.
    Retorna una lista de diccionarios con ID y Nombre.
    """
    query = "SELECT RequestTypeID, RequestType FROM RequestType WHERE Active = 1"  # Consulta SQL
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        cursor.execute(query)
        results = cursor.fetchall()
        
        # Transformar los resultados en una lista de diccionarios
        return [{"RequestTypeID": row[0], "RequestType": row[1]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo compañías: {e}")
        return []
    finally:
        # Asegúrate de cerrar la conexión si está abierta
        if conn:
            conn.close()

def get_divisions_by_company(company_id):
    """
    Obtiene las divisiones asociadas a una compañía específica.
    Retorna una lista de diccionarios con ID y Nombre.
    """
    query = "SELECT DivisionID, Division FROM Division WHERE CompanyID = ? AND DivisionID != 'CLIE' "
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        # Ejecutar la consulta con parámetros seguros
        cursor.execute(query, (company_id,))
        results = cursor.fetchall()
        
        return [{"id": row[0], "name": row[1]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo divisiones: {e}")
        return []
    finally:
        # Asegúrate de cerrar la conexión si está abierta
        if conn:
            conn.close()

def get_divisions_by_id(Divisionid):
    """
    Obtiene las divisiones asociadas a una compañía específica.
    Retorna una lista de diccionarios con ID y Nombre.
    """
    query = "SELECT  Division FROM Division WHERE DivisionID = ?"
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        # Ejecutar la consulta con parámetros seguros
        cursor.execute(query, (Divisionid,))
        results = cursor.fetchall()
        
        return [{"Division": row[0]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo divisiones: {e}")
        return []
    finally:
        # Asegúrate de cerrar la conexión si está abierta
        if conn:
            conn.close()
            
def get_departments_by_company_and_division(company_id, division_id):
    """
    Obtiene los departamentos filtrados por compañía y división.
    Retorna una lista de diccionarios con ID y Nombre.
    """
    query = """
        SELECT DepartamentID, Departament 
        FROM Departament 
        WHERE CompanyID = ? AND DivisionID = ?
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        # Ejecutar la consulta con parámetros seguros
        cursor.execute(query, (company_id, division_id))
        results = cursor.fetchall()
        
        return [{"id": row[0], "name": row[1]} for row in results]
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return []
    except Exception as e:
        print(f"[Error inesperado] Error obteniendo departamentos: {e}")
        return []
    finally:
        # Asegúrate de cerrar la conexión si está abierta
        if conn:
            conn.close()


def prepare_form_data(data, request_type):
    """
    Prepara los datos del formulario para la inserción en la base de datos
    """
    # Inicializar un diccionario para los datos procesados
    processed_data = {
        'RequestTypeID': request_type,
        'Status': 'PENDIENTE'  # Valor por defecto
    }
    
    if request_type == 'DISMÉX':
        # Procesar específicamente el formulario de México
        processed_data.update({
            'BusinessName': data.get('MX_BusinessName', ''),
            'LegalRepresentative': data.get('MX_LegalRepresentative', ''),
            'PrimaryContact': data.get('MX_PrimaryContact', ''),
            'ContactEmail': data.get('MX_ContactEmail', ''),
            'FiscalAddress': data.get('MX_FiscalAddress', ''),
            'PostalCode': data.get('MX_PostalCode', ''),
            'State': data.get('MX_State', ''),
            'City': data.get('MX_City', ''),
            'RFC': data.get('MX_RFC', ''),
            'ContactPhone': data.get('MX_ContactPhone', ''),
            'Comments': data.get('MX_Comments', ''),
            'Terms': 1 if data.get('MX_Terms') in ['true', '1', 'on', True] else 0
           
        })
    elif request_type in ['DISEUA', 'DISCAN']:
        # Procesar formulario de Distribuidores EUA/Canadá
        processed_data.update({
            'BusinessName': data.get('EU_BusinessName', ''),
            'LegalRepresentative': data.get('EU_LegalRepresentative', ''),
            'PrimaryContact': data.get('EU_PrimaryContact', ''),
            'ContactEmail': data.get('EU_ContactEmail', ''),
            'FiscalAddress': data.get('EU_FiscalAddress', ''),
            'PostalCode': data.get('EU_PostalCode', ''),
            'State': data.get('EU_State', ''),
            'City': data.get('EU_City', ''),
            'ContactPhone': data.get('EU_ContactPhone', ''),
            'Comments': data.get('EU_Comments', ''),
            'Terms': 1 if data.get('EU_Terms') in ['true', '1', 'on', True] else 0
           
        })
    elif request_type == 'EMPLEADO':
        # Procesar formulario de Empleados
        processed_data.update({
            'FirstName': data.get('EMPFirstName', ''),
            'MiddleName': data.get('EMPMiddleName', ''),
            'LastName': data.get('EMPLastName', ''),
            'SecondLastName': data.get('EMPSecondLastName', ''),
            'CorporateEmail': data.get('EMPCorporateEmail', ''),
            'ContactPhone': data.get('EMPContactPhone', ''),
            'EmployeeNumber': data.get('EMPEmployeeNumber', ''),
            'CompanyID': data.get('EMPCompanyID', ''),
            'DivisionID': data.get('EMPDivisionID', ''),
            'DepartamentID': data.get('EMPDepartamentID', ''),
            'Position': data.get('EMPPosition', ''),
            'DirectSupervisor': data.get('EMPDirectSupervisor', ''),
            'PermissionType': data.get('EMPPermissionType', ''),
            'Comments': data.get('EMPComments', ''),
            'Terms': 1 if data.get('EMPTerms') in ['true', '1', 'on', True] else 0
        })
    
    # Imprimir datos procesados para depuración
    
    return processed_data

def insert_user_request(request_data):
    """
    Inserta una nueva solicitud en la tabla UserRequests.
    Params:
        request_data: diccionario con los datos del formulario
    Returns:
        tuple: (bool, str) - (éxito, mensaje)
    """
    try:
        conn = get_connection()
        if not conn:
            raise ConnectionError("No se pudo establecer conexión con la base de datos")
        
        cursor = conn.cursor()
        
        # Construir la consulta SQL de forma dinámica (sin RequestID)
        fields = []
        values = []
        placeholders = []
        
        # Campos que no deben ser formateados
        exclude_formatting = ['CorporateEmail', 'RequestTypeID', 'CompanyID', 'DivisionID', 'DepartamentID', 'Status']
        
        
        for field, value in request_data.items():
            # Omitir el campo RequestID si existe y valores nulos para evitar problemas
            if field != 'RequestID' and value is not None:
                fields.append(field)
                
                # Aplicar formato a los valores de texto, excepto para los campos excluidos
                if isinstance(value, str) and field not in exclude_formatting:
                    # Capitalizar: primera letra mayúscula, resto minúsculas
                    formatted_value = value.strip().capitalize()
                    values.append(formatted_value)
                else:
                    values.append(value)
                
                placeholders.append('?')
        
        # Agregar CreatedAt como campo especial
        fields.append('CreatedAt')
        placeholders.append("SYSDATETIMEOFFSET() AT TIME ZONE 'Central Standard Time'")
        
        # Construir la consulta final
        query = f"""
        INSERT INTO UserRequests ({', '.join(fields)})
        VALUES ({', '.join(placeholders)})
        """
        
        
        # Ejecutar la consulta
        cursor.execute(query, values)
        
        # Obtener el ID generado
        cursor.execute("SELECT @@IDENTITY")
        new_id = cursor.fetchone()[0]
        
        conn.commit()
        
        return True, f"Solicitud registrada exitosamente "
        
    except ConnectionError as ce:
        print(f"[Error de conexión] {ce}")
        return False, f"Error de conexión: {str(ce)}"
    except pyodbc.IntegrityError as ie:
        print(f"[Error de integridad] {ie}")
        return False, f"Error: {str(ie)}"
    except ValueError as ve:
        print(f"[Error de validación] {ve}")
        return False, f"Error de validación: {str(ve)}"
    except Exception as e:
        print(f"[Error inesperado] Error insertando solicitud: {e}")
        return False, f"Error inesperado: {str(e)}"
    finally:
        if conn:
            conn.close()