## 🛠️ Estructura del proyecto

La estructura del proyecto sigue una **arquitectura modular orientada a menús funcionales**, con separación por capas (presentación, lógica y datos), diseñada en Flask, que permite escalar y mantener de forma clara y organizada la lógica de negocio de una plataforma empresarial.

Cada módulo representa un menú funcional de la aplicación:

- Clientes  
- Dirección  
- Ventas  
- Servicios  
- Operaciones  
- Supervisión y Control  

Cada uno se divide en:

- Lógica y rutas de negocio específicas por submenú (`*.py`)
- Acceso a datos (`ConsultasSQL`)
- Vistas (`templates`)
- Recursos estáticos (`static`)

```
Elephant_D102/
├── src/
│   ├── main.py           ← Punto de entrada
│   ├── app.py            ← Lógica y rutas de Flask
│   ├── config.py         ← Configuración por entorno
│   ├── .env              ← Variables sensibles
│   ├── requirements.txt  ← Archivo de dependencias externas (librerías y versiones específicas) 
│   ├── puppeteer_pdf/    ← Script generate_pdf.js (Puppeteer + Node.js)
│   ├── App/              ← Carpeta de aplicaciones desarrolladas y conectadas con app.py
│   │   ├── __init__.py   ← Archivo para definir los paquetes Python dentro de App hacia app.py (importaciones, variables globales, lógica de configuración)
│   │   ├── Utilidades.py       ← Archivo de conexión para pasar funciones de Utilidades_Module a app.py
│   │   ├── Utilidades_Module/  ← Carpeta de Aplicaciones de utilidades para consumo recurrente
│   │   │   ├── __init__.py     ← Archivo para definir los paquetes Python dentro de Utilidades_Module hacia Utilidades.py 
│   │   │   ├── DocsManagement.py               ← Lógica backend para manejo de archivos en la nube, recibe variables específicas
│   │   │   ├── EmailManagement.py              ← Lógica backend para envío de correos, recibe variables específicas
│   │   │   └── ...
│   │   ├── Security.py         ← Archivo de conexión para pasar funciones de Security_Module a app.py
│   │   ├── Security_Module/    ← Aplicaciones y lógica de seguridad (login, etc.)
│   │   │   ├── __init__.py     ← Archivo para definir los paquetes Python dentro de Security_Module hacia Security.py 
│   │   │   └── ...
│   │   ├── Clientes.py                     ← Archivo para renderizar el menú principal de clientes y la lógica del mismo HTML (src\templates\Clientes\Menuppal.html)
│   │   ├── Clientes_module/                ← Carpeta de Archivos de lógica backend desarrolladas para el menú de Clientes
│   │   │   ├── __init__.py                   ← Archivo para definir los paquetes Python dentro de Clientes_module hacia Clientes.py
│   │   │   ├── NetworkEnvironmentMenu.py     ← Archivo con la lógica del menú de Network Environment (src\templates\Clientes\NetworkEnvironment\Menu.html)
│   │   │   ├── NetworkEnvironment/           ← Carpeta de aplicaciones de lógica backend desarrolladas para el menú del cliente Network
│   │   │   │   ├── __init__.py                 ← Archivo para definir los paquetes Python dentro de NetworkEnvironment hacia NetworkEnvironmentMenu.py
│   │   │   │   ├── ScoreNetworkEnvironment.py  ← Lógica del Score de Plantas Eléctricas del cliente Network Environment
│   │   │   │   └── ...
│   │   │   └── ...
│   │   ├── Direccion.py               ← Archivo para renderizar el menú principal de Direccion y la lógica del mismo HTML (src\templates\Direccion\Menuppal.html)
│   │   ├── Direccion_module/          ← Carpeta de Aplicaciones para el menú de Direccion
│   │   │   ├── __init__.py            ← Archivo para definir los paquetes Python dentro de Direccion_module hacia Direccion.py
│   │   │   └── ...
│   │   ├── Ventas.py                  ← Archivo para renderizar el menú principal de Ventas y la lógica del mismo HTML (src\templates\Ventas\Menuppal.html)
│   │   ├── Ventas_module/             ← Carpeta de Aplicaciones para el menú de Ventas
│   │   │   ├── __init__.py            ← Archivo para definir los paquetes Python dentro de Ventas_module hacia Clientes.py
│   │   │   └── ...
│   │   ├── Servicios.py               ← Archivo para renderizar el menú principal de Servicios y la lógica del mismo HTML (src\templates\Servicios\Menuppal.html)
│   │   ├── Servicios_module/          ← Carpeta de Aplicaciones para el menú de Servicios
│   │   │   ├── __init__.py            ← Archivo para definir los paquetes Python dentro de Servicios_module hacia Clientes.py
│   │   │   └── ...
│   │   ├── Operaciones.py             ← Archivo para renderizar el menú principal de Operaciones y la lógica del mismo HTML (src\templates\Operaciones\Menuppal.html)
│   │   ├── Operaciones_module/        ← Carpeta de Aplicaciones para el menú de Operaciones
│   │   │   ├── __init__.py            ← Archivo para definir los paquetes Python dentro de Operaciones_module hacia Clientes.py
│   │   │   └── ...
│   │   ├── SupyCtrol.py               ← Archivo para renderizar el menú principal de SupyCtrol y la lógica del mismo HTML (src\templates\SupyCtrol\Menuppal.html)
│   │   ├── SupyCtrol_module/          ← Carpeta de Aplicaciones para el menú de Supervisión y Control
│   │   │   ├── __init__.py            ← Archivo para definir los paquetes Python dentro de SupyCtrol_module hacia Clientes.py
│   │   │   └── ...
│   │   └── ...
│   ├── ConsultasSQL/               ← Consultas y conexión a bases de datos (intermediarios entre lógica y base de datos)
│   │   ├── conexion.py               ← Configuracion de conexiones con las bases de datos SQL Server
│   │   ├── Utilidades/
│   │   │   ├── DocsManagementSQL.py            ← Archivo donde se manejan las Funciones SQL relacionadas con DocsManagement.py
│   │   │   ├── EmailManagementSQL.py           ← Archivo donde se manejan las Funciones SQL relacionadas con EmailManagement.py
│   │   │   └── ...
│   │   ├── Security/                 ← Carpeta de Archivos para manejo de SQL para el módulo de Seguridad
│   │   ├── Clientes/                 ← Carpeta de Archivos para manejo de SQL para el menú de Clientes
│   │   │   ├── NetworkEnvironment/             ← Carpeta de Archivos para manejo de SQL para el cliente Network Environment
│   │   │   │   ├── ScoreNetworkEnvironmentQL.py  ← Archivo donde se manejan las Funciones SQL relacionadas con ScoreNetworkEnvironment.py
│   │   │   │   └── ...
│   │   │   └── ...
│   │   ├── Direccion/                ← Carpeta de Archivos para manejo de SQL para el menú de Direccion
│   │   ├── Ventas/                   ← Carpeta de Archivos para manejo de SQL para el menú de Ventas
│   │   ├── Servicios/                ← Carpeta de Archivos para manejo de SQL para el menú de Servicios
│   │   ├── Operaciones/              ← Carpeta de Archivos para manejo de SQL para el menú de Operaciones
│   │   ├── SupyCtrol/                ← Carpeta de Archivos para manejo de SQL para el menú de Supervisión y Control
│   │   └── ...
│   └── templates/                          ← Plantillas HTML
│   │   ├── invex.html
│   │   ├── layout.html                       ← Layout base (header y footer; en el header esta el menú principal)
│   │   ├── welcome.html                      ← Html de bienvenida 
│   │   ├── Clientes/                         ← Carpeta de HTML desarrollados para el menú de Clientes
│   │   │   ├── Menuppal.html                   ← HTML del menú principal de clientes
│   │   │   ├── NetworkEnvironment/             ← Carpeta de HTML desarrollados para el cliente NetworkEnvironment
│   │   │   │   ├── Menu.html                     ← HTML del menú del cliente Network
│   │   │   │   ├── ScoreNetworkEnvironment.html  ← HTML del Score de Plantas Eléctricas del cliente NetworkEnvironment
│   │   │   │   └── ...
│   │   │   └── ...
│   │   ├── Direccion/                        ← Carpeta de HTML desarrollados para el menú de Direccion
│   │   ├── Ventas/                           ← Carpeta de HTML desarrollados para el menú de Ventas
│   │   └── ...
│   └── static/                             ← Archivos estáticos (JS, CSS, PDF, IMG, etc.)
│       ├── js/
│       │   ├── Clientes/                       ← Carpeta de JS desarrollados para el menú de Clientes
│       │   │   ├── Menuppal.js                   ← Si se necesitara un JS del menú principal de clientes, aquí se colocaría
│       │   │   ├── NetworkEnvironment/           ← Carpeta de JS desarrollados para el cliente NetworkEnvironment
│       │   │   │   ├── Menu.js                     ← Si se necesitara un JS del menú del cliente Network, aquí se colocaría
│       │   │   │   ├── ScoreNetworkEnvironment.js  ← JS del Score de Plantas Eléctricas del cliente NetworkEnvironment
│       │   │   │   └── ...
│       │   │   └── ...
│       │   ├── Direccion/                      ← Carpeta de JS desarrolladas para el menú de Direccion
│       │   ├── Ventas/                         ← Carpeta de JS desarrolladas para el menú de Ventas
│       │   ├── welcome.js                      ← JS de la pantalla de bienvenida
│       │   └── ...
│       ├── css/
│       │   ├── Clientes/                         ← Carpeta de CSS desarrollados para el menú de Clientes
│       │   │   ├── Menuppal.css                    ← Si se necesitara un CSS del menú principal de clientes, aquí se colocaría
│       │   │   ├── NetworkEnvironment/             ← Carpeta de CSS desarrollados para el cliente NetworkEnvironment
│       │   │   │   ├── Menu.css                      ← Si se necesitara un CSS del menú del cliente Network, aquí se colocaría
│       │   │   │   ├── ScoreNetworkEnvironment.css   ← CSS del Score de Plantas Eléctricas del cliente NetworkEnvironment
│       │   │   │   └── ...
│       │   │   └── ...
│       │   ├── Direccion/                        ← Carpeta de CSS desarrollados para el menú de Direccion
│       │   ├── Ventas/                           ← Carpeta de CSS desarrollados para el menú de Ventas
│       │   ├── welcome.css                       ← CSS de la pantalla de bienvenida
│       │   └── ...
│       ├── img/
│       │   ├── Clientes/                     ← Carpeta de imágenes usadas para el menú de Clientes
│       │   │   ├── Menuppal/                   ← Carpeta de imágenes usadas para el menú principal de clientes
│       │   │   ├── NetworkEnvironment/         ← Carpeta de imágenes desarrolladas para el cliente NetworkEnvironment
│       │   │   │   ├── Menu/                     ← Carpeta de imágenes usadas para el menú del cliente Network
│       │   │   │   ├── ScoreNetworkEnvironment/  ← Carpeta de imágenes usadas para el Score de Plantas Eléctricas del cliente NetworkEnvironment
│       │   │   │   └── ...
│       │   │   └── ...
│       │   ├── Direccion/                    ← Carpeta de imagenes usadas para el menú de Direccion
│       │   ├── Ventas/                       ← Carpeta de imagenes usadas para el menú de Ventas
│       │   ├── welcome/                      ← Carpeta de imagenes usadas para la pantalla de bienvenida
│       │   └── ...
│       └── ...
│   └── ...
└── ...
```

### Ejemplo Basico:

#### Estructura:

```

Elephant_D102/
├── src/
│   ├── main.py     ← Punto de entrada
│   ├── app.py      ← Lógica y rutas Flask
│   ├── config.py   ← Configuración por entorno
│   ├── .env        ← Variables sensibles
│   ├── App/        ← Carpeta de aplicaciones desarrolladas y conectadas con app.py
│   │   ├── __init__.py         ← Archivo para definir los paquetes Python dentro de App hacia app.py (importaciones, variables globales, lógica de configuración)
│   │   ├── Clientes.py         ← Archivo para renderizar el menú principal de clientes y la lógica del mismo HTML (src\templates\Clientes\Menuppal.html)
│   │   ├── Clientes_module/    ← Carpeta de aplicaciones de lógica backend desarrolladas para el menú de Clientes
│   │   │   ├── __init__.py                 ← Archivo para definir los paquetes Python dentro de Clientes_module hacia Clientes.py
│   │   │   ├── NetworkEnvironmentMenu.py   ← Archivo con la lógica del menú de Network Environment (src\templates\Clientes\NetworkEnvironment\Menu.html)
│   │   │   ├── NetworkEnvironment/         ← Carpeta de aplicaciones de lógica backend desarrolladas para el menú del cliente Network
│   │   │   │   ├── __init__.py             ← Archivo para definir los paquetes Python dentro de NetworkEnvironment hacia NetworkEnvironmentMenu.py
│   │   │   │   ├── ScoreNetworkEnvironment.py  ← Lógica del Score de Plantas Eléctricas del cliente Network
│   │   │   │   └── ...
│   │   │   └── ...
│   │   └── ...
│   ├── ConsultasSQL/
│   │   ├── Clientes/       ← Carpeta de aplicaciones de archivos que gestionan la base de datos de acuerdo a la lógica para el menú de Clientes
│   │   │   ├── NetworkEnvironment/
│   │   │   │   ├── ScoreNetworkEnvironmentQL.py  ← Aplicación que gestiona la base de datos de acuerdo a la lógica para el archivo ScoreNetwork.py
│   │   │   │   └── ...
│   │   │   └── ...
│   │   └── ...
│   └── templates/
│   │   ├── Clientes/                                 ← Carpeta de HTML desarrollados para el menú de Clientes
│   │   │   ├── Menuppal.html                         ← HTML del menú principal de clientes
│   │   │   ├── NetworkEnvironment/                   ← Carpeta de HTML desarrollados para el cliente NetworkEnvironment
│   │   │   │   ├── Menu.html                         ← HTML del menú del cliente Network
│   │   │   │   ├── ScoreNetworkEnvironment.html      ← HTML del Score de Plantas Eléctricas del cliente NetworkEnvironment
│   │   │   │   └── ...
│   │   │   └── ...
│   │   └── ...
│   └── static/   ← Carpeta donde se organizarán los archivos JS, CSS, PDF, IMG, etc.
│   │   ├── js/
│   │   │   ├── Clientes/                               ← Carpeta de JS desarrollados para el menú de Clientes
│   │   │   │   ├── Menuppal.js                         ← Si se necesitara un JS del menú principal de clientes, aquí se colocaría
│   │   │   │   ├── NetworkEnvironment/                 ← Carpeta de JS desarrollados para el cliente NetworkEnvironment
│   │   │   │   │   ├── Menu.js                         ← Si se necesitara un JS del menú del cliente Network, aquí se colocaría
│   │   │   │   │   ├── ScoreNetworkEnvironment.js      ← JS del Score de Plantas Eléctricas del cliente NetworkEnvironment
│   │   │   │   │   └── ...
│   │   │   │   └── ...
│   │   │   └── ...
│   │   ├── css/
│   │   │   ├── Clientes/                             ← Carpeta de CSS desarrollados para el menú de Clientes
│   │   │   │   ├── Menuppal.css                      ← Si se necesitara un CSS del menú principal de clientes, aquí se colocaría
│   │   │   │   ├── NetworkEnvironment/               ← Carpeta de CSS desarrollados para el cliente NetworkEnvironment
│   │   │   │   │   ├── Menu.css                      ← Si se necesitara un CSS del menú del cliente Network, aquí se colocaría
│   │   │   │   │   ├── ScoreNetworkEnvironment.css   ← CSS del Score de Plantas Eléctricas del cliente NetworkEnvironment
│   │   │   │   │   └── ...
│   │   │   │   └── ...
│   │   │   └── ...
│   │   ├── img/
│   │   │   ├── Clientes/                         ← Carpeta de imágenes usadas para el menú de Clientes
│   │   │   │   ├── Menuppal/                     ← Carpeta de imágenes usadas para el menú principal de clientes
│   │   │   │   ├── NetworkEnvironment/           ← Carpeta de imágenes desarrolladas para el cliente NetworkEnvironment
│   │   │   │   │   ├── Menu/                     ← Carpeta de imágenes usadas para el menú del cliente Network
│   │   │   │   │   ├── ScoreNetworkEnvironment/  ← Carpeta de imágenes usadas para el Score de Plantas Eléctricas del cliente NetworkEnvironment
│   │   │   │   │   └── ...
│   │   │   │   └── ...
│   │   │   └── ...
│   │   └── ...
│   └── ...
└── ...

Nota: Los archivos Menuppal.js, Menu.js, Menuppal.css y Menu.css no existen actualmente, ya que los estilos utilizados  
en esos archivos HTML son importados directamente desde el layout, y por el momento no requieren lógica en JavaScript.  
Por esta razón, dichos archivos han sido omitidos.

```

#### Ejemplo del archivo main.py de la estructura mostrada
```python
# Archivo: main.py
# Ruta: I:\Elephant_D102\main.py
# Lenguaje: Python con Flask

import sys
import os

# Agregar la carpeta `src` al PYTHONPATH
current_dir = os.path.dirname(os.path.abspath(__file__))
src_dir = os.path.join(current_dir, 'src')
sys.path.append(src_dir)

# Importar configuraciones y la app
from config import Productivo, ENVIRONMENT
from app import create_app  # Asegúrate que app.py tenga esta función

# Crear la instancia de la aplicación con base en el entorno
app = create_app(ENVIRONMENT)

# Ejecutar la aplicación
if __name__ == "__main__":
    if Productivo:
        app.run(host="0.0.0.0", port=5000)
    else:
        app.run(debug=True)
```

#### Ejemplo del archivo app.py de la estructura mostrada
```python
# Archivo: app.py
# Ruta: src/app.py
# Lenguaje: Python con Flask

import os
import subprocess
from flask import Flask, request, redirect, url_for, render_template, make_response, session
from flask_mail import Mail
from flask_babel import Babel
from config import config, Productivo

from App.Clientes import register_clientes_routes
from App.Ventas import register_ventas_routes
from App.Operaciones import register_operaciones_routes
from App.SupyCtrol import register_supyctrol_routes

from App.Traduccion import init_babel
from App.PreRegister import PreRegister_routes
from App.Security import register_security_routes
from App.Security_Module.UserAccess import check_user_access

# Inicializar extensiones fuera de la función
mail = Mail()

def create_app(entorno='development'):
    """Crea e inicializa la aplicación Flask según el entorno."""
    # Crear la instancia de Flask
    app = Flask(__name__)
    # Cargar configuración desde config.py
    app.config.from_object(config.get(entorno, config['default']))

    # Inicializar extensiones
    init_babel(app)
    mail.init_app(app)

    # Registrar todas las funciones de seguridad
    register_security_routes(app, mail)

    # Ruta para cambiar el idioma
    @app.route("/set_language/<lang_code>")
    def set_language(lang_code):
        if lang_code in app.config['BABEL_SUPPORTED_LOCALES']:
            referrer = request.referrer or url_for('welcome')
            resp = make_response(redirect(referrer))
            resp.set_cookie('lang', lang_code)
            return resp
        return redirect(url_for('welcome'))

    # RUTAS PRINCIPALES
    @app.route("/")
    def welcome():
        """
        Página principal de bienvenida.
        Renderiza el archivo `welcome.html`.
        """
        return render_template("welcome.html")

    @app.route("/login")
    def login():
        """
        Página de inicio de sesión.
        Renderiza el archivo `login.html`.
        """
        return render_template("Security/login.html")

    # Registrar rutas de PreRegistro
    PreRegister_routes(app)

    # RUTAS DEL MENÚ PRINCIPAL

    @app.route("/index")
    def index():
        """
        Página de inicio (index).
        Renderiza el archivo index.html.
        """
        UserID = session.get('user_id')
        ruta = "/index.html"
        return check_user_access(UserID, ruta, Restricted_Access)
    
    # Registrar rutas de Clientes
    register_clientes_routes(app, mail)    

    @app.route("/Direccion")
    def Direccion():
        """
        Página del menú de Dirección.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Dirección"

    # Registrar rutas de Ventas
    register_ventas_routes(app, mail)

    @app.route("/Servicios")
    def Servicios():
        """
        Página del menú de Servicios.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Servicios"
    
    # Registrar rutas de Operaciones
    register_operaciones_routes(app, mail)     
    
    # Registrar rutas de Supervisión y Control
    register_supyctrol_routes(app, mail)      

    @app.route("/Sitios")
    def Sitios():
        """
        Página del menú de Sitios.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Sitios"

    @app.route("/About_us")
    def About_us():
        """
        Página de Acerca de (About Us).
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - About Us"
```

#### Ejemplo del archivo __init__.py de la estructura mostrada (src\App\__init__.py)
```python
# Archivo: Clientes.py
# Ruta: src\App\__init__.py
# Descripción: Archivo para definir los paquetes Python dentro de APP hacia app.py (Importaciones, variables globales, lógica de configuración)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

# Importa init_babel para configuraciones relacionadas con traducción
from .Traduccion import init_babel

# Importa SupyCtrol para asegurar que las rutas se registren correctamente
from .SupyCtrol import register_supyctrol_routes

# Importa Ventas para asegurar que las rutas se registren correctamente
from .Ventas import register_ventas_routes

# Importa Operaciones para asegurar que las rutas se registren correctamente
from .Operaciones import register_operaciones_routes

# Importa Clientes para asegurar que las rutas se registren correctamente
from .Clientes import register_clientes_routes

# Importa Archivos de Raiz para asegurar que las rutas se registren correctamente
from .PreRegister import PreRegister_routes
from .Security import register_security_routes 
from .Subir_Archivo import subir_archivo_ftp_desde_request
```
#### Ejemplo del archivo Clientes.py de la estructura mostrada
```python
# Archivo: Clientes.py
# Ruta: src\App\Clientes.py
# Descripción: Archivo para renderizar el menu principal de clientes y la logica del mismo html (src\templates\Clientes\Menuppal.html)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from flask import render_template, session
from App.Security_Module.UserAccess import check_user_access
from App.Clientes_Module.NetworkEnvironmentMenu import menu_NetworkEnvironment

def register_clientes_routes(app, mail):
    """
    Registra las rutas relacionadas con Operaciones en la aplicación Flask.
    """

    # ================== RUTA PRINCIPAL ================== (Se renderiza a si mismo src/templates/Clientes/Menuppal.html)
    @app.route("/Clientes")
    def Clientes():
        """
        Renderiza el html propio de esta logica.
        Archivo: Menuppal.html
        Ruta: src/templates/Clientes/Menuppal.html
        """
        Restricted_Access = False
        UserID = session.get('user_id')
        ruta = "/Clientes/Menuppal.html"  
        return check_user_access(UserID, ruta, Restricted_Access)
    
# Logica del menú de Clientes
    
    # ================== SUBMENÚ: SCORE ==================
    @app.route("/Clientes/NetworkEnvironment")
    def NetworkEnvironmentMenu():
        """
        Renderiza el menú del Data Master.
        Archivo: Menu.html
        Ruta: src/templates/Clientes/Score/Menu.html
        """
        Restricted_Access = True
        UserID = session.get('user_id')
        ruta = "/Clientes/NetworkEnvironment/Menu.html"
        
        return check_user_access(UserID, ruta,Restricted_Access)
    
    menu_NetworkEnvironment(app, mail)
```
#### Ejemplo del archivo __init__.py de la estructura mostrada (src\App\Clientes_Module\__init__.py)
```python
# Archivo: __init__.py
# Ruta: src\App\Clientes_Module\__init__.py
# Descripción: Archivo para definir los paquetes Python dentro de Clientes_Module hacia Clientes.py (Importaciones, variables globales, lógica de configuración)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from .NetworkEnvironmentMenu import menu_NetworkEnvironment
```
#### Ejemplo del archivo NetworkEnvironmentMenu.py de la estructura mostrada
```python
# Archivo: NetworkEnvironmentMenu.py
# Ruta: src\App\Clientes_Module\NetworkEnvironmentMenu.py
# Descripción: Archivo con la logica del menú de Network Environment (src\templates\Clientes\NetworkEnvironment\Menu.html)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from flask import render_template, session
from App.Security_Module.UserAccess import check_user_access
from App.Clientes_Module.NetworkEnvironment import function_ScoreNetworkEnvironment

def menu_NetworkEnvironment(app, mail):
    """
    Registra las rutas relacionadas con Clientes en la aplicación Flask.
    """
    # ================== Modulo de CLIENTE_SCORE ==================
    @app.route("/Clientes/ClientesScore/ClientScore")
    @app.login_required
    def ScoreNetworkEnvironment():
        """
        Renderiza la página de catalogo de Cliente Score.
        """
        Restricted_Access = True
        UserID = session.get('user_id')
        ruta = "/Clientes/NetworkEnvironment/ScoreNetworkEnvironment.html"

        return check_user_access(UserID, ruta,Restricted_Access)
    
    function_ScoreNetworkEnvironment(app, mail)    
```
#### Ejemplo del archivo __init__.py de la estructura mostrada (src\App\Clientes_Module\NetworkEnvironment\__init__.py)
```python
# Archivo: __init__.py
# Ruta: src\App\Clientes_Module\NetworkEnvironment\__init__.py
# Descripción: Archivo para definir los paquetes Python dentro de NetworkEnvironment hacia NetworkEnvironmentMenu.py (Importaciones, variables globales, lógica de configuración)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from .ScoreNetworkEnvironment import function_ScoreNetworkEnvironment
```
#### Ejemplo del archivo ScoreNetworkEnvironment.py de la estructura mostrada
```python
# Archivo: ScoreNetworkEnvironment.py
# Ruta: src\App\Clientes_Module\NetworkEnvironment\ScoreNetworkEnvironment.py
# Descripción: Archivo con la logica del Score de Network Environment (src\templates\Clientes\NetworkEnvironment\ScoreNetworkEnvironment.html)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from datetime import datetime
from venv import logger
from flask import jsonify, render_template, request, session
from App.Security_Module.UserAccess import check_user_access

# Importar funciones SQL
from Consultas_SQL.Clientes.NetworkEnvironment.ScoreNetworkEnvironmentQL import (
    buscar_score_view
);

def function_ScoreNetworkEnvironment(app, mail):
    """
    Agrupamos todas las funciones que importamos app en la aplicación Flask.
    """

    @app.route("/Clientes/ClientesScore/BuscarScoreView", methods=['POST'])
    def buscar_score_view_endpoint():
        """
        Endpoint para buscar el score más reciente en la base de datos.
        """

        try:
            logger.info("Buscando Score mas reciente...")

            #Busca Score en la BD
            score_date = buscar_score_view()

            if score_date:
                #Score encontrado
                logger.info("Score encontrado exitosamente")
                return jsonify({
                    "success": True,
                    "data": score_date,
                    "message": "Score encontrado exitosamente"
                }), 200
            else:
                # Score no encntrado
                logger.warning("No se encontraron registros de Score")
                return jsonify({
                    "success": False,
                    "error": {
                        "message": "No se encontraron registros de score en el sistema",
                        "code": "SCORE_NOT_FOUND",
                        "alert_type": "info"
                    }
                }), 404
        except ConnectionError as e:
            logger.error(f"Error de conexión en búsqueda de score: {str(e)}")
            return jsonify({
                "success": False,
                "error": {
                    "message": "Error de conexión con la base de datos. Intente nuevamente en unos momentos.",
                    "code": "DATABASE_CONNECTION_ERROR",
                    "alert_type": "error"
                }
            }), 500
        
        except Exception as e:
            logger.error(f"Error inesperado en búsqueda de score: {str(e)}", exc_info=True)
            return jsonify({
                "success": False,
                "error": {
                    "message": "Error interno del servidor. Si el problema persiste, contacte al administrador.",
                    "code": "INTERNAL_SERVER_ERROR",
                    "alert_type": "error",
                    "timestamp": datetime.now().isoformat()
                }
            }), 500
```
#### Ejemplo del archivo ScoreNetworkEnvironmentQL.py de la estructura mostrada
```python
# Archivo: ScoreNetworkEnvironmentQL.py
# Ruta: src\Consultas_SQL\Clientes\NetworkEnvironment\ScoreNetworkEnvironmentQL.py
# Descripción: Aplicacion que gestionan la base de datos de acuerdo a la logica para el archivo ScoreNetwork.py (src\App\Clientes_Module\NetworkEnvironment\ScoreNetworkEnvironment.py)
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

import pyodbc
import logging
from datetime import datetime
from typing import Dict, List, Optional, Union
from config import Productivo, ENVIRONMENT
from Consultas_SQL.conexion import get_connection

# Configurar logging
logger = logging.getLogger('clientes_score_sql')

def buscar_score_view() -> Optional[List[Dict]]:
    """
    Buscar los registros de score en la base de datos.
    """
    query = """
        SELECT 
            Score.[OrderNum&Line] AS ordernumandline,
            Score.OrderNum, Score.Capacidad,
            Score.Voltaje, Score.Tipo, 
            Score.Caseta, Score.JobNum2,
            Score.FechaProducción,
            Score.FechaMat, Score.AvanceDeSurtimiento,
            Score.AvisoDeTerminacion, Score.MaterialFaltante,
            Score.PartNum_M, Score.Description_M,
            Score.RefCategory_M, 
            Score.NoSerie_M, Score.Faltante_M,
            Score.En_PO_Altern_M, Score.Estatus_M,
            Score.Fecha_Llegada_M, Score.PartNum_G,
            Score.Description_G, 
            Score.NoSerie_G,
            Score.Faltante_G, Score.En_PO_Altern_G,
            Score.Estatus_G, Score.Fecha_Llegada_G
        FROM 
            Score 
        WHERE
            Score.OrderNum IN (
            '126309', '126308', '126307', '126306', '126305', '126304', '126303', '126302', '126301', '126300',
            '126299', '126298', '126297', '126296', '126295', '126294', '126293', '126292', '126291', '126290',
            '126289', '126288', '126287', '126286', '126285', '126284', '126283', '126282', '126281', '126280',
            '126279', '126278', '126277', '126276', '126275', '126274', '126273', '126272', '126271', '126270',
            '126269', '126268', '126267', '126266', '126265', '126264', '126263', '126262', '126261', '126260',
            '126259', '126258', '126257', '126256', '126255', '126254', '126253', '126252', '126251', '126250',
            '126249', '126248', '126247', '126246', '126245', '126244'
        )
        """

    # Resto del código y manejo de errores...


```
#### Ejemplo del archivo Menuppal.html de la estructura mostrada
```html
<!--
Archivo: Menuppal.html
Ruta: src\templates\Clientes\Menuppal.html
Descripción: Plantilla para el menú principal de clientes.
Autor: Equipo de Desarrollo IGSA
Fecha: 2025
-->

{% extends "layout.html" %}

{% block title %}{{ _('Menú Clientes - IGSA') }}{% endblock %}

{% block content %}

<main class="page">
    <section class="clean-block clean-blog-list dark">
        <div class="container">
            <div class="block-heading">
                <h2 class="text-info">{{ _('Clientes') }}</h2>
                <p>{{ _('Bienvenido al sistema de Clientes. Aquí podrá dar seguimiento de proyectos de con IGSA.') }}</p>
            </div>
            <div class="block-content">
                <!-- NETWORK ENVIRONMENT -->
                <div class="clean-blog-post">
                    <div class="row align-items-center">
                        <div class="col-lg-5">
                            <img class="rounded img-fluid" src="{{ url_for('static', filename='img/tech/image4.jpg') }}" alt="{{ _('Network Environment') }}">
                        </div>
                        <div class="col-lg-7">
                            <h3>{{ _('Network Environment') }}</h3>
                            <p>{{ _('Módulos y herramientas diseñados para la gestión y seguimiento de proyectos de plantas eléctricas de emergencia para Network Environment.') }}</p>
                            <a class="btn btn-outline-primary btn-sm" href="{{ url_for('NetworkEnvironmentMenu') }}">{{ _('Ir a Menu') }}</a>
                        </div>
                    </div>
                </div>                
            </div>
        </div>
    </section>
</main>
{% endblock %}
```
#### Ejemplo del archivo Menu.html de la estructura mostrada
```html
<!--
Archivo: Menu.html
Ruta: src\templates\Clientes\NetworkEnvironment\Menu.html
Descripción: Plantilla para el menú de Network Environment.
Autor: Equipo de Desarrollo IGSA
Fecha: 2025
-->

{% extends "layout.html" %}

{% block title %}{{ _('Score Plantas Eléctricas - Network Environment | IGSA') }}{% endblock %}

{% block content %}

<main class="page">
    <section class="clean-block clean-services dark">
        <div class="container">
            <div class="block-heading">
                <h2 class="text-info">{{ _('Network Environment') }}</h2>
                <p>
                    {{ _('Módulos y herramientas diseñados para la gestión y seguimiento de proyectos de plantas eléctricas de emergencia para Network Environment.') }}
                </p>
            </div>
            <div class="row">
                <!-- Planta Eléctrica -->
                <div class="col-md-6 col-lg-4">
                    <div class="card">
                        <img class="card-img-top w-100 d-block" src="{{ url_for('static', filename='img/scenery/image5.jpg') }}" alt="{{ _('Plantas Eléctricas') }}">
                        <div class="card-body">
                            <h4 class="card-title">{{ _('Score de Network Environment') }}</h4>
                            <p class="card-text">
                                {{ _('Visualiza el score consolidado de todos los proyectos de fabricación y seguimiento de plantas eléctricas de emergencia para Network Environment.') }}
                            </p>
                            <a class="btn btn-outline-primary btn-sm" href="{{ url_for('ScoreNetworkEnvironment') }}">
                                {{ _('Ir al módulo') }}
                            </a>
                        </div>
                    </div>
                </div>                
            </div>
        </div>
    </section>
</main>

{% endblock %}

```
#### Ejemplo del archivo ScoreNetworkEnvironment.html de la estructura mostrada
```html
<!--
Archivo: ScoreNetworkEnvironment.html
Ruta: src\templates\Clientes\NetworkEnvironment\ScoreNetworkEnvironment.html
Descripción: Plantilla para visualizar el score de Network Environment.
Autor: Equipo de Desarrollo IGSA
Fecha: 2025
-->

{% extends "layout.html" %}

{% block title %}{{ _('Score') }}{% endblock %}

{% block content %}

<!-- Conexion de CSS Casetas -->
<link rel="stylesheet" href="{{ url_for('static', filename='css/Clientes/NetworkEnvironment/ScoreNetworkEnvironment.css') }}">

<!-- DataTables CSS (Última versión estable) -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/jquery.dataTables.min.css">

<!-- Botones de DataTables (Última versión estable) -->
<link rel="stylesheet" href="https://cdn.datatables.net/buttons/2.3.6/css/buttons.dataTables.min.css">

<!-- FontAwesome (para iconos) -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">

<main class="page">

  <!-- Aqui va todo el codigo del HTML desarrollado -->

</main>

<!-- jQuery (Debe ir antes de cualquier otro script de DataTables) -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

<!-- DataTables -->
<script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>

<!-- Botones de DataTables (Última versión estable) -->
<script src="https://cdn.datatables.net/buttons/2.3.6/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.3.6/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.3.6/js/buttons.print.min.js"></script>

<!-- Dependencias para exportar a Excel/PDF -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.1.3/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.36/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.36/vfs_fonts.js"></script>

<script>
    var TXT_BUTTON = "{{_('Exportar a Excel')}}";
    var TXT_DETAIL = "{{_('Detalle:')}}";
</script>
<!-- Conexion de JS Casetas -->
<script src="{{ url_for('static', filename='js/Clientes/NetworkEnvironment/ScoreNetworkEnvironment.js') }}"></script>

{% endblock %}
```
#### Ejemplo del archivo ScoreNetworkEnvironment.js de la estructura mostrada
```js
/**
 * Archivo: ScoreNetworkEnvironment.js
 * Ruta: src\static\js\Clientes\NetworkEnvironment\ScoreNetworkEnvironment.js
 * Descripción: Funciones JavaScript optimizadas para el módulo de score de Network Environment
 * Autor: Equipo de Desarrollo IGSA
 * Fecha: 2025
 */

// ===============================
// 🔹 CONSTANTES Y VARIABLES
// ===============================
const loadingIndicator = document.getElementById("loading-indicator"); // Indicador de carga
let table; // Variable para almacenar la instancia de DataTable
let selectedDistributorId = null; // ID del distribuidor seleccionado
let selectedDistributorIdForReject = null;

// Resto del código...

```
#### Ejemplo del archivo ScoreNetworkEnvironment.css de la estructura mostrada
```css
/*
 * Archivo: ScoreNetworkEnvironment.css
 * Ruta: src\static\css\Clientes\NetworkEnvironment\ScoreNetworkEnvironment.css
 * Descripción: Estilos para el módulo de score de Network Environment
 * Autor: Equipo de Desarrollo IGSA
 * Fecha: 2025
*/

/*----------------------------------------------
 1. Variables Globales (CSS Custom Properties)
----------------------------------------------*/
:root {
    /* Colores principales */
    --primary-color: #007bff;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    
/*Resto del código ...*/
```

---