# Archivo: Testmail.py
# Ruta: src\Testmail.py
# Descripción: Módulo de prueba para enviar correos electrónicos utilizando Flask-Mail.
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

import os
import sys
from datetime import datetime
from flask import Flask
from config import config
from App.Utilities_module.MailManagement import enviar_correo_universal, obtener_info_envio, generar_preview_correo

def crear_app_prueba():
    """
    Crea una instancia de Flask para pruebas de correo.
    """
    app = Flask(__name__)
    
    # Cargar configuración
    config_name = os.getenv('FLASK_ENV', 'development')
    app.config.from_object(config[config_name])
    
    return app

def test_envio_cotizacion_especial():
    """
    Prueba el envío de correo para cotización especial.
    """
    
    print("Iniciando prueba de envio de correo - Cotizacion Especial")
    print("=" * 60)
    
    # Configuración de la prueba
    MAIL_LIST_ID = 1000  # Cambiar por un ID real de tu base de datos
    
    # Datos del template
    template_data = {
        'numero_cotizacion': 'COT-2024-001',
        'nombre_cliente': 'Empresa XYZ S.A. de C.V.',
        'nombre_contacto': 'Ing. Juan Carlos Pérez',
        'fecha_solicitud': datetime.now().strftime('%d de %B, %Y'),
        'ejecutivo_asignado': 'Ing. María González',
        'tiempo_estimado': '3-5 días hábiles',
        'descripcion_proyecto': 'Desarrollo de sistema integral de gestión empresarial con módulos de ventas, inventario y facturación electrónica.',
        'enlace_seguimiento': 'https://sycelephant.com/seguimiento/COT-2024-001',
        'telefono_contacto': '+52 (999) 123-4567',
        'email_contacto': 'ventas@igsa.com'
    }
    
    # Archivos adjuntos mixtos (local + URL)
    archivos_adjuntos = [
        {
            'tipo': 'local',
            'ruta': 'static/pdf/Ventas/Cotizaciones/EUA/DMITEMS/BREAKER_CATALOG/Carlitos.pdf',
            'nombre': 'Catalogo_Breakers.pdf',
            'mimetype': 'application/pdf'
        },
        {
            'tipo': 'url',
            'ruta': 'https://file.sycelephant.com/Ventas/Formularios/1002-1/1002-1-FormPEmx/Alta_de_Bases.png',
            'nombre': 'Alta_de_Bases.png',
            'mimetype': 'image/png'
        }
    ]
    
    # Destinatarios adicionales
    destinatarios_adicionales = {
        'TO': ['cliente.prueba@test.com'],
        'CC': ['supervisor.prueba@test.com'],
        'BCC': ['auditoria@igsa.com']
    }
    
    try:
        # Crear app de prueba
        app = crear_app_prueba()
        
        with app.app_context():
            # CASO 1: Envío usando BD + destinatarios adicionales
            print("CASO 1: BD + Destinatarios Adicionales")
            print("-" * 40)
            
            # asunto='Solicitud de Cotización Especial - COT-2024-001',
            
            resultado = enviar_correo_universal(
                template_path='Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                
                template_data=template_data,
                mail_list_id=MAIL_LIST_ID,
                destinatarios_adicionales=destinatarios_adicionales,
                archivos_adjuntos=archivos_adjuntos
            )
            
            mostrar_resultado(resultado, "Caso 1")
            
            # CASO 2: Envío solo con destinatarios adicionales (sin BD)
            print("\nCASO 2: Solo Destinatarios Adicionales")
            print("-" * 40)
            
            # asunto='Prueba - Solo Destinatarios Adicionales',
            
            resultado2 = enviar_correo_universal(
                template_path='Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                asunto='Prueba - Solo Destinatarios Adicionales',
                template_data=template_data,
                destinatarios_adicionales={
                    'TO': ['cliente.directo@test.com', 'vendedor@igsa.com'],
                    'CC': ['gerente@igsa.com']
                },
                archivos_adjuntos=[archivos_adjuntos[0]]  # Solo archivo local
            )
            
            mostrar_resultado(resultado2, "Caso 2")
            
            # CASO 3: Envío solo con BD (sin destinatarios adicionales)
            print("\nCASO 3: Solo Base de Datos")
            print("-" * 40)
            
            resultado3 = enviar_correo_universal(
                template_path='Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                asunto='Prueba - Solo Base de Datos',
                template_data=template_data,
                mail_list_id=MAIL_LIST_ID,
                archivos_adjuntos=[archivos_adjuntos[1]]  # Solo archivo URL
            )
            
            mostrar_resultado(resultado3, "Caso 3")
            
            # Resumen final
            print("\n" + "=" * 60)
            casos_exitosos = sum([
                1 if resultado['success'] else 0,
                1 if resultado2['success'] else 0,
                1 if resultado3['success'] else 0
            ])
            print(f"RESUMEN: {casos_exitosos}/3 casos exitosos")
            
            return casos_exitosos >= 1  # Al menos un caso exitoso
            
    except Exception as e:
        print(f"Error crítico en la prueba: {str(e)}")
        return False


def mostrar_resultado(resultado, caso_nombre):
    """
    Muestra los resultados de un envío de correo de forma detallada.
    """
    print(f"RESULTADOS - {caso_nombre}:")
    print(f"Exito: {resultado['success']}")
    print(f"Mensaje: {resultado['mensaje']}")
    print(f"MailListID: {resultado['mail_list_id']}")
    print(f"Template: {resultado['template_usado']}")
    print(f"Tiempo: {resultado['tiempo_ejecucion']}")
    
    # Destinatarios
    dest = resultado['destinatarios']
    print(f"DESTINATARIOS ({dest['total_destinatarios']} total):")
    if dest['TO']:
        print(f"   TO: {', '.join(dest['TO'])}")
    if dest['CC']:
        print(f"   CC: {', '.join(dest['CC'])}")
    if dest['BCC']:
        print(f"   BCC: {', '.join(dest['BCC'])}")
    
    # Archivos adjuntos
    if resultado['archivos_enviados']:
        print(f"ARCHIVOS ADJUNTOS ({len(resultado['archivos_enviados'])}):")
        for archivo in resultado['archivos_enviados']:
            print(f"   - {archivo}")
    
    # Warnings
    if resultado['warnings']:
        print("ADVERTENCIAS:")
        for warning in resultado['warnings']:
            print(f"   - {warning}")
    
    # Errores
    if resultado['errores']:
        print("ERRORES:")
        for error in resultado['errores']:
            print(f"   - {error}")
    
    print()


def test_validacion_parametros():
    """
    Prueba la validación de parámetros de entrada.
    """
    
    print("\nIniciando prueba de validación de parámetros")
    print("=" * 50)
    
    app = crear_app_prueba()
    
    with app.app_context():
        # Casos de prueba
        casos_prueba = [
            {
                'nombre': 'Template vacío',
                'params': {
                    'template_path': None,
                    'asunto': 'Prueba template vacío'
                },
                'debe_fallar': True
            },
            {
                'nombre': 'Asunto vacío',
                'params': {
                    'template_path': 'Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                    'asunto': None
                },
                'debe_fallar': True
            },
            {
                'nombre': 'MailListID inexistente',
                'params': {
                    'template_path': 'Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                    'asunto': 'Prueba MailListID inexistente',
                    'mail_list_id': 99999
                },
                'debe_fallar': False  # Debe continuar con warning
            },
            {
                'nombre': 'Template inexistente',
                'params': {
                    'template_path': 'template_inexistente.html',
                    'asunto': 'Prueba template inexistente',
                    'destinatarios_adicionales': {'TO': ['test@example.com']}
                },
                'debe_fallar': True
            },
            {
                'nombre': 'Caso válido solo con destinatarios',
                'params': {
                    'template_path': 'Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
                    'asunto': 'Prueba válida',
                    'destinatarios_adicionales': {'TO': ['test@example.com']}
                },
                'debe_fallar': False
            }
        ]
        
        for caso in casos_prueba:
            print(f"\nProbando: {caso['nombre']}")
            
            try:
                resultado = enviar_correo_universal(**caso['params'])
                
                if caso['debe_fallar']:
                    if not resultado['success']:
                        print(f"   Falló correctamente: {resultado['mensaje']}")
                    else:
                        print(f"   Debería haber fallado pero tuvo éxito")
                else:
                    if resultado['success'] or resultado['warnings']:
                        print(f"   Resultado esperado: {resultado['mensaje']}")
                    else:
                        print(f"   Falló inesperadamente: {resultado['mensaje']}")
                        
            except Exception as e:
                print(f"   Error inesperado: {str(e)}")


def test_solo_destinatarios_adicionales():
    """
    Prueba específica para envío solo con destinatarios adicionales.
    """
    
    print("\nIniciando prueba SOLO destinatarios adicionales")
    print("=" * 50)
    
    app = crear_app_prueba()
    
    with app.app_context():
        # Datos del template
        template_data = {
            'numero_cotizacion': 'COT-TEST-001',
            'nombre_cliente': 'Cliente Prueba S.A.',
            'nombre_contacto': 'Ing. Test',
            'fecha_solicitud': datetime.now().strftime('%d de %B, %Y')
        }
        
        print("Enviando correo solo con destinatarios adicionales...")
        
        resultado = enviar_correo_universal(
            template_path='Emails/Ventas/Cotiz/CotizEspSolicitudMail.html',
            asunto='Prueba - Solo Destinatarios Adicionales',
            template_data=template_data,
            destinatarios_adicionales={
                'TO': ['prueba1@test.com', 'prueba2@test.com'],
                'CC': ['supervisor@test.com']
            }
        )
        
        mostrar_resultado(resultado, "Solo Destinatarios")
        
        return resultado['success']


def main():
    """
    Función principal para ejecutar las pruebas.
    """
    
    print("IGSA ELEPHANT - PRUEBAS DE CORREO ELECTRONICO")
    print("=" * 60)
    print(f"Fecha: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"Entorno: {os.getenv('FLASK_ENV', 'development')}")
    print()
    
    try:
        # Ejecutar pruebas
        print("Iniciando batería de pruebas...")
        
        # 1. Prueba de validación de parámetros
        test_validacion_parametros()
        
        # 2. Prueba específica solo destinatarios adicionales
        success_adicionales = test_solo_destinatarios_adicionales()
        
        # 3. Prueba principal de envío
        success_principal = test_envio_cotizacion_especial()
        
        print("\n" + "=" * 60)
        if success_adicionales or success_principal:
            print("ALGUNAS PRUEBAS COMPLETADAS EXITOSAMENTE")
            print("El sistema de correo está funcionando")
            if not success_principal:
                print("NOTA: Crear MailListID 1001 en BD para pruebas completas")
        else:
            print("PROBLEMAS EN LAS PRUEBAS")
            print("Revisar configuración y logs")
            
    except KeyboardInterrupt:
        print("\n\nPruebas interrumpidas por el usuario")
    except Exception as e:
        print(f"\n\nError crítico en las pruebas: {str(e)}")


if __name__ == '__main__':
    main()
    
