# Archivo: app.py
# Ruta: src/app.py
# Lenguaje: Python con Flask

import os
import subprocess
from flask import Flask, request, redirect, url_for, render_template, make_response, session, send_from_directory
from flask_mail import Mail
from flask_babel import Babel
from config import config, Productivo

from App.Clientes import register_clientes_routes
from App.Ventas import register_ventas_routes
from App.Operaciones import register_operaciones_routes
from App.SupyCtrol import register_supyctrol_routes

from App.Traduccion import init_babel
from App.PreRegister import PreRegister_routes
from App.Security import register_security_routes
from App.Security_Module.UserAccess import check_user_access

from App.api.controller.business_central_controller import business_central_router
# En la sección de importaciones de app.py
# ...
from App.api.controller.quote_reception_controller import router as quote_reception_router # <-- Nuevo Import
from App.api.controller.quotation_Controller import router as quotation_router
from App.api.controller.FtpController import router as ftp_router
from App.api.controller.quotation_cotiz_controller import router as quotation_cotiz_router

from Consultas_SQL.Operaciones.Ingenieria.Cotiz.CotizCreatedIngSQL import  router as CotizCreatedIngSQL_router


# Inicializar extensiones fuera de la función
mail = Mail()



def create_app(entorno='development'):
    """Crea e inicializa la aplicación Flask según el entorno."""
    # Crear la instancia de Flask
    app = Flask(__name__)
    # Cargar configuración desde config.py
    app.config.from_object(config.get(entorno, config['default']))

    # Inicializar extensiones
    init_babel(app)
    mail.init_app(app)

    # Registrar todas las funciones de seguridad
    register_security_routes(app, mail)


    app.register_blueprint(business_central_router, url_prefix='/api/bc')
    
    app.register_blueprint(quote_reception_router, url_prefix='/api/reception')
    app.register_blueprint(quotation_router, url_prefix='/api/quotation')
    app.register_blueprint(ftp_router, url_prefix='/api/ftp')
    app.register_blueprint(quotation_cotiz_router, url_prefix='/api/quotation/cotiz')

    app.register_blueprint(CotizCreatedIngSQL_router, url_prefix='/api/Operaciones/Ingenieria/Cotiz/CotizCreatedIng')


    # Ruta para cambiar el idioma
    @app.route("/set_language/<lang_code>")
    def set_language(lang_code):
        if lang_code in app.config['BABEL_SUPPORTED_LOCALES']:
            referrer = request.referrer or url_for('welcome')
            resp = make_response(redirect(referrer))
            resp.set_cookie('lang', lang_code)
            return resp
        return redirect(url_for('welcome'))


    @app.route('/templates/<path:filename>')
    def serve_template(filename):
        return send_from_directory('templates', filename)

    # RUTAS PRINCIPALES
    @app.route("/")
    def welcome():
        """
        Página principal de bienvenida.
        Renderiza el archivo `welcome.html`.
        """
        return render_template("welcome.html")

    @app.route("/login")
    def login():
        """
        Página de inicio de sesión.
        Renderiza el archivo `login.html`.
        """
        return render_template("Security/login.html")

    # Registrar rutas de PreRegistro
    PreRegister_routes(app)

    # RUTAS DEL MENÚ PRINCIPAL

    @app.route("/index")
    def index():
        """
        Página de inicio (index).
        Renderiza el archivo index.html.
        """
        UserID = session.get('user_id')
        ruta = "/index.html"
        Restricted_Access = False
        return check_user_access(UserID, ruta, Restricted_Access)
    
    # Registrar rutas de Clientes
    register_clientes_routes(app, mail)    

    @app.route("/Direccion")
    def Direccion():
        """
        Página del menú de Dirección.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Dirección"

    # Registrar rutas de Ventas
    register_ventas_routes(app, mail)

    @app.route("/Servicios")
    def Servicios():
        """
        Página del menú de Servicios.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Servicios"
    
    # Registrar rutas de Operaciones
    register_operaciones_routes(app, mail)     
    
    # Registrar rutas de Supervisión y Control
    register_supyctrol_routes(app, mail)      

    @app.route("/Sitios")
    def Sitios():
        """
        Página del menú de Sitios.
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - Sitios"

    @app.route("/About_us")
    def About_us():
        """
        Página de Acerca de (About Us).
        Actualmente muestra un mensaje de desarrollo.
        """
        return "Página en desarrollo - About Us"
    
    #Probar Puppeteer
    # 1. Endpoint de prueba con HTML simple
    @app.route('/mi_html')
    def mi_html():
        # HTML sencillo con un título y un párrafo
        html = """
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Prueba de PDF</title>
        </head>
        <body>
            <h1>Hola, este es un HTML de prueba </h1>
            <p>Convirtiendo HTML a PDF usando Node.js + Puppeteer.</p>
        </body>
        </html>
        """
        return html
    # 2. Endpoint para generar el PDF en D:\PruebasPDF
    @app.route('/generar_pdf')
    def generar_pdf():
        # Construir la URL local de Flask para el HTML de prueba
        flask_url = url_for('mi_html', _external=True) # Ajusta el puerto si usas otro

        # Ruta donde guardarás el PDF (ejemplo en D:\PruebasPDF)
        if Productivo:
            pdf_path = r'/mnt/pruebapdf/prueba.pdf'
        else:
            pdf_path = r'D:\PruebasPDF\prueba.pdf' #Alexis
            #pdf_path = r'E:\PruebasPDF\prueba.pdf' #Carlos
        
        # Ruta al script Node.js (generate_pdf.js) dentro de tu proyecto
        # Si "generate_pdf.js" está en la raiz de la carpeta de puppeteer_pdf, usa:
        script_path = os.path.join(app.root_path, 'puppeteer_pdf', 'generate_pdf.js')
        # Si "generate_pdf.js" está dentro de una carpeta que esta dentro de la carpeta de puppeteer_pdf (src\puppeteer_pdf\Pruebas\generate_pdf.js), usa:
        # script_path = os.path.join(app.root_path, 'puppeteer_pdf', 'Pruebas', 'generate_pdf.js')
        
        try:
            # Llamamos al script de Node con subprocess.run
            # Pasamos como argumentos la URL de Flask y la ruta de salida del PDF
            subprocess.run(["node", script_path, flask_url, pdf_path], check=True)
            return f"PDF generado en: {pdf_path}"
        except subprocess.CalledProcessError as e:
            return f"Error al generar PDF: {e}"

    return app
        
    