# Archivo: config.py
# Ruta: src\config.py
# Descripción: Módulo de configuración de variables de entorno
# Autor: Equipo de Desarrollo IGSA
# Fecha: 2025

from dotenv import load_dotenv
import os

# Cargar variables desde el archivo .env
load_dotenv()

ENVIRONMENT = os.getenv("FLASK_ENV", "development")
IS_PRODUCTION = ENVIRONMENT == "production"

if IS_PRODUCTION:
    Productivo = True
else:
    Productivo = False

if Productivo == False:
    Host = "localhost:5000"
    SCHEME = "http"
else:
    Host = "sycelephant.com"
    SCHEME = "https" 

class Config:
    """Configuraciones generales."""
    SECRET_KEY = os.getenv("SECRET_KEY")  # Clave secreta desde .env
    BABEL_DEFAULT_LOCALE = 'es'
    BABEL_SUPPORTED_LOCALES = ['en', 'es']
    
    # Configuración de correo electrónico
    MAIL_SERVER = os.getenv('MAIL_SERVER')
    MAIL_PORT = int(os.getenv('MAIL_PORT', 587))
    MAIL_USE_TLS = os.getenv('MAIL_USE_TLS', 'True') == 'True'
    MAIL_USERNAME = os.getenv('MAIL_USERNAME')
    MAIL_PASSWORD = os.getenv('MAIL_PASSWORD')
    MAIL_DEFAULT_SENDER = os.getenv('MAIL_USERNAME')  # Opcional: usar el mismo correo como remitente predeterminado

    # 🆕 Agregar estas líneas para Office 365
    MAIL_USE_SSL = os.getenv('MAIL_USE_SSL', 'False') == 'True'
    MAIL_ASCII_ATTACHMENTS = os.getenv('MAIL_ASCII_ATTACHMENTS', 'False') == 'True'
    MAIL_SUPPRESS_SEND = os.getenv('MAIL_SUPPRESS_SEND', 'False') == 'True'     
    
    # Configuración OneDrive
    ONEDRIVE_CLIENT_ID = os.getenv('ONEDRIVE_CLIENT_ID')
    ONEDRIVE_CLIENT_SECRET = os.getenv('ONEDRIVE_CLIENT_SECRET')
    ONEDRIVE_REDIRECT_URI = os.getenv('ONEDRIVE_REDIRECT_URI')
    ONEDRIVE_USERNAME = os.getenv('ONEDRIVE_USERNAME')
    ONEDRIVE_PASSWORD = os.getenv('ONEDRIVE_PASSWORD')
    
    # Configuración de archivos
    UPLOAD_FOLDER = os.getenv('UPLOAD_FOLDER', 'static/uploads/quotes')
    ALLOWED_EXTENSIONS = set(os.getenv('ALLOWED_EXTENSIONS', 'pdf,xlsx,docx,png,jpg,jpeg').split(','))
    MAX_CONTENT_LENGTH = int(os.getenv('MAX_CONTENT_LENGTH', 16 * 1024 * 1024))  # 16MB por defecto
    
    BC_TENANT_ID = os.getenv("BC_TENANT_ID")
    BC_CLIENT_ID = os.getenv("BC_CLIENT_ID")
    BC_CLIENT_SECRET = os.getenv("BC_CLIENT_SECRET")    
    
    PROTOTIPO = os.getenv('PROTOTIPO', 'True').lower() in ('true', '1', 't')

class DevelopmentConfig(Config):
    """Configuración para entorno de desarrollo."""
    DB_CONFIG = {
        "server": os.getenv("DB_SERVER_DEV"),
        "database": os.getenv("DB_DATABASE_DEV"),
        "username": os.getenv("DB_USERNAME_DEV"),
        "password": os.getenv("DB_PASSWORD_DEV"),
        
        "server2": os.getenv("DB_SERVER_2"),
        "database2": os.getenv("DB_DATABASE_2"),
        "username2": os.getenv("DB_USERNAME_2"),
        "password2": os.getenv("DB_PASSWORD_2"),
        
        "server3": os.getenv("DB_SERVER_3"),
        "database3": os.getenv("DB_DATABASE_3"),
        "username3": os.getenv("DB_USERNAME_3"),
        "password3": os.getenv("DB_PASSWORD_3"),
        
        "server4": os.getenv("DB_SERVER_4"),
        "database4": os.getenv("DB_DATABASE_4"),
        "username4": os.getenv("DB_USERNAME_4"),
        "password4": os.getenv("DB_PASSWORD_4"),
    }
    DEBUG = True
    BC_ENV_NAME = os.getenv("BC_DEV_ENV_NAME")
    BC_COMPANY_NAME = os.getenv("BC_DEV_COMPANY_NAME")    

class ProductionConfig(Config):
    """Configuración para entorno de producción."""
    DB_CONFIG = {
        "server": os.getenv("DB_SERVER_PROD"),
        "database": os.getenv("DB_DATABASE_PROD"),
        "username": os.getenv("DB_USERNAME_PROD"),
        "password": os.getenv("DB_PASSWORD_PROD"),
        
        "server2": os.getenv("DB_SERVER_2"),
        "database2": os.getenv("DB_DATABASE_2"),
        "username2": os.getenv("DB_USERNAME_2"),
        "password2": os.getenv("DB_PASSWORD_2"),
        
        "server3": os.getenv("DB_SERVER_3"),
        "database3": os.getenv("DB_DATABASE_3"),
        "username3": os.getenv("DB_USERNAME_3"),
        "password3": os.getenv("DB_PASSWORD_3"),
        
        "server4": os.getenv("DB_SERVER_4"),
        "database4": os.getenv("DB_DATABASE_4"),
        "username4": os.getenv("DB_USERNAME_4"),
        "password4": os.getenv("DB_PASSWORD_4"),
    }
    DEBUG = False
    BC_ENV_NAME = os.getenv("BC_PROD_ENV_NAME")
    BC_COMPANY_NAME = os.getenv("BC_PROD_COMPANY_NAME")
    
# Mapeo de entornos a configuraciones
config = {
    'development': DevelopmentConfig,
    'production': ProductionConfig,
    'default': DevelopmentConfig
}

