// PreviewCotizacion.js - VERSIÓN CON DEBUGGING Y MEJOR MANEJO DE ERRORES
const puppeteer = require('puppeteer');
const https = require('https');
const http = require('http');

(async () => {
  const args = process.argv.slice(2);
  
  if (args.length < 3) {
    console.error('❌ Error: Faltan argumentos');
    console.error('Uso: node PreviewCotizacion.js <url> <output_path> <token>');
    process.exit(1);
  }

  const url = args[0];
  const outputPath = args[1];
  const token = args[2];

  console.log('Iniciando generación de PDF...');
  console.log('URL:', url);
  console.log('Output:', outputPath);
  console.log('Token:', token);

  let browser = null;

  try {
    console.log('Buscando Chrome en el sistema...');
    
    const possiblePaths = [
      'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe',
      'C:\\Program Files (x86)\\Google\\Chrome\\Application\\chrome.exe',
      '/Applications/Google Chrome.app/Contents/MacOS/Google Chrome',
      '/usr/bin/google-chrome',
      '/usr/bin/chromium-browser'
    ];

    let chromePath = null;
    const fs = require('fs');
    
    for (const path of possiblePaths) {
      if (fs.existsSync(path)) {
        chromePath = path;
        console.log(`✅ Chrome encontrado en: ${chromePath}`);
        break;
      }
    }

    if (!chromePath) {
      throw new Error('Chrome/Chromium no encontrado');
    }

    console.log('Lanzando navegador...');
    browser = await puppeteer.launch({
      executablePath: chromePath,
      headless: true,
      args: [
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-dev-shm-usage',
        '--disable-accelerated-2d-canvas',
        '--disable-gpu'
      ]
    });

    console.log('✅ Navegador lanzado correctamente');
    const page = await browser.newPage();
    
    console.log('Configurando viewport...');
    await page.setViewport({
      width: 1200,
      height: 1600,
      deviceScaleFactor: 1
    });

    console.log('Obteniendo HTML desde el servidor...');
    
    // Hacer petición HTTP
    let htmlContent;
    try {
      htmlContent = await new Promise((resolve, reject) => {
        const postData = JSON.stringify({ token: token });
        
        const options = {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Content-Length': Buffer.byteLength(postData),
            'X-Auth-Token': token
          }
        };

        const client = url.startsWith('https') ? https : http;
        
        const req = client.request(url, options, (res) => {
          let data = '';
          
          console.log(`   HTTP Status: ${res.statusCode}`);
          console.log(`   Content-Type: ${res.headers['content-type']}`);
          
          res.on('data', (chunk) => {
            data += chunk;
          });
          
          res.on('end', () => {
            if (res.statusCode >= 200 && res.statusCode < 300) {
              resolve(data);
            } else {
              reject(new Error(`HTTP ${res.statusCode}: ${data.substring(0, 500)}`));
            }
          });
        });
        
        req.on('error', (error) => {
          reject(error);
        });
        
        req.setTimeout(30000, () => {
          req.destroy();
          reject(new Error('Timeout: La petición tardó más de 30 segundos'));
        });
        
        req.write(postData);
        req.end();
      });
      
      console.log('✅ Respuesta HTTP recibida');
      console.log('Tamaño de la respuesta:', htmlContent.length, 'caracteres');
      
      // ✅ DEBUGGING: Ver inicio del contenido
      console.log('=== PRIMEROS 500 CARACTERES ===');
      console.log(htmlContent.substring(0, 500));
      console.log('=== FIN DE MUESTRA ===');
      
      // Si es muy pequeño, mostrar todo
      if (htmlContent.length < 2000) {
        console.log('⚠️ Contenido sospechosamente pequeño (<2000 caracteres)');
        console.log('=== CONTENIDO COMPLETO ===');
        console.log(htmlContent);
        console.log('=== FIN CONTENIDO ===');
      }
      
      // Verificar que sea HTML válido
      if (!htmlContent.includes('<html') && !htmlContent.includes('<!DOCTYPE')) {
        console.log('⚠️ La respuesta NO parece HTML válido');
        
        // Intentar parsear como JSON (por si es un error del servidor)
        try {
          const jsonResponse = JSON.parse(htmlContent);
          console.log('   Es JSON:', jsonResponse);
          throw new Error(`El servidor retornó JSON en lugar de HTML: ${JSON.stringify(jsonResponse)}`);
        } catch (jsonError) {
          // No es JSON válido tampoco
          throw new Error('La respuesta no es HTML válido ni JSON: ' + htmlContent.substring(0, 200));
        }
      }
      
      console.log('✅ HTML válido confirmado');
      
    } catch (httpError) {
      console.error('❌ Error en petición HTTP:', httpError.message);
      
      // HTML de error para el PDF
      htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="UTF-8">
          <title>Error</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              padding: 40px;
              background-color: #fee;
            }
            .error {
              background-color: white;
              border: 2px solid #f00;
              padding: 20px;
              border-radius: 8px;
            }
          </style>
        </head>
        <body>
          <div class="error">
            <h1>❌ Error HTTP</h1>
            <p><strong>Error:</strong> ${httpError.message}</p>
            <p><strong>URL:</strong> ${url}</p>
            <p><strong>Token:</strong> ${token}</p>
          </div>
        </body>
        </html>
      `;
    }

    console.log('Configurando contenido en la página...');
    
    // ✅ Intentar cargar con diferentes estrategias
    try {
      console.log('   Intento 1: domcontentloaded (30s)');
      await page.setContent(htmlContent, {
        waitUntil: 'domcontentloaded',
        timeout: 30000
      });
      console.log('   ✅ Cargado exitosamente con domcontentloaded');
    } catch (timeout1) {
      console.log('   ⚠️ Timeout, intentando con load (15s)');
      try {
        await page.setContent(htmlContent, {
          waitUntil: 'load',
          timeout: 15000
        });
        console.log('   ✅ Cargado exitosamente con load');
      } catch (timeout2) {
        console.log('   ⚠️ Timeout, inyectando directamente');
        await page.evaluate((html) => {
          document.open();
          document.write(html);
          document.close();
        }, htmlContent);
        console.log('   ✅ Inyectado directamente');
      }
    }

    console.log('Página cargada, esperando renderizado...');
    await new Promise(resolve => setTimeout(resolve, 2000));

    console.log('Generando PDF...');
    
    await page.pdf({
      path: outputPath,
      format: 'A4',
      printBackground: true,
      margin: {
        top: '20mm',
        right: '10mm',
        bottom: '20mm',
        left: '10mm'
      },
      preferCSSPageSize: false
    });

    console.log(`✅ PDF generado correctamente en: ${outputPath}`);
    
    await browser.close();
    process.exit(0);

  } catch (error) {
    console.error('❌ Error general:', error.message);
    console.error('Stack:', error.stack);
    
    if (browser) {
      try {
        await browser.close();
      } catch (closeError) {
        console.error('Error al cerrar navegador:', closeError.message);
      }
    }
    
    process.exit(1);
  }
})();