/**
 * Archivo: CotizFormQueryIng.js
 * Ruta: src\static\js\Operaciones\Ingenieria\Cotiz\CotizFormQueryIng.js
 * Descripción: Funciones JavaScript completas y optimizadas para consultar formularios de cotización en el módulo de Ingeniería.
 */

// ===============================
// ?? CONSTANTES Y VARIABLES
// ===============================

// Indicador de carga principal (el spinner pequeno)
const loadingIndicator = document.getElementById("loading-indicator");
// Variable para almacenar la instancia de DataTable
let table;

// ===============================
// ?? FUNCIONES GENERICAS DE CARGA
// ===============================

/**
 * Muestra el indicador de carga tipo spinner.
 */
function showLoading() {
  if (loadingIndicator) loadingIndicator.style.display = "block";
}

/**
 * Oculta el indicador de carga tipo spinner.
 */
function hideLoading() {
  if (loadingIndicator) loadingIndicator.style.display = "none";
}

/**
 * Muestra el overlay global (pantalla completa) con un mensaje.
 * @param {string} [message="Procesando solicitud..."] - Mensaje a mostrar.
 */
function showGlobalLoading(message = "Procesando solicitud...") {
  const overlay = document.getElementById("global-loading-overlay");
  const textElement = document.querySelector(
    "#global-loading-overlay .loading-text"
  );

  if (textElement) {
    textElement.textContent = message;
  }
  if (overlay) {
    overlay.style.display = "flex";
  }
  // Bloquear el scroll del body
  document.body.style.overflow = "hidden";
}

/**
 * Oculta el overlay global (pantalla completa).
 */
function hideGlobalLoading() {
  const overlay = document.getElementById("global-loading-overlay");
  if (overlay) {
    overlay.style.display = "none";
  }
  // Restaurar el scroll del body
  document.body.style.overflow = "auto";
}

// ===============================
// ?? FUNCIONES ESPECIFICAS DEL MODULO
// ===============================

/**
 * Inicializa la tabla DataTable con todas las configuraciones.
 */
function initializeDataTable() {
    table = $("#TaskCotizFormQueryIng").DataTable({
        responsive: true,
        scrollX: true,
        dom: "Bfrtip",

        buttons: [
            {
                extend: "excel",
                text: "Exportar a Excel",
                className: "btn btn-success",
                exportOptions: {
                    columns: ":visible",
                    modifier: { page: "all" },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $(
                                "#TaskCotizFormQueryIng_wrapper .dataTables_scrollHead thead tr:nth-child(1) th"
                            );
                            if (headerSelector.length === 0) {
                                headerSelector = $("#TaskCotizFormQueryIng thead th");
                            }
                            return headerSelector.eq(columnIdx).text();
                        },
                    },
                },
            },
            {
                extend: "pdf",
                text: "Exportar a PDF",
                className: "btn btn-danger",
                exportOptions: {
                    columns: ":visible",
                    modifier: { page: "all" },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $(
                                "#TaskCotizFormQueryIng_wrapper .dataTables_scrollHead thead tr:nth-child(1) th"
                            );
                            if (headerSelector.length === 0) {
                                headerSelector = $("#TaskCotizFormQueryIng thead th");
                            }
                            return headerSelector.eq(columnIdx).text();
                        },
                    },
                },
            },
        ],

        pageLength: 10,
        ordering: false,
        searching: true,
        lengthMenu: [
            [10, 25, 50, -1],
            [10, 25, 50, "Todos"],
        ],

        columnDefs: [
            { targets: -1, orderable: false, searchable: false },
        ],

        order: [[0, "asc"]],

        // ✅ SOLUCIÓN CORRECTA: Buscar inputs en el header con scrollX
        initComplete: function () {
            let api = this.api();

            api.columns().every(function (columnIndex) {
                let column = this;
                let input;

                // ✅ SOLUCIÓN: Buscar en el header clonado de DataTables
                if ($('.dataTables_scrollHead').length > 0) {
                    // Cuando scrollX está activo, DataTables clona el header
                    input = $('.dataTables_scrollHead thead tr:nth-child(2) th')
                        .eq(columnIndex)
                        .find('input');
                } else {
                    // Fallback si no hay scroll
                    input = $('#TaskCotizFormQueryIng thead tr:nth-child(2) th')
                        .eq(columnIndex)
                        .find('input');
                }

                if (input.length > 0) {
                    console.log(`✅ Input encontrado en columna ${columnIndex}`);

                    // Restaurar valor de búsqueda si existe
                    let searchValue = column.search();
                    if (searchValue) {
                        input.val(searchValue);
                    }

                    // ✅ Evitar que el clic en el input ordene la columna
                    input.on("click", function (e) {
                        e.stopPropagation();
                        e.preventDefault();
                    });

                    // ✅ Filtrar con múltiples eventos
                    input.on("keyup change paste input", function () {
                        let value = this.value;

                        if (column.search() !== value) {
                            console.log(`🔍 Filtrando columna ${columnIndex}: "${value}"`);
                            column.search(value).draw();
                        }
                    });

                    // ✅ Limpiar con ESC
                    input.on("keydown", function (e) {
                        if (e.key === "Escape" || e.keyCode === 27) {
                            this.value = "";
                            column.search("").draw();
                            console.log(`🗑️ Filtro limpiado en columna ${columnIndex}`);
                        }
                    });
                } else {
                    console.warn(`⚠️ No se encontró input en columna ${columnIndex}`);
                }
            });

            console.log("✅ DataTable inicializada con filtros activos");
        },

        drawCallback: function () {
            console.log("📊 Tabla redibujada");
        },
    });
}

/**
 * Carga los datos de las tareas desde el backend.
 */
function loadTasks() {
  showLoading();

  // Esta es la ruta Flask que crearemos en el archivo Python
  fetch("/Operaciones_Module/Ingenieria/Cotiz/list")
    .then((response) => {
      if (!response.ok) {
        throw new Error(
          "Error en la respuesta del servidor: " + response.status
        );
      }
      return response.json();
    })
    .then((data) => {
      // Apuntamos al ID del 'tbody' de tu tabla
      const tableBody = document.getElementById("TaskCotizFormQueryIngbody");
      tableBody.innerHTML = ""; // Limpiamos la tabla

      // Llenamos la tabla con los datos
      data.forEach((task) => {
        console.log("NroFormulario:", task.NroFormulario);
        console.log(task);

        const row = document.createElement("tr");

        // Creamos las 27 celdas (<td>) para cada fila
        // Las claves (ej. task.NroTarea) deben coincidir con los ALIAS del SQL
        row.innerHTML = `
                    <td>${task.NroTarea || ""}</td>
                    <td>${task.Estatus || ""}</td>
                    <td>${task.Oportunidad || ""}</td>
                    <td>${task.TipoFormulario || ""}</td>
                    <td>${task.FechaCreacion || ""}</td>
                    <td>${task.Contacto || ""}</td>
                    <td>${task.Vendedor || ""}</td>
                    <td>${task.CorreoVendedor || ""}</td>
                    <td>${task.TelefonoVendedor || ""}</td>
                    <td>${task.NroCosteo || ""}</td>
                    <td>${task.CasoCosteo || ""}</td>
                    <td>${task.CostoDirecto || ""}</td>
                    <td>${task.PctIndirecto || ""}</td>
                    <td>${task.MontoIndirecto || ""}</td>
                    <td>${task.PctFinanciamiento || ""}</td>
                    <td>${task.MontoFinanciamiento || ""}</td>
                    <td>${task.PctUtilidad || ""}</td>
                    <td>${task.MontoUtilidad || ""}</td>
                    <td>${task.PctGtosOp || ""}</td>
                    <td>${task.MontoGtosOp || ""}</td>
                    <td>${task.FactorSobrecosto || ""}</td>
                    <td>${task.PrecioVentaMin || ""}</td>
                    <td>${task.PctDescMax || ""}</td>
                    <td>${task.MontoDescuento || ""}</td>
                    <td>${task.PrecioLista || ""}</td>
                    <td>${task.TiempoEjecucion || ""}</td>
                    <td>
                        <button class="btn-consultar" onclick="consultarTarea('${
                          task.NroFormulario
                        }')" title="Consultar Tarea">
                            Consultar
                        </button>
                    </td>
                `;
        tableBody.appendChild(row);
      });

      // Destruimos la instancia anterior de DataTable (si existe) y la reinicializamos
      if ($.fn.DataTable.isDataTable("#TaskCotizFormQueryIng")) {
        table.destroy();
      }
      initializeDataTable(); // Inicializamos la tabla con los nuevos datos
    })
    .catch((error) => {
      console.error("Error al cargar las tareas:", error);
      alert("Hubo un problema al cargar las tareas. Intenta nuevamente.");
    })
    .finally(() => {
      hideLoading(); // Ocultamos el spinner
    });
}

/**
 * Consulta una tarea y genera el PDF con adjuntos.
 * Esta función se ejecuta cuando el usuario hace clic en "Consultar".
 * 
 * @param {string} nroFormulario - El FormID de la tarea (ej: '1002-1-FormGral')
 */
function consultarTarea(nroFormulario) {
    console.log("🔍 Consultando FormID:", nroFormulario);
    
    // ========================================
    // PASO 1: VALIDACIÓN DE PARÁMETROS
    // ========================================
    
    if (!nroFormulario) {
        alert("❌ Error: No se proporcionó un FormID válido");
        return;
    }
    
    // ========================================
    // PASO 2: MOSTRAR FEEDBACK VISUAL
    // ========================================
    
    showGlobalLoading(`Generando PDF para FormID: ${nroFormulario}...`);
    
    // ========================================
    // PASO 3: CONSTRUIR LA URL DEL ENDPOINT
    // ========================================
    
    const params = new URLSearchParams({
        form_id: nroFormulario,
        adjuntos: 'true',
        zip: 'true'
    });
    
    const url = `/Operaciones_Module/Ingenieria/Cotiz/generar-pdf?${params.toString()}`;
    
    console.log("📡 URL del endpoint:", url);
    
    // ========================================
    // PASO 4: HACER REQUEST AL BACKEND
    // ========================================
    
    fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/pdf, application/zip'
        }
    })
    .then(response => {
        console.log("📥 Respuesta recibida. Status:", response.status);
        
        if (response.status === 404) {
            throw new Error(`FormID no encontrado: ${nroFormulario}`);
        }
        
        if (response.status === 400) {
            throw new Error('Parámetros inválidos en la solicitud');
        }
        
        if (response.status === 500) {
            throw new Error('Error interno del servidor al generar PDF');
        }
        
        if (!response.ok) {
            throw new Error(`Error HTTP ${response.status}: ${response.statusText}`);
        }
        
        // Extraer nombre del archivo del header
        const contentDisposition = response.headers.get('Content-Disposition');
        let filename = `formulario_${nroFormulario.replace(/-/g, '_')}.pdf`;
        
        if (contentDisposition) {
            const filenameMatch = contentDisposition.match(/filename="?(.+)"?/);
            if (filenameMatch && filenameMatch[1]) {
                filename = filenameMatch[1];
            }
        }
        
        console.log("📄 Nombre del archivo:", filename);
        
        return response.blob().then(blob => ({ blob, filename }));
    })
    .then(({ blob, filename }) => {
        console.log("💾 Blob recibido. Tamaño:", (blob.size / 1024).toFixed(2), "KB");
        
        // Crear URL temporal y descargar
        const blobUrl = window.URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = blobUrl;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(blobUrl);
        
        console.log("✅ Descarga iniciada exitosamente");
        
        setTimeout(() => {
            hideGlobalLoading();
            alert(`✅ PDF generado exitosamente\n\n📄 Archivo: ${filename}\n📁 Revisa tu carpeta de Descargas\n\nEl archivo se descargó automáticamente.`);
        }, 500);
    })
    .catch(error => {
        console.error("❌ Error al generar PDF:", error);
        hideGlobalLoading();
        alert(`❌ Error al generar el PDF\n\nFormID: ${nroFormulario}\nError: ${error.message}\n\nDetalles:\n- Verifica que el FormID sea correcto\n- Revisa la consola del navegador (F12) para más información\n- Contacta al administrador si el problema persiste`);
    });
}

// ===============================
// ?? EVENT LISTENERS (INICIO)
// ===============================

/**
 * Se ejecuta cuando todo el HTML ha sido cargado.
 * Es el punto de entrada principal.
 */
document.addEventListener("DOMContentLoaded", () => {
  // Carga los datos de las tareas en la tabla
  loadTasks();
});
