/*
  UserPassword.js
  Ruta: src\static\js\Security\UserPassword.js
*/

function agregarEstilosOverlay() {
    // Verificar si ya existe el estilo
    if (!document.getElementById('overlay-styles')) {
        const style = document.createElement('style');
        style.id = 'overlay-styles';
        style.innerHTML = `
            .page-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
                flex-direction: column;
            }
            .spinner-container {
                text-align: center;
                color: white;
            }
            .spinner-container .spinner-border {
                width: 3rem;
                height: 3rem;
            }
            .spinner-text {
                margin-top: 15px;
                font-size: 18px;
                font-weight: bold;
            }
        `;
        document.head.appendChild(style);
    }
}

function mostrarSpinner(mensaje = "Actualizando fichas técnicas...") {
    agregarEstilosOverlay();
    
    // Crear el overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-overlay';
    overlay.id = 'actualizacion-overlay';
    
    // Crear el spinner y el mensaje
    overlay.innerHTML = `
        <div class="spinner-container">
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <div class="spinner-text">${mensaje}</div>
        </div>
    `;
    
    // Agregar a la página
    document.body.appendChild(overlay);
    
    // Bloquear el scroll
    document.body.style.overflow = 'hidden';
}

function ocultarSpinner() {
    const overlay = document.getElementById('actualizacion-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = 'auto';
    }
}
async function generarToken() {
    const userID = document.getElementById("UserID").value;
    const email = document.getElementById("Email").value;
    const nombre = document.getElementById("Nombre").value;

    try {
        // Codificar los parámetros para la URL
         // Mostrar el spinner antes de iniciar la petición
        mostrarSpinner();
        const encodedEmail = encodeURIComponent(email);
        const encodedNombre = encodeURIComponent(nombre);
        const encodedUserID = encodeURIComponent(userID);
        
        console.log('Haciendo fetch a:', `/Tokensign/${encodedUserID}/${encodedEmail}/${encodedNombre}`);
        
        // Realizar la petición con manejo explícito de la respuesta
        const response = await fetch(`/Tokensign/${encodedUserID}/${encodedEmail}/${encodedNombre}`);
        
        // Verificar si la respuesta es exitosa
        if (!response.ok) {
            throw new Error(`Error HTTP: ${response.status}`);
        }
        
        // Obtener el texto de la respuesta primero para debugging
        const responseText = await response.text();
        console.log('Respuesta raw:', responseText);
        
        // Intentar parsear el texto como JSON
        let data;
        try {
            data = JSON.parse(responseText);
        } catch (parseError) {
            console.error("Error al parsear JSON:", parseError);
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'No se pudo procesar la respuesta del servidor'
            });
            return;
        }

        // Mostrar el resultado con SweetAlert2
        Swal.fire({
            icon: 'success',
            title: 'Token generado',
            html: `<p>Token generado: ${data.token}</p>
                   <p>Token ID: ${data.token_id || 'No disponible'}</p>`
        });
        
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
    } catch (error) {
        console.error("Error al generar el token:", error);
        // Reemplazado por SweetAlert2
        Swal.fire({
            icon: 'error',
            title: 'Error',
            text: `Error interno del cliente: ${error.message}`
        });
        ocultarSpinner();
    }
}

// Función para manejar la recuperación de contraseña
function setupForgotPasswordForm() {
    const forgotPasswordForm = document.getElementById('forgotPasswordForm');
    
    // Si el formulario no existe en esta página, salimos
    if (!forgotPasswordForm) return;
    
    const emailField = document.getElementById('email');
    const recoveryButton = document.querySelector('.recovery-btn');
    
    // Verificar que todos los elementos existen
    if (!emailField || !recoveryButton) {
        console.error('No se pudieron encontrar todos los elementos del formulario');
        return;
    }
    
    const spinner = recoveryButton.querySelector('.spinner-border');
    
    // Función básica para validar email
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function shakeForm() {
        forgotPasswordForm.classList.add('shake');
        setTimeout(() => forgotPasswordForm.classList.remove('shake'), 500);
    }
    
    function startLoading() {
        recoveryButton.disabled = true;
        if (spinner) spinner.classList.remove('d-none');
        const buttonText = recoveryButton.textContent.trim();
        recoveryButton.setAttribute('data-original-text', buttonText);
        recoveryButton.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Enviando...`;
    }
    
    function stopLoading() {
        recoveryButton.disabled = false;
        const originalText = recoveryButton.getAttribute('data-original-text') || 'Enviar Enlace de Recuperación';
        recoveryButton.innerHTML = originalText;
    }
    
    // Función para manejar el envío del formulario
    function handleFormSubmit(e) {
        e.preventDefault();
        console.log('Formulario de recuperación enviado');
        
        // Validar campo email
        const email = emailField.value.trim();
        
        if (!email) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu correo electrónico.'
            });
            emailField.focus();
            return;
        }
        
        if (!isValidEmail(email)) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Email inválido',
                text: 'Por favor, ingresa un correo electrónico válido.'
            });
            emailField.focus();
            return;
        }
        
        // Mostrar spinner durante la carga
        startLoading();
        
        // Enviar solicitud al servidor
        fetch('/auth/forgot-password', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ email: email })
        })
        .then(response => {
            console.log('Respuesta recibida:', response);
            return response.json();
        })
        .then(data => {
            console.log('Datos recibidos:', data);
            stopLoading();
            
            if (data.success) {
                // Mostrar mensaje de éxito con SweetAlert2
                Swal.fire({
                    icon: 'success',
                    title: '¡Éxito!',
                    text: data.message || 'Se ha enviado un enlace de recuperación a tu correo electrónico.',
                    timer: 3000,
                    timerProgressBar: true
                });
                
                // Limpiar el campo de email
                emailField.value = '';
                
                // Deshabilitar el formulario temporalmente
                emailField.disabled = true;
                recoveryButton.disabled = true;
                
                // Habilitar después de 30 segundos para evitar spam
                setTimeout(() => {
                    emailField.disabled = false;
                    recoveryButton.disabled = false;
                }, 30000);
            } else {
                // Mostrar mensaje de error con SweetAlert2
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: data.message || 'Error al procesar la solicitud.'
                });
                shakeForm();
            }
        })
        .catch(error => {
            console.error('Error en la petición:', error);
            stopLoading();
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Error de conexión',
                text: 'Error de conexión. Por favor, inténtalo más tarde.'
            });
        });
    }
    
    // Asignar el manejador de eventos
    forgotPasswordForm.addEventListener('submit', handleFormSubmit);
    
    // También podemos agregar el manejador directamente al botón como alternativa
    recoveryButton.addEventListener('click', function(e) {
        // Si el formulario ya está manejando el submit, esto es solo una capa adicional
        // para asegurarnos de que se active
        if (e.target.type !== 'submit') {
            e.preventDefault();
            forgotPasswordForm.dispatchEvent(new Event('submit'));
        }
    });
    
    console.log('Formulario de recuperación de contraseña configurado correctamente');
}

// Función para configurar el formulario de login
function setupLoginForm() {
    const loginForm = document.getElementById('loginForm');
    if (!loginForm) return;
    
    const emailField = document.getElementById('email');
    const passwordField = document.getElementById('password');
    const loginButton = document.querySelector('.login-btn');
    
    if (!emailField || !passwordField || !loginButton) {
        console.error('No se pudieron encontrar todos los elementos del formulario de login');
        return;
    }
    
    const spinner = loginButton.querySelector('.spinner-border');
    
    // Manejador del formulario
    loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Validar campos
        const email = emailField.value.trim();
        const password = passwordField.value;
        
        if (!email) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu correo electrónico.'
            });
            emailField.focus();
            return;
        }
        
        if (!isValidEmail(email)) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Email inválido',
                text: 'Por favor, ingresa un correo electrónico válido.'
            });
            emailField.focus();
            return;
        }
        
        if (!password) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu contraseña.'
            });
            passwordField.focus();
            return;
        }
        
        // Mostrar spinner durante la carga
        startLoading();
        
        // Recolectar datos del formulario
        const formData = {
            email: email,
            password: password,
            remember: document.getElementById('rememberMe').checked
        };

        // Enviar datos al servidor
        fetch('/auth/login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(formData)
        })
        .then(response => response.json())
        .then(data => {
            stopLoading();
            
            if (data.success) {
                // Redireccionar al dashboard o página principal
                window.location.href = data.redirect || '/index';
            } else {
                // Mostrar mensaje de error con SweetAlert2
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: data.message || 'Credenciales incorrectas. Por favor, verifica tu correo y contraseña.'
                });
                shakeForm();
            }
        })
        .catch(error => {
            stopLoading();
            console.error('Error:', error);
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Error de conexión',
                text: 'Error de conexión. Por favor, inténtalo más tarde.'
            });
        });
    });
    
    // Funciones auxiliares
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function shakeForm() {
        loginForm.classList.add('shake');
        setTimeout(() => loginForm.classList.remove('shake'), 500);
    }
    
    function startLoading() {
        loginButton.disabled = true;
        if (spinner) spinner.classList.remove('d-none');
        const buttonText = loginButton.textContent.trim();
        loginButton.setAttribute('data-original-text', buttonText);
        loginButton.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Cargando...`;
    }
    
    function stopLoading() {
        loginButton.disabled = false;
        const originalText = loginButton.getAttribute('data-original-text') || 'Iniciar sesión';
        loginButton.innerHTML = originalText;
    }
}

// Función para manejar el toggle de visibilidad de la contraseña
function setupPasswordToggles() {
    const toggleButtons = document.querySelectorAll('.toggle-password');
    
    toggleButtons.forEach(button => {
        button.addEventListener('click', function() {
            const input = this.previousElementSibling;
            const icon = this.querySelector('i');
            
            if (input && icon) {
                if (input.type === 'password') {
                    input.type = 'text';
                    icon.classList.remove('fa-eye');
                    icon.classList.add('fa-eye-slash');
                } else {
                    input.type = 'password';
                    icon.classList.remove('fa-eye-slash');
                    icon.classList.add('fa-eye');
                }
            }
        });
    });
}

// Función para configurar el formulario de confirmación de contraseña
function setupPasswordConfirmForm() {
    const passwordForm = document.getElementById('passwordConfirmForm');
    if (!passwordForm) return;
    
    const passwordField = document.getElementById('password');
    const confirmPasswordField = document.getElementById('confirmPassword');
    
    if (!passwordField || !confirmPasswordField) {
        console.error('No se pudieron encontrar todos los elementos del formulario de confirmación de contraseña');
        return;
    }
    
    // Función para validar la contraseña
    function validatePassword(password) {
        const requirements = {
            length: password.length >= 8,
            uppercase: /[A-Z]/.test(password),
            lowercase: /[a-z]/.test(password),
            number: /[0-9]/.test(password),
            special: /[^A-Za-z0-9]/.test(password)
        };

        // Actualizar UI para cada requisito
        for (const [requirement, isValid] of Object.entries(requirements)) {
            const reqElement = document.getElementById(`req-${requirement}`);
            if (reqElement) {
                if (isValid) {
                    reqElement.classList.add('valid');
                } else {
                    reqElement.classList.remove('valid');
                }
            }
        }

        // Devolver si todos los requisitos se cumplen
        return Object.values(requirements).every(Boolean);
    }
    
    // Validar coincidencia de contraseñas
    function validatePasswordMatch() {
        const isMatch = passwordField.value === confirmPasswordField.value;
        if (isMatch || confirmPasswordField.value === '') {
            confirmPasswordField.classList.remove('is-invalid');
        } else {
            confirmPasswordField.classList.add('is-invalid');
        }
        return isMatch;
    }
    
    // Evento de validación en tiempo real para el campo de contraseña
    if (passwordField) {
        passwordField.addEventListener('input', function() {
            validatePassword(this.value);
        });
    }
    
    // Evento de validación en tiempo real para confirmar contraseña
    if (confirmPasswordField) {
        confirmPasswordField.addEventListener('input', validatePasswordMatch);
    }
    
    // Manejo del envío del formulario
    passwordForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Validar contraseña
        const isPasswordValid = validatePassword(passwordField.value);
        if (!isPasswordValid) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Contraseña inválida',
                text: 'La contraseña no cumple con todos los requisitos.'
            });
            passwordField.classList.add('shake');
            setTimeout(() => passwordField.classList.remove('shake'), 500);
            return;
        }

        // Validar coincidencia de contraseñas
        if (!validatePasswordMatch()) {
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Las contraseñas no coinciden',
                text: 'Por favor verifica que ambas contraseñas sean idénticas.'
            });
            confirmPasswordField.classList.add('shake');
            setTimeout(() => confirmPasswordField.classList.remove('shake'), 500);
            return;
        }

        // Recolectar datos del formulario
        const formData = {
            token: document.getElementById('token').value,
            userId: document.getElementById('userId').value,
            password: passwordField.value,
            confirmPassword: confirmPasswordField.value
        };
        
        mostrarSpinner("Estableciendo contraseña...");
        
        // Enviar datos al servidor
        fetch('/set_password', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(formData)
        })
        .then(response => response.json())
        .then(data => {
            ocultarSpinner();
            
            if (data.success) {
                // Mostrar mensaje de éxito con SweetAlert2
                Swal.fire({
                    icon: 'success',
                    title: '¡Éxito!',
                    text: data.message || 'Contraseña establecida correctamente.',
                    timer: 3000,
                    timerProgressBar: true
                });
                
                // Deshabilitar el formulario
                Array.from(passwordForm.elements).forEach(element => {
                    element.disabled = true;
                });
                
                // Redirigir después de 3 segundos a la página de login
                setTimeout(() => {
                    window.location.href = '/login';
                }, 3000);
            } else {
                // Mostrar mensaje de error con SweetAlert2
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: data.message || 'Error al establecer la contraseña.'
                });
            }
        })
        .catch(error => {
            ocultarSpinner();
            console.error('Error:', error);
            // Reemplazado por SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Error de conexión',
                text: 'Error de conexión. Por favor, inténtalo más tarde.'
            });
        });
    });
}

// Función principal de inicialización
function initializeUserPasswordFunctions() {
    console.log('Inicializando funciones de UserPassword.js');
    
    // Configurar los toggles de contraseña
    setupPasswordToggles();
    
    // Configurar el formulario de recuperación de contraseña
    setupForgotPasswordForm();
    
    // Configurar el formulario de login
    setupLoginForm();
    
    // Configurar el formulario de confirmación de contraseña
    setupPasswordConfirmForm();
}

// Ejecutar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM completamente cargado');
    initializeUserPasswordFunctions();
});

// Exponemos funciones globales para llamarlas directamente desde el HTML si es necesario
window.sendPasswordRecoveryEmail = function() {
    const forgotPasswordForm = document.getElementById('forgotPasswordForm');
    if (forgotPasswordForm) {
        forgotPasswordForm.dispatchEvent(new Event('submit'));
    } else {
        console.error('Formulario de recuperación no encontrado');
        // Reemplazado por SweetAlert2
        Swal.fire({
            icon: 'error',
            title: 'Error',
            text: 'Formulario de recuperación no encontrado'
        });
    }
};