/*
  login.js
  Ruta: src\static\js\login.js
*/

// Ejecutar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    // Configurar formulario de inicio de sesión
    setupLoginForm();
    
    // Configurar toggle de contraseña
    setupPasswordToggles();
    
    // Configurar formulario de recuperación de contraseña
    setupForgotPasswordForm();
});

// Función para guardar las credenciales en localStorage
function saveCredentials(email, password, rememberMe) {
    if (rememberMe) {
        // Usar btoa para codificación básica (Base64)
        localStorage.setItem('rememberedEmail', email);
        localStorage.setItem('rememberedPassword', btoa(password)); // Codificación básica
    } else {
        // Si no se marca "Recuérdame", eliminar cualquier dato guardado
        localStorage.removeItem('rememberedEmail');
        localStorage.removeItem('rememberedPassword');
    }
}

// Función para cargar las credenciales guardadas
function loadSavedCredentials() {
    const email = localStorage.getItem('rememberedEmail');
    const encodedPassword = localStorage.getItem('rememberedPassword');

    if (email) {
        document.getElementById('email').value = email;
        document.getElementById('rememberMe').checked = true;
        
    }
    if (encodedPassword) {
        try {
            // Decodificar la contraseña (atob para descodificar Base64)
            const password = atob(encodedPassword);
            document.getElementById('password').value = password;
        } catch (e) {
            console.error('Error al decodificar la contraseña guardada:', e);
        }
    }
}
// Función para configurar el formulario de login
function setupLoginForm() {
    const loginForm = document.getElementById('loginForm');
    if (!loginForm) return;
    // Cargar credenciales guardadas si existen
    loadSavedCredentials();
    const emailField = document.getElementById('email');
    const passwordField = document.getElementById('password');
    const loginButton = document.querySelector('.login-btn');
    
    if (!emailField || !passwordField || !loginButton) {
        console.error('No se pudieron encontrar todos los elementos del formulario de login');
        return;
    }
    
    const spinner = loginButton.querySelector('.spinner-border');
    
    // Manejador del formulario
    loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Validar campos
        const email = emailField.value.trim();
        const password = passwordField.value;
        const rememberMe = document.getElementById('rememberMe').checked;
         // Guardar credenciales si "Recuérdame" está marcado
        saveCredentials(email, rememberMe);
        if (!email) {
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu correo electrónico.'
            });
            emailField.focus();
            return;
        }
        
        if (!isValidEmail(email)) {
            Swal.fire({
                icon: 'error',
                title: 'Email inválido',
                text: 'Por favor, ingresa un correo electrónico válido.'
            });
            emailField.focus();
            return;
        }
        
        if (!password) {
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu contraseña.'
            });
            passwordField.focus();
            return;
        }
        // Guardar credenciales si "Recuérdame" está marcado
        saveCredentials(email, password, rememberMe);
        // Mostrar spinner durante la carga
        startLoading();
        
        // Recolectar datos del formulario
        const formData = {
            email: email,
            password: password,
            remember: document.getElementById('rememberMe').checked
        };

        // Enviar datos al servidor
        fetch('/auth/login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(formData)
        })
        

        
        .then(response => {

            // Verifica si la respuesta es exitosa (código 200-299)
            if (!response.ok) {
                throw new Error('Error en la solicitud: ' + response.statusText);
            }
           
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            console.log('Respuesta del servidor:');
            return response.json(); // Parsea la respuesta como JSON
        })
        


        //.then(response => response.json())
        .then(data => {
            stopLoading();
            
            if (data.success) {
                // Redireccionar al dashboard o página principal
                window.location.href = data.redirect || '/index';
            } else {
                // Mostrar mensaje de error
                Swal.fire({
                    icon: 'error',
                    title: 'Error de inicio de sesión',
                    text: data.message || 'Credenciales incorrectas. Por favor, verifica tu correo y contraseña.'
                });
                shakeForm();
            }
        })
        .catch(error => {
            stopLoading();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error de inicio de sesión',
                text: 'Credenciales incorrectas. Por favor, verifica tu correo y contraseña.'
            });
        });
        
    });
    
    // Funciones auxiliares
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function shakeForm() {
        loginForm.classList.add('shake');
        setTimeout(() => loginForm.classList.remove('shake'), 500);
    }
    
    function startLoading() {
        loginButton.disabled = true;
        if (spinner) spinner.classList.remove('d-none');
        const buttonText = loginButton.textContent.trim();
        loginButton.setAttribute('data-original-text', buttonText);
        loginButton.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Cargando...`;
    }
    
    function stopLoading() {
        loginButton.disabled = false;
        const originalText = loginButton.getAttribute('data-original-text') || 'Iniciar sesión';
        loginButton.innerHTML = originalText;
    }
}

// Función para manejar el toggle de visibilidad de la contraseña
function setupPasswordToggles() {
    const toggleButtons = document.querySelectorAll('.toggle-password');
    
    toggleButtons.forEach(button => {
        button.addEventListener('click', function() {
            const input = this.previousElementSibling;
            const icon = this.querySelector('i');
            
            if (input && icon) {
                if (input.type === 'password') {
                    input.type = 'text';
                    icon.classList.remove('fa-eye');
                    icon.classList.add('fa-eye-slash');
                } else {
                    input.type = 'password';
                    icon.classList.remove('fa-eye-slash');
                    icon.classList.add('fa-eye');
                }
            }
        });
    });
}

// Función para configurar el formulario de recuperación de contraseña
function setupForgotPasswordForm() {
    const forgotPasswordForm = document.getElementById('forgotPasswordForm');
    if (!forgotPasswordForm) return;
    
    const emailField = document.getElementById('recovery-email');
    const recoveryButton = forgotPasswordForm.querySelector('.recovery-btn');
    
    if (!emailField || !recoveryButton) {
        console.error('No se pudieron encontrar todos los elementos del formulario de recuperación');
        return;
    }
    
    const spinner = recoveryButton.querySelector('.spinner-border');
    
    // Resetear el formulario cuando se abre el modal
    document.getElementById('forgotPasswordModal').addEventListener('show.bs.modal', function() {
        // Limpiar campos
        if (emailField) emailField.value = '';
    });
    
    // Manejador del formulario
    forgotPasswordForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Validar email
        const email = emailField.value.trim();
        
        if (!email) {
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'Por favor, ingresa tu correo electrónico.'
            });
            emailField.focus();
            return;
        }
        
        if (!isValidEmail(email)) {
            Swal.fire({
                icon: 'error',
                title: 'Email inválido',
                text: 'Por favor, ingresa un correo electrónico válido.'
            });
            emailField.focus();
            return;
        }
        
        // Mostrar spinner durante la carga
        startLoading();
        
        // Enviar solicitud al servidor
        fetch('/auth/forgot-password', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ email: email })
        })
        .then(response => response.json())
        .then(data => {
            stopLoading();
            
            if (data.success) {
                // Mostrar mensaje de éxito
                Swal.fire({
                    icon: 'success',
                    title: '¡Éxito!',
                    text: data.message || 'Se ha enviado un enlace de recuperación a tu correo electrónico.',
                    timer: 3000,
                    timerProgressBar: true
                });
                
                // Limpiar el campo de email
                emailField.value = '';
                
                // Deshabilitar el formulario temporalmente
                emailField.disabled = true;
                recoveryButton.disabled = true;
                
                // Habilitar después de 30 segundos para evitar spam
                setTimeout(() => {
                    emailField.disabled = false;
                    recoveryButton.disabled = false;
                }, 30000);
                
                // Cerrar el modal después de 3 segundos
                setTimeout(() => {
                    const modal = bootstrap.Modal.getInstance(document.getElementById('forgotPasswordModal'));
                    if (modal) {
                        modal.hide();
                    }
                }, 3000);
            } else {
               // Determinar si el error está relacionado con la generación del token
                let errorTitle = 'Error';
                let errorMessage = data.message || 'Error al procesar la solicitud.';
                
                // Verificar si el mensaje contiene indicaciones de error en la generación del token
                if (errorMessage.includes('token') || 
                    errorMessage.includes('Token') || 
                    errorMessage.includes('correo') || 
                    errorMessage.includes('email')) {
                    errorTitle = 'Error de generación de token';
                }
                
                // Mostrar mensaje de error específico
                Swal.fire({
                    icon: 'error',
                    title: errorTitle,
                    text: errorMessage,
                    confirmButtonText: 'Entendido'
                });
                shakeForm();
            }
        })
        .catch(error => {
            stopLoading();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error de conexión',
                text: 'Error de conexión. Por favor, inténtalo más tarde.'
            });
        });
    });
    
    // Funciones auxiliares
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function shakeForm() {
        forgotPasswordForm.classList.add('shake');
        setTimeout(() => forgotPasswordForm.classList.remove('shake'), 500);
    }
    
    function startLoading() {
        recoveryButton.disabled = true;
        if (spinner) spinner.classList.remove('d-none');
        const buttonText = recoveryButton.textContent.trim();
        recoveryButton.setAttribute('data-original-text', buttonText);
        recoveryButton.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Enviando...`;
    }
    
    function stopLoading() {
        recoveryButton.disabled = false;
        const originalText = recoveryButton.getAttribute('data-original-text') || 'Enviar Enlace de Recuperación';
        recoveryButton.innerHTML = originalText;
    }
}
