// Archivo: DMAccessEmployee.js
// Ruta: src\static\js\SupYCtrol\DataMaster\DMAccessEmployee.js
// Descripción: Sistema de aprobación de acceso para empleados
// Última modificación: Reorganización y mejora de estructura del código

/****************************************************
 * SECCIÓN 1: CONFIGURACIÓN INICIAL Y UTILIDADES
 ****************************************************/
// Constantes globales
const loadingIndicator = document.getElementById("loading-indicator");
let table; // Variable global para almacenar la DataTable
let selectedId = null; // Variable para almacenar el ID del empleado seleccionado
let fechaActual = new Date().toISOString().split('T')[0]; // Formato YYYY-MM-DD
let tituloPagina = document.title.trim(); // Obtener el título de la página

// Funciones de utilidad general

/**
 * Muestra el indicador de carga
 */
function showLoading() {
    if (loadingIndicator) loadingIndicator.style.display = "block";
}

/**
 * Oculta el indicador de carga
 */
function hideLoading() {
    if (loadingIndicator) loadingIndicator.style.display = "none";
}

/**
 * Cierra el modal y resetea sus campos
 */
function closeModal() {
    const modalElement = document.getElementById('EstandarModal');
    const modalInstance = bootstrap.Modal.getInstance(modalElement);
    if (modalInstance) {
        modalInstance.hide();
    }
    resetModal();
}

/**
 * Resetea los campos y estilos del modal cuando se cierra
 */
function resetModal() {
    document.querySelectorAll("#EstandarModal input, #EstandarModal textarea").forEach(el => el.value = "");
    document.querySelectorAll(".invalid-input").forEach(el => el.classList.remove("invalid-input"));
    selectedId = null;
}

/****************************************************
 * SECCIÓN 2: GESTIÓN DE TABLAS Y DATATABLES
 ****************************************************/
// Funciones

/**
 * Inicializa la tabla de DataTables con todas sus configuraciones
 * @returns {Object} Instancia de DataTable configurada
 */
function initializeDataTable() {
    return $('#ApprovalAccessTable').DataTable({
        responsive: true,
        scrollX: true, // Permitir scroll horizontal
        dom: 'Bfrtip', // Incluir botones de exportación
        buttons: [
            {
                extend: 'excel',
                text: 'Exportar a Excel',
                className: 'btn btn-success',
                filename: `${tituloPagina}_${fechaActual}`, // Nombre con título + fecha
                title: `${tituloPagina} - ${fechaActual}`, // Título dentro del archivo
                exportOptions: {
                    columns: ':visible',
                    modifier: {
                        page: 'all'
                    },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $('#ApprovalAccessTable_wrapper .dataTables_scrollHead thead tr:nth-child(1) th');
                            if (headerSelector.length === 0) {
                                headerSelector = $('#ApprovalAccessTable thead th');
                            }
                            return headerSelector.eq(columnIdx).text();
                        }
                    }
                }
            },
            {
                extend: 'pdf',
                text: 'Exportar a PDF',
                className: 'btn btn-danger',
                filename: `${tituloPagina}_${fechaActual}`, // Nombre con título + fecha
                title: `${tituloPagina} - ${fechaActual}`, // Título dentro del archivo
                exportOptions: {
                    columns: ':visible',
                    modifier: {
                        page: 'all'
                    },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $('#ApprovalAccessTable_wrapper .dataTables_scrollHead thead tr:nth-child(1) th');
                            if (headerSelector.length === 0) {
                                headerSelector = $('#ApprovalAccessTable thead th');
                            }
                            return headerSelector.eq(columnIdx).text();
                        }
                    }
                }
            }
        ],
        pageLength: 10, // Número de registros por página
        ordering: false, // Deshabilitar ordenamiento global
        searching: true, // Activar búsqueda
        lengthMenu: [[10, 25, 50, -1], [10, 25, 50, "Todos"]], // Opciones de registros por página
        columnDefs: [
            {
                targets: -1, // Última columna (acciones)
                orderable: false, // No permitir ordenamiento
                searchable: false // No incluir en la búsqueda
            }
        ],
        order: [[0, 'asc']] // Orden por defecto (por nombre de compañía)
    });
}

/**
 * Configura los filtros de búsqueda en cada columna de la tabla
 */
function setupTableFilters() {
    $('#ApprovalAccessTable thead tr:eq(1) th input').on('keyup change', function () {
        let columnIndex = $(this).parent().index(); // Obtener el índice de la columna
        table.column(columnIndex).search(this.value).draw(); // Aplicar búsqueda en esa columna
    });
}

/**
 * Carga los datos de empleados desde el servidor y los muestra en la tabla
 */
function loadTableDate() {
    showLoading();
    fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/list')
        .then(response => response.json())
        .then(data => {
            const tableBody = document.getElementById("ApprovalAccessbody");
            tableBody.innerHTML = "";

            data.forEach(employee => {
                const row = document.createElement("tr");
                row.innerHTML = `
                    <td>${employee.RequestID || ''}</td>
                    <td>${employee.FirstName || ''}</td>
                    <td>${employee.LastName || ''}</td>
                    <td>${employee.Division || ''}</td>
                    <td>${employee.Departament || ''}</td>
                    <td>${employee.Position || ''}</td>
                    <td>${employee.DirectSupervisor || ''}</td>
                    <td>${employee.PermissionType || ''}</td>
                    <td>${employee.CreatedAt || ''}</td>
                    <td>
                        <button class="btn-approve" onclick="approve('${employee.RequestID}')" title="Aprobar">
                            Aprobar
                        </button>
                        <button class="btn-reject" onclick="reject('${employee.RequestID}')" title="Rechazar">
                            Rechazar
                        </button>
                    </td>
                `;
                tableBody.appendChild(row);
            });

            if (table) {
                table.destroy();
            }

            table = initializeDataTable();
            setupTableFilters();
        })
        .catch(error => {
            console.error("Error al cargar los empleados:", error);
            alert("Hubo un problema al cargar los empleados. Intenta nuevamente.");
        })
        .finally(() => hideLoading());
}

/****************************************************
 * SECCIÓN 3: ACCIONES SOBRE EMPLEADOS (APROBAR)
 ****************************************************/
// Funciones

/**
 * Abre el modal para aprobar un empleado y carga su información
 * @param {string} RequestID - ID del empleado a aprobar
 */
function approve(RequestID) {
    selectedId = RequestID;
    let modal = new bootstrap.Modal(document.getElementById('EstandarModal'));
    modal.show();
    loadinfo();
}

/**
 * Valida que todos los campos requeridos estén completos
 * @returns {boolean} - true si todos los campos están completos
 */
function validateRequiredFields() {
    let valid = true;
    
    // Lista de IDs de campos requeridos
    const requiredFields = [
        "FirstName", "LastName", "SecondLastName", "Company", 
        "Division", "Departament", "RequestType", "DirectSupervisor", 
        "Position", "RolAssig", "PermissionType", "CorporateEmail", 
        "ContactPhone", "EmployeeNumber"
    ];
    
    // Validar cada campo requerido
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (!field.value.trim()) {
            field.classList.add("invalid-input");
            valid = false;
        } else {
            field.classList.remove("invalid-input");
        }
        
        // Agregar event listener para quitar la clase de error al editar
        field.addEventListener("input", function() {
            this.classList.remove("invalid-input");
        });
    });
    
    if (!valid) {
        Swal.fire({
            icon: 'error',
            title: 'Campos incompletos',
            text: 'Por favor, complete todos los campos obligatorios marcados con *',
            confirmButtonColor: '#1f71c2'
        });
        return false;
    }
    
    // Si todos los campos tienen valores, validar el formato del correo y teléfono
    return validateEmailFormat() && validatePhoneFormat();
}

/**
 * Valida que el formato del correo electrónico sea correcto
 * @returns {boolean} - true si el formato es válido
 */
function validateEmailFormat() {
    const emailField = document.getElementById("CorporateEmail");
    const emailValue = emailField.value.trim();
    
    // Expresión regular para validar correos electrónicos
    const emailRegex = /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,6}$/;
    
    if (!emailRegex.test(emailValue)) {
        emailField.classList.add("invalid-input");
        
        Swal.fire({
            icon: 'error',
            title: 'Formato de correo inválido',
            text: 'Por favor, ingrese un correo electrónico válido',
            confirmButtonColor: '#1f71c2'
        });
        
        return false;
    }
    
    emailField.classList.remove("invalid-input");
    return true;
}

/**
 * Valida que el formato del número de teléfono sea correcto
 * @returns {boolean} - true si el formato es válido
 */
function validatePhoneFormat() {
    const phoneField = document.getElementById("ContactPhone");
    const phoneValue = phoneField.value.trim();
    
    // Expresión regular para validar números de teléfono (10 dígitos)
    // Se aceptan formatos: 1234567890, 123-456-7890, (123) 456-7890, etc.
    const phoneRegex = /^(\+?\d{1,3}[- ]?)?\(?(\d{3})\)?[- ]?(\d{3})[- ]?(\d{4})$/;
    
    if (!phoneRegex.test(phoneValue)) {
        phoneField.classList.add("invalid-input");
        
        Swal.fire({
            icon: 'error',
            title: 'Formato de teléfono inválido',
            text: 'Por favor, ingrese un número de teléfono válido',
            confirmButtonColor: '#1f71c2'
        });
        
        return false;
    }
    
    phoneField.classList.remove("invalid-input");
    return true;
}

/**
 * Verifica si el correo ya existe en la base de datos
 * @returns {Promise<boolean>} - Promise que resuelve a true si el correo no existe
 */
async function checkEmailExists() {
    const emailField = document.getElementById("CorporateEmail");
    const emailValue = emailField.value.trim();
    
    try {
        showLoading();
        
        const response = await fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/check-email?email=${encodeURIComponent(emailValue)}`);
        const data = await response.json();
        
        if (data.exists) {
            emailField.classList.add("invalid-input");
            
            Swal.fire({
                icon: 'warning',
                title: 'Correo ya registrado',
                text: 'El correo electrónico ya está registrado en el sistema. El usuario puede restablecer su contraseña dando clic en "Olvidé contraseña".',
                confirmButtonColor: '#1f71c2'
            });
            
            return false;
        }
        
        return true;
    } catch (error) {
        console.error("Error al verificar el correo:", error);
        
        Swal.fire({
            icon: 'error',
            title: 'Error de conexión',
            text: 'No se pudo verificar el correo electrónico. Por favor, intente nuevamente.',
            confirmButtonColor: '#1f71c2'
        });
        
        return false;
    } finally {
        hideLoading();
    }
}

/**
 * Guarda los datos actualizados del empleado
 */
async function guardarDatosEmpleado() {
    // Validar campos requeridos, formato de correo y teléfono
    if (!validateRequiredFields()) return;
    
    // Verificar si el correo ya existe
    const emailNotExists = await checkEmailExists();
    if (!emailNotExists) return;
    
    // Si pasa todas las validaciones, continuar con el proceso de guardado
    let payload = {};
    
    // Obtener todos los valores de los inputs y textareas normales
    document.querySelectorAll("#EstandarModal input, #EstandarModal textarea").forEach(el => {
        if (el.id !== 'Company' && el.id !== 'Division' && el.id !== 'Departament' &&
            el.id !== 'RequestType' && el.id !== 'RolAssig' && el.id !== 'PermissionAssign') {
            payload[el.id] = el.value;
        }
    });
    
    // Obtener los valores y IDs de los selects
    const companySelect = document.getElementById('Company');
    const divisionSelect = document.getElementById('Division');
    const departamentSelect = document.getElementById('Departament');
    const requestTypeSelect = document.getElementById('RequestType');
    const rolSelect = document.getElementById('RolAssig');
    const permissionSelect = document.getElementById('PermissionAssign');
    
    // Guardar los textos visibles para referencia
    payload.CompanyText = companySelect.options[companySelect.selectedIndex].text;
    payload.DivisionText = divisionSelect.options[divisionSelect.selectedIndex].text;
    payload.DepartamentText = departamentSelect.options[departamentSelect.selectedIndex].text;
    payload.RequestTypeText = requestTypeSelect.options[requestTypeSelect.selectedIndex].text;
    
    // Guardar los IDs para la base de datos (esto es lo importante)
    payload.CompanyID = companySelect.value;
    payload.DivisionID = divisionSelect.value;
    payload.DepartamentID = departamentSelect.value;
    payload.RequestTypeID = requestTypeSelect.value;
    
    // Añadir los valores de los nuevos selects
    payload.RoleID = rolSelect.value;
    payload.PermissionAssign = permissionSelect.value;
    
    // RequestID desde el backend
    payload.RequestID = selectedId;
    
    // Mostrar loading y enviar la solicitud
    showLoading();
    
    fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
    })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                // Mostrar error con SweetAlert
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Error: ' + data.error,
                    confirmButtonColor: '#1f71c2'
                });
            } else {
                closeModal();
                
                // Mostrar mensaje de éxito con SweetAlert y recargar al cerrar
                Swal.fire({
                    icon: 'success',
                    title: '¡Éxito!',
                    text: 'Se envió satisfactoriamente el correo al usuario',
                    confirmButtonColor: '#1f71c2',
                    timer: 2000, // Cerrará automáticamente después de 2 segundos
                    willClose: () => {
                        window.location.reload(); // Recarga la página cuando se cierra la alerta
                    }
                });
            }
        })
        .catch(error => {
            console.error("Error al guardar datos:", error);
            
            // Mostrar error con SweetAlert
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Hubo un problema al guardar los datos. Por favor, inténtelo nuevamente.',
                confirmButtonColor: '#1f71c2'
            });
        })
        .finally(() => hideLoading());
}

/****************************************************
 * SECCIÓN 4: ACCIONES SOBRE EMPLEADOS (RECHAZAR)
 ****************************************************/


// Variables globales - agregar debajo de selectedId en tu código
let selectedIdForReject = null;

/**
 * Abre el modal para rechazar un empleado y carga su información
 * @param {string} RequestID - ID del empleado a rechazar
 */
function reject(RequestID) {
    // Guardar el ID del empleado seleccionado para rechazo
    selectedIdForReject = RequestID;
    
    // Mostrar indicador de carga
    showLoading();
    
    // Cargar la información del empleado, incluyendo comentarios
    fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/getinfo?RequestID=${RequestID}`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`Error de red: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (!data) throw new Error("No se encontraron datos");
            
            // Asignar los comentarios del empleado al campo correspondiente
            document.getElementById('Comments2').value = data.Comments || '';
            
            // Mostrar el modal de rechazo
            const rejectModal = new bootstrap.Modal(document.getElementById('RejectModal'));
            rejectModal.show();
        })
        .catch(error => {
            console.error("Error al cargar comentarios:", error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'No se pudieron cargar los comentarios del empleado.',
                confirmButtonColor: '#d33',
                confirmButtonText: 'Aceptar'
            });
        })
        .finally(() => {
            hideLoading();
        });
}

/**
 * Cierra el modal de rechazo
 */
function closeRejectModal() {
    const modalElement = document.getElementById('RejectModal');
    const modalInstance = bootstrap.Modal.getInstance(modalElement);
    if (modalInstance) {
        modalInstance.hide();
    }
    resetRejectModal();
}

/**
 * Resetea los campos del modal de rechazo
 */
function resetRejectModal() {
    // Limpiar los campos
    document.getElementById('Comments2').value = '';
    document.getElementById('Commentsresp').value = '';
    
    // Resetear el ID del empleado seleccionado
    selectedIdForReject = null;
}

/**
 * Confirma el rechazo del empleado y envía la solicitud al servidor
 */
function confirmReject() {
    // Verificar que tengamos un ID de empleado seleccionado
    if (!selectedIdForReject) {
        Swal.fire({
            icon: 'error',
            title: 'Error',
            text: 'No se ha seleccionado un empleado para rechazar.',
            confirmButtonColor: '#d33',
            confirmButtonText: 'Aceptar'
        });
        return;
    }
    
    // Obtener el comentario de rechazo (opcional)
    const comentario = document.getElementById('Commentsresp').value.trim();
    
    // Mostrar confirmación antes de proceder
    Swal.fire({
        title: '¿Confirmar rechazo?',
        text: "Esta acción rechazará la solicitud del empleado. No se podrá deshacer.",
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sí, rechazar',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            // Mostrar indicador de carga global
            showGlobalLoading("Procesando rechazo de empleado...");
            
            // Enviar solicitud al servidor
            fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/reject', {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    employeeId: selectedIdForReject,
                    comentario: comentario || null // Enviar null si no hay comentario
                })
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(data => {
                        throw new Error(data.error || 'Error al procesar la solicitud');
                    });
                }
                return response.json();
            })
            .then(data => {
                // Ocultar indicador de carga global ANTES de mostrar el mensaje de Swal
                hideGlobalLoading();
                
                // Cerrar el modal
                closeRejectModal();
                
                if (data.error_mail) {
                    // El rechazo se completó pero hubo un problema con el correo
                    Swal.fire({
                        icon: 'warning',
                        title: 'Solicitud rechazada',
                        text: 'La solicitud ha sido rechazada, pero hubo un problema al enviar la notificación por correo electrónico.',
                        confirmButtonColor: '#3085d6',
                        confirmButtonText: 'Aceptar'
                    }).then(() => {
                        window.location.reload();
                    });
                } else {
                    // Todo se completó correctamente
                    Swal.fire({
                        icon: 'success',
                        title: 'Solicitud rechazada',
                        text: 'La solicitud del empleado ha sido rechazada y se ha enviado una notificación.',
                        confirmButtonColor: '#3085d6',
                        confirmButtonText: 'Aceptar'
                    }).then(() => {
                        window.location.reload();
                    });
                }
            })
            .catch(error => {
                // Ocultar indicador de carga global ANTES de mostrar el error
                hideGlobalLoading();
                
                console.error("Error al rechazar empleado:", error);
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: error.message || 'Hubo un problema al rechazar la solicitud. Inténtelo de nuevo.',
                    confirmButtonColor: '#d33',
                    confirmButtonText: 'Aceptar'
                });
            });
        }
    });
}

// Función para mostrar el overlay de carga global
function showGlobalLoading(message = "Procesando solicitud...") {
    // Actualizar el mensaje si es necesario
    document.querySelector("#global-loading-overlay .loading-text").textContent = message;
    
    // Mostrar el overlay
    document.getElementById("global-loading-overlay").style.display = "flex";
    
    // Bloquear el scroll
    document.body.style.overflow = "hidden";
}

// Función para ocultar el overlay de carga global
function hideGlobalLoading() {
    // Ocultar el overlay
    document.getElementById("global-loading-overlay").style.display = "none";
    
    // Restaurar el scroll
    document.body.style.overflow = "auto";
}

// Agregar al final del archivo, en el evento DOMContentLoaded
document.addEventListener("DOMContentLoaded", function() {
    // ... código existente ...
    
    // Asignar evento al botón de confirmación de rechazo
    const confirmRejectBtn = document.getElementById('confirmRejectBtn');
    if (confirmRejectBtn) {
        confirmRejectBtn.addEventListener('click', confirmReject);
    }
    
    // Evento para limpiar el modal cuando se cierra
    document.getElementById('RejectModal').addEventListener('hidden.bs.modal', resetRejectModal);
});


/****************************************************
 * SECCIÓN 5: GESTIÓN DE DATOS DE EMPLEADO
 ****************************************************/
// Funciones

/**
 * Carga la información del empleado seleccionado 
 * y configura todos los campos del formulario modal
 */
function loadinfo() {
    showLoading();
    
    // Primero cargamos todos los datos de los selectbox
    loadCompanies();
    loadRequestTypes();
    
    fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/getinfo?RequestID=${selectedId}`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`Error de red: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (!data) throw new Error("No se encontraron datos");
            
            // Llenar los campos normales
            Object.keys(data).forEach(key => {
                let field = document.getElementById(key);
                if (field && key !== 'Company' && key !== 'Division' && key !== 'Departament') {
                    field.value = data[key] || '';
                }
            });
            
            // Para el select de tipo de formulario
            if (data.RequestType) {
                fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/request-types')
                    .then(response => response.json())
                    .then(requestTypes => {
                        const requestType = requestTypes.find(rt => rt.name === data.RequestType);
                        if (requestType) {
                            const requestTypeSelect = document.getElementById('RequestType');
                            requestTypeSelect.value = requestType.id;
                        }
                    })
                    .catch(error => console.error("Error al cargar tipos de formulario:", error));
            }
            
            // Para los selects en cascada, necesitamos hacer un manejo especial
            if (data.Company) {
                // Primero, encontrar el ID de la compañía seleccionada
                fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/companies')
                    .then(response => response.json())
                    .then(companies => {
                        const company = companies.find(c => c.name === data.Company);
                        if (company) {
                            const companySelect = document.getElementById('Company');
                            companySelect.value = company.id;
                            
                            // Ahora cargamos las divisiones para esta compañía
                            loadDivisions(company.id);
                            
                            // Y luego, seleccionamos la división correcta y cargamos los departamentos
                            if (data.Division) {
                                setTimeout(() => {
                                    fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/divisions?companyId=${company.id}`)
                                        .then(response => response.json())
                                        .then(divisions => {
                                            const division = divisions.find(d => d.name === data.Division);
                                            if (division) {
                                                const divisionSelect = document.getElementById('Division');
                                                divisionSelect.value = division.id;
                                                
                                                // Finalmente, cargamos los departamentos y seleccionamos el correcto
                                                loadDepartments(company.id, division.id);
                                                
                                                if (data.Departament) {
                                                    setTimeout(() => {
                                                        fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/departments?companyId=${company.id}&divisionId=${division.id}`)
                                                            .then(response => response.json())
                                                            .then(departments => {
                                                                const department = departments.find(d => d.name === data.Departament);
                                                                if (department) {
                                                                    const departamentSelect = document.getElementById('Departament');
                                                                    departamentSelect.value = department.id;
                                                                    
                                                                    // Cargar los roles después de que se hayan seleccionado los departamentos
                                                                    loadFilteredRoles();
                                                                    
                                                                    // Si tenemos un RoleID en los datos, seleccionarlo después de que se carguen los roles
                                                                    if (data.RoleID) {
                                                                        setTimeout(() => {
                                                                            const roleSelect = document.getElementById('RolAssig');
                                                                            roleSelect.value = data.RoleID;
                                                                        }, 500);
                                                                    }
                                                                }
                                                            })
                                                            .catch(error => console.error("Error al cargar departamentos:", error));
                                                    }, 300);
                                                }
                                            }
                                        })
                                        .catch(error => console.error("Error al cargar divisiones:", error));
                                }, 300);
                            }
                        }
                    })
                    .catch(error => console.error("Error al cargar compañías:", error));
            }
        })
        .catch(error => {
            console.error("Error al cargar información:", error);
            alert("Hubo un problema al cargar la información del empleado. Intente nuevamente.");
        })
        .finally(() => hideLoading());
}

/****************************************************
 * SECCIÓN 6: GESTIÓN DE SELECTS DEPENDIENTES
 ****************************************************/
/*-------------------- Tipos de Formulario --------------------*/
// Funciones
/**
 * Carga las opciones para el select de tipos de formulario
 */
function loadRequestTypes() {
    fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/request-types')
        .then(response => response.json())
        .then(data => {
            const requestTypeSelect = document.getElementById('RequestType');
            requestTypeSelect.innerHTML = '<option value="">Seleccione un tipo de formulario</option>';

            data.forEach(requestType => {
                const option = document.createElement('option');
                option.value = requestType.id;
                option.textContent = requestType.name;
                requestTypeSelect.appendChild(option);
            });
        })
        .catch(error => console.error("Error al cargar tipos de formulario:", error));
}

/*-------------------- Compañías --------------------*/
// Funciones
/**
 * Carga las opciones para el select de compañías
 */
function loadCompanies() {
    fetch('/SupyCtrol_Module/DataMaster/DMAccessEmployee/companies')
        .then(response => response.json())
        .then(data => {
            const companySelect = document.getElementById('Company');
            companySelect.innerHTML = '<option value="">Seleccione una compañía</option>';

            data.forEach(company => {
                const option = document.createElement('option');
                option.value = company.id;
                option.textContent = company.name;
                companySelect.appendChild(option);
            });
        })
        .catch(error => console.error("Error al cargar compañías:", error));
}

/*-------------------- Divisiones --------------------*/
// Funciones
/**
 * Carga las divisiones basadas en la compañía seleccionada
 * @param {string} companyId - ID de la compañía seleccionada
 */
function loadDivisions(companyId) {
    const divisionSelect = document.getElementById('Division');
    const departamentSelect = document.getElementById('Departament');

    // Resetear y deshabilitar los select dependientes
    divisionSelect.innerHTML = '<option value="">Seleccione una división</option>';
    departamentSelect.innerHTML = '<option value="">Seleccione un departamento</option>';

    departamentSelect.disabled = true;

    if (!companyId) {
        divisionSelect.disabled = true;
        return;
    }

    divisionSelect.disabled = false;

    fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/divisions?companyId=${companyId}`)
        .then(response => response.json())
        .then(data => {
            data.forEach(division => {
                const option = document.createElement('option');
                option.value = division.id;
                option.textContent = division.name;
                divisionSelect.appendChild(option);
            });
        })
        .catch(error => console.error("Error al cargar divisiones:", error));
}

/*-------------------- Departamentos --------------------*/
// Funciones
/**
 * Carga los departamentos basados en la compañía y división seleccionadas
 * @param {string} companyId - ID de la compañía
 * @param {string} divisionId - ID de la división
 */
function loadDepartments(companyId, divisionId) {
    const departamentSelect = document.getElementById('Departament');

    // Resetear el select de departamentos
    departamentSelect.innerHTML = '<option value="">Seleccione un departamento</option>';

    if (!companyId || !divisionId) {
        departamentSelect.disabled = true;
        return;
    }

    departamentSelect.disabled = false;

    fetch(`/SupyCtrol_Module/DataMaster/DMAccessEmployee/departments?companyId=${companyId}&divisionId=${divisionId}`)
        .then(response => response.json())
        .then(data => {
            data.forEach(department => {
                const option = document.createElement('option');
                option.value = department.id;
                option.textContent = department.name;
                departamentSelect.appendChild(option);
            });
        })
        .catch(error => console.error("Error al cargar departamentos:", error));
}

/*-------------------- Roles Filtrados --------------------*/
// Funciones
/**
 * Carga roles filtrados según la selección de compañía, división y departamento
 */
function loadFilteredRoles() {
    const companySelect = document.getElementById('Company');
    const divisionSelect = document.getElementById('Division');
    const departamentSelect = document.getElementById('Departament');
    
    // Verificar si los tres selects tienen valores seleccionados
    if (!companySelect.value || !divisionSelect.value || !departamentSelect.value) {
        // Si alguno no tiene valor, limpiar el select de roles
        const roleSelect = document.getElementById('RolAssig');
        roleSelect.innerHTML = '';
        
        const defaultOption = document.createElement('option');
        defaultOption.value = "";
        defaultOption.textContent = "Seleccione Compañía, División y Departamento primero";
        defaultOption.disabled = true;
        defaultOption.selected = true;
        roleSelect.appendChild(defaultOption);
        
        roleSelect.disabled = true;
        return;
    }
    
    // Si todos tienen valores, cargar los roles filtrados
    const url = `/SupyCtrol_Module/DataMaster/DMAccessEmployee/filtered-roles?companyId=${companySelect.value}&divisionId=${divisionSelect.value}&departamentId=${departamentSelect.value}`;
    
    fetch(url)
        .then(response => response.json())
        .then(data => {
            const roleSelect = document.getElementById('RolAssig');
            
            // Limpiar todas las opciones actuales
            roleSelect.innerHTML = '';
            
            // Habilitar el select
            roleSelect.disabled = false;
            
            // Añadir la opción predeterminada como primera opción
            const defaultOption = document.createElement('option');
            defaultOption.value = "";
            defaultOption.textContent = "Seleccione un rol. Si no lo encuentra, regístrelo.";
            defaultOption.disabled = true;
            defaultOption.selected = true;
            roleSelect.appendChild(defaultOption);
            
            if (data.length === 0) {
                // Si no hay roles disponibles para esta combinación
                const noRolesOption = document.createElement('option');
                noRolesOption.value = "";
                noRolesOption.textContent = "No hay roles disponibles para esta selección";
                noRolesOption.disabled = true;
                roleSelect.appendChild(noRolesOption);
            } else {
                // Añadir los roles disponibles
                data.forEach(role => {
                    const option = document.createElement('option');
                    option.value = role.id;
                    option.textContent = role.name;
                    roleSelect.appendChild(option);
                });
            }
            
            // Forzar que la primera opción sea la seleccionada inicialmente
            roleSelect.selectedIndex = 0;
        })
        .catch(error => {
            console.error("Error al cargar roles filtrados:", error);
            
            // En caso de error, mostrar un mensaje
            const roleSelect = document.getElementById('RolAssig');
            roleSelect.innerHTML = '';
            
            const errorOption = document.createElement('option');
            errorOption.value = "";
            errorOption.textContent = "Error al cargar roles. Intente nuevamente.";
            errorOption.disabled = true;
            errorOption.selected = true;
            roleSelect.appendChild(errorOption);
        });
}

/****************************************************
 * SECCIÓN 7: INICIALIZACIÓN Y EVENT LISTENERS
 ****************************************************/
// Inicialización al cargar el documento
document.addEventListener("DOMContentLoaded", function() {
    // Cargar tabla de empleados
    loadTableDate();
    
    // Event listener para el modal
    document.getElementById('EstandarModal').addEventListener('hidden.bs.modal', resetModal);
    
    // Event listener para el botón guardar
    if (document.getElementById('guardarEmpleado')) {
        document.getElementById('guardarEmpleado').addEventListener('click', guardarDatosEmpleado);
    }
    
    // Event listener para cambios en el select de compañía
    document.getElementById('Company').addEventListener('change', function() {
        loadDivisions(this.value);
        // Resetear el select de roles al cambiar la compañía
        const roleSelect = document.getElementById('RolAssig');
        roleSelect.innerHTML = '';
        const defaultOption = document.createElement('option');
        defaultOption.value = "";
        defaultOption.textContent = "Seleccione Compañía, División y Departamento primero";
        defaultOption.disabled = true;
        defaultOption.selected = true;
        roleSelect.appendChild(defaultOption);
        roleSelect.disabled = true;
    });
    
    // Event listener para cambios en el select de división
    document.getElementById('Division').addEventListener('change', function() {
        const companyId = document.getElementById('Company').value;
        loadDepartments(companyId, this.value);
        // Resetear el select de roles al cambiar la división
        const roleSelect = document.getElementById('RolAssig');
        roleSelect.innerHTML = '';
        const defaultOption = document.createElement('option');
        defaultOption.value = "";
        defaultOption.textContent = "Seleccione Compañía, División y Departamento primero";
        defaultOption.disabled = true;
        defaultOption.selected = true;
        roleSelect.appendChild(defaultOption);
        roleSelect.disabled = true;
    });
    
    // Event listener para cambios en el select de departamento
    document.getElementById('Departament').addEventListener('change', function() {
        // Cargar roles filtrados cuando se selecciona un departamento
        loadFilteredRoles();
    });
});