// ===============================================================================
// Archivo: RecepcionCotiz.js
// ===============================================================================

/**
 * Resetea y limpia todos los campos del formulario.
 */
function resetForm() {
  // Campos de CRM
  const crmInputs = [
    "CRM_OpportunityNumber",
    "CRM_ContactName",
    "CRM_ContactType",
    "CRM_AssignedSalesperson",
    "CRM_ContactAdress",
    "CRM_ContactColonia",
    "CRM_ContactCity",
    "CRM_ContactNumber",
    "CRM_ContactCountry",
    "CRM_ContactLegalIdentifier",
    "CRM_ContactZip",
    "CRM_ContactState",
    "CRM_ContactEmail",
  ];
  crmInputs.forEach((id) => {
    const input = document.getElementById(id);
    if (input) {
      input.value = "";
      input.classList.add("disabled-style");
    }
  });

  // Campos financieros
  const financeInputs = [
    "Finance_ListPrice",
    "Finance_Discount",
    "Finance_SalePriceMin",
    "Finance_CaseCost",
    "Finance_OfferPrice",
    "Finance_PriceWithIVA",
  ];
  financeInputs.forEach((id) => {
    const input = document.getElementById(id);
    if (input) {
      input.value = "";
      input.classList.add("disabled-style");
      input.disabled = true;
    }
  });

  // Select de Moneda
  const currencySelect = document.getElementById("Finance_CurrencyType");
  if (currencySelect) {
    currencySelect.selectedIndex = 0;
    currencySelect.classList.add("disabled-style");
    currencySelect.disabled = true;
  }

  // Select de IVA
  const ivaSelect = document.getElementById("Finance_IVA");
  if (ivaSelect) {
    ivaSelect.selectedIndex = 0;
    ivaSelect.classList.add("disabled-style");
    ivaSelect.disabled = true;
  }

  // Resetear tabla de líneas de cotización
  resetQuotationLinesTable();
}

/**
 * Limpia la tabla de líneas de cotización.
 */
function resetQuotationLinesTable() {
  const tableBody = document.getElementById("ApprovalAccessDistributorsbody");

  if (tableBody) {
    tableBody.innerHTML = "";
    console.log("Tabla de líneas de cotización limpiada");
  }

  // Deshabilitar botón de envío cuando se limpia la tabla
  const sendButton = document.querySelector(".row-actions.center button");
  if (sendButton) {
    sendButton.disabled = true;
  }
}

/**
 * Aplica un filtro al input del Número de Cotización para aceptar solo números.
 */
function applyNumericInputFilter(inputElement) {
  if (!inputElement) return;

  inputElement.addEventListener("input", (e) => {
    const oldValue = inputElement.value;
    const newValue = oldValue.replace(/[^0-9]/g, "");

    if (oldValue !== newValue) {
      inputElement.value = newValue;
    }
  });

  inputElement.addEventListener("paste", (e) => {
    const paste = (e.clipboardData || window.clipboardData).getData("text");
    if (/[^0-9]/g.test(paste)) {
      e.preventDefault();
    }
  });
}

/**
 * Formatea un número como moneda mexicana con símbolo de pesos y separadores de miles.
 */
function formatCurrency(value) {
  const num = parseFloat(value) || 0;
  return (
    "$ " +
    num.toLocaleString("es-MX", {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    })
  );
}

/**
 * Extrae el número de un valor formateado como moneda.
 */
function unformatCurrency(value) {
  if (!value) return 0;
  // Remover símbolo $, espacios, y solo mantener números y punto
  let cleanValue = value.replace(/[\$\s]/g, "").replace(/,/g, "");
  return parseFloat(cleanValue) || 0;
}

/**
 * Agrega event listeners al campo de Descuento para activar el campo de Precio con IVA.
 */
function initDiscountListener(unifiedData) {
  const listPriceInput = document.getElementById("Finance_ListPrice");
  const discountInput = document.getElementById("Finance_Discount");
  const offerPriceInput = document.getElementById("Finance_OfferPrice");
  const ivaSelect = document.getElementById("Finance_IVA");
  const priceWithIVAInput = document.getElementById("Finance_PriceWithIVA");
  const currencyType = document.getElementById("Finance_CurrencyType");

  if (
    !listPriceInput ||
    !discountInput ||
    !offerPriceInput ||
    !ivaSelect ||
    !priceWithIVAInput ||
    !currencyType
  ) {
    console.warn(
      "No se encontraron todos los inputs financieros para el listener."
    );
    return;
  }

  function updatePrices() {
    const listPrice = unformatCurrency(listPriceInput.value);
    const discountPercent = parseFloat(discountInput.value) || 0;

    const selectedOption = ivaSelect.options[ivaSelect.selectedIndex];
    const taxAmount = selectedOption.dataset.taxAmount
      ? parseFloat(selectedOption.dataset.taxAmount)
      : 0;
    const ivaEnabled = ivaSelect.value.trim() !== "";

    const caseCost = unifiedData.CaseCost;

    // Caso A: sin descuento
    if (caseCost === "A") {
      // En caso A, precio de oferta = precio de lista (sin descuento)
      offerPriceInput.value = formatCurrency(listPrice.toFixed(2));

      if (ivaEnabled) {
        let priceWithIVA = listPrice * (1 + taxAmount);
        priceWithIVAInput.value = formatCurrency(priceWithIVA.toFixed(2));
      } else {
        // Si no hay impuesto, precio con IVA = precio de oferta
        priceWithIVAInput.value = formatCurrency(listPrice.toFixed(2));
      }
      priceWithIVAInput.disabled = true;
      toggleNextSections();
      return;
    }

    // Casos B y C: con descuento (0% si no se ingresa)
    const maxDiscountPercent = unifiedData.DiscountMaxPercent
      ? parseFloat(unifiedData.DiscountMaxPercent)
      : 0;
    if (discountPercent > maxDiscountPercent) {
      Swal.fire({
        icon: "error",
        title: "Descuento Excedido",
        text: `El descuento ingresado (${discountPercent.toFixed(
          2
        )}%) no puede ser mayor al descuento máximo autorizado (${maxDiscountPercent.toFixed(
          2
        )}%).`,
        confirmButtonText: "Aceptar",
      });
      discountInput.value = maxDiscountPercent.toFixed(4);
      updatePrices();
      return;
    }

    // Calcular siempre, incluso si descuento es 0
    const offerPrice = listPrice * (1 - discountPercent / 100);
    offerPriceInput.value = formatCurrency(offerPrice.toFixed(2));

    let priceWithIVA = offerPrice;
    if (ivaEnabled) {
      priceWithIVA *= 1 + taxAmount;
    }
    priceWithIVAInput.value = formatCurrency(priceWithIVA.toFixed(2));

    priceWithIVAInput.disabled = true;

    toggleNextSections();
  }

  discountInput.addEventListener("input", updatePrices);
  ivaSelect.addEventListener("change", updatePrices);
  updatePrices();
}

/**
 * Habilita o deshabilita los campos de las secciones siguientes.
 * MODIFICADO: No valida si el descuento es 0, solo verifica que el campo tenga un valor.
 */
function toggleNextSections() {
  const listPriceInput = document.getElementById("Finance_ListPrice");
  const discountInput = document.getElementById("Finance_Discount");
  const ivaSelect = document.getElementById("Finance_IVA");
  const priceWithIVAInput = document.getElementById("Finance_PriceWithIVA");
  const caseCost = document.getElementById("Finance_CaseCost");

  // Habilitar el botón Previsualizar si hay precio de propuesta
  const allButtons = document.querySelectorAll("button");
  let previewButton = null;
  allButtons.forEach((btn) => {
    if (btn.textContent.includes("Previsualizar")) {
      previewButton = btn;
    }
  });

  if (previewButton) {
    previewButton.disabled = listPriceInput.value.trim() === "";
  }

  // Determinar si los campos financieros están completos
  let areFinanceFieldsFilled;

  if (caseCost && caseCost.value === "A") {
    // Caso A: solo requiere IVA y precio con IVA
    areFinanceFieldsFilled =
      listPriceInput.value.trim() !== "" &&
      ivaSelect.value.trim() !== "" &&
      priceWithIVAInput.value.trim() !== "";
  } else {
    // 🔥 Casos B/C: El descuento NO es obligatorio
    // Si está vacío se considera 0 automáticamente
    // NO bloquea la activación de las siguientes secciones
    areFinanceFieldsFilled =
      listPriceInput.value.trim() !== "" &&
      ivaSelect.value.trim() !== "" &&
      priceWithIVAInput.value.trim() !== "";
  }

  // 🔥 ACTIVAR/DESACTIVAR LA SECCIÓN DE TIEMPO DE ENTREGA
  const timeForm = document.getElementById("formTiempo");
  if (timeForm) {
    const timeInputs = timeForm.querySelectorAll("input, select");
    timeInputs.forEach((input) => {
      input.disabled = !areFinanceFieldsFilled;
      if (input.disabled) {
        input.classList.add("disabled-style");
      } else {
        input.classList.remove("disabled-style");
      }
    });
  }

  // 🔥 ACTIVAR/DESACTIVAR LA SECCIÓN DE TÉRMINOS Y CONDICIONES
  const condicionesTextarea = document.getElementById("address");
  if (condicionesTextarea) {
    condicionesTextarea.disabled = !areFinanceFieldsFilled;
    if (condicionesTextarea.disabled) {
      condicionesTextarea.classList.add("disabled-style");
    } else {
      condicionesTextarea.classList.remove("disabled-style");
    }
  }
}
/**
 * Activa los campos de acuerdo con la lógica de CaseCost.
 */
function enableFinanceFields(unifiedData) {
  const listPriceInput = document.getElementById("Finance_ListPrice");
  const discountInput = document.getElementById("Finance_Discount");
  const ivaSelect = document.getElementById("Finance_IVA");
  const offerPriceInput = document.getElementById("Finance_OfferPrice");
  const priceWithIVAInput = document.getElementById("Finance_PriceWithIVA");
  const currencyType = document.getElementById("Finance_CurrencyType");

  [
    listPriceInput,
    discountInput,
    ivaSelect,
    offerPriceInput,
    priceWithIVAInput,
    currencyType,
  ].forEach((input) => {
    if (input) {
      input.disabled = true;
      input.classList.add("disabled-style");
    }
  });

  const caseCost = unifiedData.CaseCost;

  if (listPriceInput) {
    listPriceInput.disabled = true;
    listPriceInput.classList.add("disabled-style");
  }

  if (caseCost === "A") {
    if (ivaSelect) {
      ivaSelect.disabled = false;
      ivaSelect.classList.remove("disabled-style");
    }
    if (currencyType) {
      currencyType.disabled = false;
      currencyType.classList.remove("disabled-style");
    }
  }

  if (caseCost === "B" || caseCost === "C") {
    if (discountInput) {
      discountInput.disabled = false;
      discountInput.classList.remove("disabled-style");
    }
    if (ivaSelect) {
      ivaSelect.disabled = false;
      ivaSelect.classList.remove("disabled-style");
    }
    if (currencyType) {
      currencyType.disabled = false;
      currencyType.classList.remove("disabled-style");
    }
  }

  const financeInputs = [
    listPriceInput,
    discountInput,
    ivaSelect,
    offerPriceInput,
    priceWithIVAInput,
  ];
  financeInputs.forEach((input) => {
    if (input) {
      input.addEventListener("input", toggleNextSections);
      input.addEventListener("change", toggleNextSections);
    }
  });

  toggleNextSections();
}

/**
 * Mapea y asigna los datos de la oportunidad CRM y costeo unificados.
 */
function populateOpportunityForm(unifiedData) {
  if (!unifiedData) {
    console.warn("No hay datos unificados para poblar el formulario.");
    return;
  }

  const crmInputs = [
    "CRM_OpportunityNumber",
    "CRM_ContactName",
    "CRM_ContactType",
    "CRM_AssignedSalesperson",
    "CRM_ContactAdress",
    "CRM_ContactColonia",
    "CRM_ContactCity",
    "CRM_ContactNumber",
    "CRM_ContactCountry",
    "CRM_ContactLegalIdentifier",
    "CRM_ContactZip",
    "CRM_ContactState",
    "CRM_ContactEmail",
  ];

  crmInputs.forEach((id, index) => {
    const input = document.getElementById(id);
    if (input) {
      const dataKeys = [
        "CRM_OpportunityNumber",
        "CRM_ContactName",
        "CRM_ContactType",
        "CRM_AssignedSalesperson",
        "CRM_ContactAdress",
        "CRM_ContactColonia",
        "CRM_ContactCity",
        "CRM_ContactNumber",
        "CRM_ContactCountry",
        "CRM_ContactLegalIdentifier",
        "CRM_ContactZip",
        "CRM_ContactState",
        "CRM_ContactEmail",
      ];
      input.value = unifiedData[dataKeys[index]] || "";
      input.classList.remove("disabled-style");
    }
  });

  const listPriceInput = document.getElementById("Finance_ListPrice");
  if (listPriceInput) {
    const priceValue = unifiedData.SalePriceList
      ? parseFloat(unifiedData.SalePriceList).toFixed(2)
      : "0.00";
    listPriceInput.value = formatCurrency(priceValue);
    listPriceInput.classList.remove("disabled-style");
  }

  const discountInput = document.getElementById("Finance_Discount");
  if (discountInput) {
    discountInput.classList.remove("disabled-style");
  }

  const salePriceMinInput = document.getElementById("Finance_SalePriceMin");
  if (salePriceMinInput) {
    salePriceMinInput.value = unifiedData.SalePriceMin
      ? parseFloat(unifiedData.SalePriceMin).toFixed(2)
      : "0.00";
    salePriceMinInput.classList.remove("disabled-style");
  }

  const caseCostInput = document.getElementById("Finance_CaseCost");
  if (caseCostInput) {
    caseCostInput.value = unifiedData.CaseCost || "";
    caseCostInput.classList.remove("disabled-style");
  }
}

let allTaxes = [];
let previousPriceWithIVA = ""; // Variable para guardar el precio anterior

/**
 * Llama a la API para obtener todos los impuestos con su moneda.
 */
async function fillTaxSelectInitial() {
  try {
    const response = await axios.get("/api/reception/taxes");
    allTaxes = response.data;
    console.log("Todos los impuestos cargados:", allTaxes);
  } catch (error) {
    console.error("Error al obtener los impuestos:", error);
  }
}

/**
 * Filtra los impuestos según la moneda seleccionada.
 */
function filterTaxesByCurrency(currencyCode) {
  const ivaSelect = document.getElementById("Finance_IVA");
  ivaSelect.innerHTML = '<option value="">Seleccione una opción</option>';

  if (!currencyCode) {
    ivaSelect.disabled = true;
    ivaSelect.classList.add("disabled-style");
    return;
  }

  const filteredTaxes = allTaxes.filter(
    (tax) => tax.CurrencyCode === currencyCode
  );

  console.log("Impuestos filtrados para", currencyCode, ":", filteredTaxes);

  if (filteredTaxes.length > 0) {
    filteredTaxes.forEach((tax) => {
      const option = document.createElement("option");
      option.value = tax.TaxCode;
      option.textContent = tax.FrontES;
      option.dataset.taxAmount = tax.TaxAmount;
      ivaSelect.appendChild(option);
    });
    ivaSelect.disabled = false;
    ivaSelect.classList.remove("disabled-style");
  } else {
    ivaSelect.disabled = true;
    ivaSelect.classList.add("disabled-style");
  }
}

/**
 * Inicializa el listener para cambios en la moneda.
 */
function initCurrencyListener() {
  const currencySelect = document.getElementById("Finance_CurrencyType");
  if (!currencySelect) {
    console.error("No se encontró el elemento Finance_CurrencyType");
    return;
  }

  currencySelect.addEventListener("change", (e) => {
    const priceWithIVAInput = document.getElementById("Finance_PriceWithIVA");
    const ivaSelect = document.getElementById("Finance_IVA");

    // Si se quita la moneda (valor vacío)
    if (!e.target.value || e.target.value.trim() === "") {
      console.log("Moneda quitada");

      // Restaurar el precio anterior
      if (priceWithIVAInput && previousPriceWithIVA) {
        priceWithIVAInput.value = previousPriceWithIVA;
      }

      // Resetear impuesto
      if (ivaSelect) {
        ivaSelect.selectedIndex = 0;
        ivaSelect.innerHTML = '<option value="">Seleccione una opción</option>';
        ivaSelect.disabled = true;
        ivaSelect.classList.add("disabled-style");
      }
    } else {
      // Se selecciona una nueva moneda
      console.log("Moneda seleccionada:", e.target.value);

      // Guardar el valor actual antes de resetear
      if (priceWithIVAInput && priceWithIVAInput.value.trim() !== "") {
        previousPriceWithIVA = priceWithIVAInput.value;
      }

      // Resetear solo el impuesto
      if (ivaSelect) {
        ivaSelect.selectedIndex = 0;
        ivaSelect.innerHTML = '<option value="">Seleccione una opción</option>';
      }

      // Restaurar el precio anterior con impuesto
      if (priceWithIVAInput && previousPriceWithIVA) {
        priceWithIVAInput.value = previousPriceWithIVA;
      }

      // Filtrar impuestos por nueva moneda
      filterTaxesByCurrency(e.target.value);
    }
  });
}

/**
 * Realiza la llamada a la API para obtener los datos unificados de costeo y CRM.
 */
async function fetchCostingDataByCostingNum(costingNum) {
  const url = `/api/reception/costing/${costingNum}`;

  try {
    const response = await axios.get(url, {
      headers: { Accept: "application/json" },
    });

    const data = response.data;
    console.log("Datos unificados recibidos del backend:", data);

    if (!data || Object.keys(data).length === 0) {
      return null;
    }
    return data;
  } catch (error) {
    if (
      axios.isAxiosError(error) &&
      error.response &&
      error.response.status === 404
    ) {
      console.log(
        `Cotización con número ${costingNum} no encontrada. (Error 404)`
      );
    } else {
      console.error(
        "Error al obtener los datos de costeo:",
        error.response ? error.response.data : error.message
      );
    }
    return null;
  }
}

/**
 * Llama a la API para obtener las monedas y llena el select de Moneda.
 */
async function fillCurrencySelect() {
  const currencySelect = document.getElementById("Finance_CurrencyType");
  currencySelect.innerHTML = '<option value="">Seleccione una opción</option>';
  try {
    const response = await axios.get("/api/reception/currencies");
    const currencies = response.data;
    if (currencies && currencies.length > 0) {
      currencies.forEach((currency) => {
        const option = document.createElement("option");
        option.value = currency.CurrencyCode;
        option.textContent = currency.FrontES;
        currencySelect.appendChild(option);
      });
    }
  } catch (error) {
    console.error("Error al obtener las monedas:", error);
  }
}

/**
 * Inicializa los event listeners para el botón de búsqueda.
 */
function initReceptionSearch() {
  const opnumberInput = document.getElementById("opnumber");
  const opbtn = document.getElementById("opbtn");

  applyNumericInputFilter(opnumberInput);
  opbtn.addEventListener("click", async () => {
    resetForm();

    const costingNum = opnumberInput.value.trim();

    if (!costingNum) {
      Swal.fire({
        icon: "warning",
        title: "Entrada requerida",
        text: "Por favor, ingrese un número de cotización.",
        confirmButtonText: "Aceptar",
      });
      return;
    }

    const unifiedData = await fetchCostingDataByCostingNum(costingNum);

    if (unifiedData) {
      Swal.fire({
        icon: "success",
        title: "Cotización Encontrada",
        html: `Cotización ${costingNum} cargada exitosamente.`,
        confirmButtonText: "Aceptar",
      });

      currentUnifiedData = unifiedData;
      populateOpportunityForm(unifiedData);
      await fillCurrencySelect();
      await fillTaxSelectInitial();
      initCurrencyListener();
      enableFinanceFields(unifiedData);
      initDiscountListener(unifiedData);
    } else {
      Swal.fire({
        icon: "error",
        title: "Cotización No Encontrada",
        text: "No se encontró una cotización con el número proporcionado.",
        confirmButtonText: "Aceptar",
      });
    }
  });
}

document.addEventListener("DOMContentLoaded", () => {
  initReceptionSearch();
  toggleNextSections();

  // Agregar listener al botón Previsualizar y Enviar
  setTimeout(() => {
    const allButtons = document.querySelectorAll("button");
    console.log("Buscando botones, total encontrados:", allButtons.length);

    allButtons.forEach((btn) => {
      const buttonText = btn.textContent.trim();
      console.log("Botón encontrado:", buttonText);

      if (buttonText.includes("Previsualizar")) {
        btn.addEventListener("click", handlePreviewClick);
        console.log("Listener agregado a Previsualizar");
      }
      if (
        buttonText.includes("Enviar Cotización") ||
        buttonText.includes("Enviar")
      ) {
        btn.addEventListener("click", handleSendQuotation);
        console.log("Listener agregado a Enviar Cotización");
      }
    });
  }, 1000);
});

// Variable global para almacenar las líneas de cotización
let currentQuotationLines = [];
let currentUnifiedData = null; // Nueva variable para guardar los datos unificados
/**
 * Maneja el clic en el botón Previsualizar.
 * Solo carga las líneas de cotización en la tabla del formulario.
 */
async function handlePreviewClick() {
  const opnumberInput = document.getElementById("opnumber");
  const listPriceInput = document.getElementById("Finance_ListPrice");
  const offerPriceInput = document.getElementById("Finance_OfferPrice");

  const costingNum = opnumberInput.value.trim();
  let offerPrice = unformatCurrency(offerPriceInput.value);

  if (offerPrice === 0 || !offerPrice) {
    offerPrice = unformatCurrency(listPriceInput.value);
  }

  console.log(`Precio de la oferta: ${offerPrice}`);

  // Obtener las líneas de costeo
  try {
    const response = await axios.get(
      `/api/reception/costing-details/${costingNum}`
    );
    const costingDetails = response.data;

    if (costingDetails && costingDetails.length > 0) {
      const originalTotalAmount = costingDetails.reduce(
        (sum, detail) => sum + detail.Amount,
        0
      );

      console.log(`Suma original de importes: ${originalTotalAmount}`);
      console.log(`Precio de oferta a alcanzar: ${offerPrice}`);

      const adjustmentFactor = offerPrice / originalTotalAmount;
      console.log(`Factor de ajuste: ${adjustmentFactor}`);

      const quotationLines = costingDetails.map((detail) => ({
        ...detail,
        OriginalAmount: detail.Amount,
        Amount: detail.Amount * adjustmentFactor,
      }));

      console.log("Líneas de cotización recalculadas:", quotationLines);

      const newTotal = quotationLines.reduce(
        (sum, line) => sum + line.Amount,
        0
      );
      console.log(`Nuevo total: ${newTotal}`);

      // Guardar las líneas en la variable global
      currentQuotationLines = quotationLines;

      // Mostrar las líneas en la tabla
      displayQuotationLines(quotationLines);

      // ✅ Habilitar el botón "Enviar Cotización" después de previsualizar
      const sendButton = document.querySelector(".row-actions.center button");
      if (sendButton) {
        sendButton.disabled = false;
        console.log("Botón 'Enviar Cotización' habilitado");
      }
    } else {
      Swal.fire({
        icon: "warning",
        title: "Sin líneas",
        text: "No se encontraron líneas de costeo para esta cotización.",
        confirmButtonText: "Aceptar",
      });
    }
  } catch (error) {
    console.error("Error al obtener líneas de costeo:", error);
    Swal.fire({
      icon: "error",
      title: "Error",
      text: "Error al obtener las líneas de costeo.",
      confirmButtonText: "Aceptar",
    });
  }
}

/**
 * Agrega botones fijos (Confirmar y Cancelar) a la ventana de vista previa.
 */
function addFixedButtonsToPreview(previewWindow, quotationData) {
  const doc = previewWindow.document;

  // 🔥 Guardar referencia a la ventana principal
  const mainWindow = window;

  // Crear contenedor de botones
  const buttonContainer = doc.createElement("div");
  buttonContainer.style.cssText = `
    position: fixed;
    bottom: 0;
    left: 0;
    width: 100%;
    background: linear-gradient(to top, rgba(255,255,255,0.98) 0%, rgba(255,255,255,0.95) 100%);
    padding: 20px;
    box-shadow: 0 -2px 10px rgba(0,0,0,0.1);
    display: flex;
    justify-content: center;
    gap: 20px;
    z-index: 9999;
    border-top: 2px solid #e0e0e0;
  `;

  // Botón Confirmar
  const confirmButton = doc.createElement("button");
  confirmButton.textContent = "✅ Confirmar y Enviar Cotización";
  confirmButton.style.cssText = `
    background-color: #059669;
    color: white;
    padding: 15px 40px;
    border: none;
    border-radius: 8px;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 6px rgba(5, 150, 105, 0.3);
  `;

  confirmButton.onmouseover = () => {
    confirmButton.style.backgroundColor = "#047857";
    confirmButton.style.transform = "translateY(-2px)";
    confirmButton.style.boxShadow = "0 6px 8px rgba(5, 150, 105, 0.4)";
  };

  confirmButton.onmouseout = () => {
    confirmButton.style.backgroundColor = "#059669";
    confirmButton.style.transform = "translateY(0)";
    confirmButton.style.boxShadow = "0 4px 6px rgba(5, 150, 105, 0.3)";
  };

 // En RecepcionCotiz.js - Actualizar el onclick del botón Confirmar

confirmButton.onclick = async () => {
    // Deshabilitar botón
    confirmButton.disabled = true;
    confirmButton.textContent = "⏳ Procesando...";
    confirmButton.style.backgroundColor = "#6b7280";
    confirmButton.style.cursor = "not-allowed";

    try {
        console.log("📋 Iniciando proceso de confirmación y envío...");
        
        // PASO 1: Generar PDF
        console.log("📄 Generando PDF...");
        const pdfResponse = await fetch(
            "/api/reception/generate-pdf/" + quotationData.CostingID,
            {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(quotationData),
            }
        );

        const pdfResult = await pdfResponse.json();

        if (!pdfResult.success) {
            throw new Error(pdfResult.error || "Error al generar PDF");
        }

        console.log("✅ PDF generado exitosamente:", pdfResult.pdf_url);
        console.log("📧 Email enviado:", pdfResult.email_enviado ? "Sí" : "No");
        console.log("👥 Destinatarios:", pdfResult.email_destinatarios || 0);

        // PASO 2: Crear cotización en BD
        console.log("💾 Guardando cotización en base de datos...");
        const quotationResponse = await fetch("/api/reception/create-quotation", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
                ...quotationData,
                pdf_url: pdfResult.pdf_url,
                pdf_version: pdfResult.version
            }),
        });

        const quotationResult = await quotationResponse.json();

        if (quotationResult.success) {
            previewWindow.close();

            // Mostrar mensaje de éxito completo
            let mensajeHTML = `
                ✅ Cotización <strong>${quotationResult.QuotationID}</strong> creada exitosamente.<br>
                📋 Versión: <strong>${quotationResult.Version}</strong><br>
                📄 PDF: <strong>${pdfResult.pdf_ruta_ftp}</strong><br><br>
            `;

            if (pdfResult.email_enviado) {
                mensajeHTML += `✅ <strong>Email enviado correctamente</strong><br>`;
                mensajeHTML += `👥 Destinatarios: <strong>${pdfResult.email_destinatarios}</strong><br>`;
                mensajeHTML += `📧 Mensaje: ${pdfResult.email_mensaje}`;
            } else {
                mensajeHTML += `⚠️ <strong>Advertencia:</strong> ${pdfResult.email_mensaje}<br>`;
                mensajeHTML += `📄 El PDF fue generado y guardado correctamente`;
            }

            if (mainWindow && mainWindow.Swal) {
                mainWindow.Swal.fire({
                    icon: pdfResult.email_enviado ? "success" : "warning",
                    title: "Cotización Procesada",
                    html: mensajeHTML,
                    confirmButtonText: "Aceptar",
                    width: 600
                });
            }
        } else {
            throw new Error(quotationResult.error || "Error desconocido");
        }
    } catch (error) {
        console.error("❌ Error en el proceso:", error);

        // Mostrar error usando SweetAlert2
        if (!previewWindow.Swal) {
            const script = doc.createElement("script");
            script.src = "https://cdn.jsdelivr.net/npm/sweetalert2@11";
            doc.head.appendChild(script);

            script.onload = () => {
                previewWindow.Swal.fire({
                    icon: "error",
                    title: "Error",
                    text: "Error al procesar la cotización: " + error.message,
                    confirmButtonText: "Aceptar",
                });
            };
        } else {
            previewWindow.Swal.fire({
                icon: "error",
                title: "Error",
                text: "Error al procesar la cotización: " + error.message,
                confirmButtonText: "Aceptar",
            });
        }

        // Rehabilitar botón
        confirmButton.disabled = false;
        confirmButton.textContent = "✅ Confirmar y Enviar Cotización";
        confirmButton.style.backgroundColor = "#059669";
        confirmButton.style.cursor = "pointer";
    }
};
  // Botón Cancelar
  const cancelButton = doc.createElement("button");
  cancelButton.textContent = "❌ Cancelar";
  cancelButton.style.cssText = `
    background-color: #dc2626;
    color: white;
    padding: 15px 40px;
    border: none;
    border-radius: 8px;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 6px rgba(220, 38, 38, 0.3);
  `;

  cancelButton.onmouseover = () => {
    cancelButton.style.backgroundColor = "#b91c1c";
    cancelButton.style.transform = "translateY(-2px)";
    cancelButton.style.boxShadow = "0 6px 8px rgba(220, 38, 38, 0.4)";
  };

  cancelButton.onmouseout = () => {
    cancelButton.style.backgroundColor = "#dc2626";
    cancelButton.style.transform = "translateY(0)";
    cancelButton.style.boxShadow = "0 4px 6px rgba(220, 38, 38, 0.3)";
  };

  cancelButton.onclick = () => {
    previewWindow.close();
  };

  // Agregar botones al contenedor
  buttonContainer.appendChild(confirmButton);
  buttonContainer.appendChild(cancelButton);

  // Agregar contenedor al body
  doc.body.appendChild(buttonContainer);

  console.log("Botones fijos agregados a la vista previa");
}

/**
 * Maneja el envío de la cotización.
 * Abre una nueva pestaña con la vista previa completa y botones de confirmación.
 */
async function handleSendQuotation() {
  console.log("Iniciando envío de cotización...");

  const opnumberInput = document.getElementById("opnumber");
  const listPriceInput = document.getElementById("Finance_ListPrice");
  const discountInput = document.getElementById("Finance_Discount");
  const offerPriceInput = document.getElementById("Finance_OfferPrice");
  const priceWithIVAInput = document.getElementById("Finance_PriceWithIVA");
  const ivaSelect = document.getElementById("Finance_IVA");
  const currencySelect = document.getElementById("Finance_CurrencyType");

  // Validaciones
  if (!currencySelect.value || currencySelect.value.trim() === "") {
    Swal.fire({
      icon: "error",
      title: "Moneda no seleccionada",
      text: "Por favor seleccione una moneda antes de enviar.",
      confirmButtonText: "Aceptar",
    });
    return;
  }

  if (!ivaSelect.value || ivaSelect.value.trim() === "") {
    Swal.fire({
      icon: "error",
      title: "Impuesto no seleccionado",
      text: "Por favor seleccione un impuesto antes de enviar.",
      confirmButtonText: "Aceptar",
    });
    return;
  }

  const costingNum = opnumberInput.value.trim();

  if (!costingNum || currentQuotationLines.length === 0 || !currentUnifiedData) {
    Swal.fire({
      icon: "error",
      title: "Datos incompletos",
      text: "Por favor complete todos los campos y previsualice la cotización antes de enviar.",
      confirmButtonText: "Aceptar",
    });
    return;
  }

    // ✅ OBTENER DATOS DEL IMPUESTO SELECCIONADO
  const selectedTaxOption = ivaSelect.options[ivaSelect.selectedIndex];
  const taxCode = ivaSelect.value;
  const taxName = selectedTaxOption.textContent; // Ejemplo: "IVA 16%"
  const taxAmount = parseFloat(selectedTaxOption.dataset.taxAmount) || 0; // 0.16
  const taxPercent = taxAmount * 100; // 16

    // ✅ CALCULAR MONTOS
  const salePrice = unformatCurrency(listPriceInput.value);
  const discountPercent = parseFloat(discountInput?.value) || 0;
  const offerPrice = unformatCurrency(offerPriceInput.value);
  const totalAmount = unformatCurrency(priceWithIVAInput.value);

    // Calcular monto del descuento
  const discountAmount = salePrice - offerPrice;

    // Calcular monto del impuesto
  const taxAmountValue = totalAmount - offerPrice;

    console.log("📊 Datos calculados:");
  console.log(`   Precio Lista: ${salePrice}`);
  console.log(`   Descuento: ${discountPercent}% ($${discountAmount.toFixed(2)})`);
  console.log(`   Precio Oferta: ${offerPrice}`);
  console.log(`   Impuesto: ${taxName} - ${taxPercent}% ($${taxAmountValue.toFixed(2)})`);
  console.log(`   Total: ${totalAmount}`);  

  // Construir objeto de datos
  const quotationData = {
    CostingID: currentUnifiedData.CostingID,
    CaseCost: currentUnifiedData.CaseCost,
    SalePrice: unformatCurrency(listPriceInput.value),
    DiscountPercent: parseFloat(discountInput?.value) || 0,
    OvercostFactor: 1,
    Amount: unformatCurrency(offerPriceInput.value),
    TaxCode: ivaSelect.value,
    TotalAmount: unformatCurrency(priceWithIVAInput.value),
    CurrencyCode: currencySelect.value,

        // ✅ NUEVOS CAMPOS PARA EL EMAIL
    TaxName: taxName,
    TaxPercent: taxPercent,
    TaxAmountValue: taxAmountValue,
    DiscountAmountValue: discountAmount,
    
    // ✅ AGREGAR DATOS DEL CRM PARA EL EMAIL
    CRM_ContactName: currentUnifiedData.CRM_ContactName,
    CRM_ContactType: currentUnifiedData.CRM_ContactType,
    CRM_OpportunityNumber: currentUnifiedData.CRM_OpportunityNumber,
    CRM_ContactEmail: currentUnifiedData.CRM_ContactEmail,
    CRM_ContactNumber: currentUnifiedData.CRM_ContactNumber,
    CRM_ContactCity: currentUnifiedData.CRM_ContactCity,
    CRM_ContactState: currentUnifiedData.CRM_ContactState,
    CRM_ContactCountry: currentUnifiedData.CRM_ContactCountry,
    CRM_AssignedSalesperson: currentUnifiedData.CRM_AssignedSalesperson,
    QuotationLines: currentQuotationLines.map((line) => ({
      QuotationLine: line.CostingLine,
      CostingLineID: `${currentUnifiedData.CostingID}-${line.CostingLine}`,
      PartNum: line.PartNum,
      PartDescription: line.PartDescription,
      Qty: line.Qty,
      UOMCode: line.UOMCode,
      UnitPrice: line.UnitPrice,
      Amount: line.Amount,
    })),
  };

  try {
    // 🆕 PASO 1: Abrir nueva ventana con loading
    const previewWindow = window.open("", "_blank");
    
    if (!previewWindow) {
      Swal.fire({
        icon: "error",
        title: "Bloqueador de ventanas",
        text: "Por favor permite ventanas emergentes para ver la vista previa.",
        confirmButtonText: "Aceptar",
      });
      return;
    }

    previewWindow.document.write(`
      <html>
        <head>
          <title>Cargando vista previa...</title>
          <style>
            body {
              margin: 0;
              padding: 0;
              font-family: Arial, sans-serif;
              display: flex;
              justify-content: center;
              align-items: center;
              height: 100vh;
              background-color: #f5f7fa;
            }
            .loading-container {
              text-align: center;
            }
            .spinner {
              border: 4px solid #f3f3f3;
              border-top: 4px solid #003366;
              border-radius: 50%;
              width: 50px;
              height: 50px;
              animation: spin 1s linear infinite;
              margin: 0 auto 20px;
            }
            @keyframes spin {
              0% { transform: rotate(0deg); }
              100% { transform: rotate(360deg); }
            }
            p {
              color: #333;
              font-size: 16px;
            }
          </style>
        </head>
        <body>
          <div class="loading-container">
            <div class="spinner"></div>
            <p>Cargando vista previa de la cotización...</p>
          </div>
        </body>
      </html>
    `);

    // 🆕 PASO 2: Obtener vista previa del servidor
    const previewResponse = await axios.post(
      "/api/reception/preview-quotation",
      quotationData,
      {
        headers: {
          'Content-Type': 'application/json'
        }
      }
    );

    // 🆕 PASO 3: Escribir el HTML en la nueva ventana
    previewWindow.document.open();
    previewWindow.document.write(previewResponse.data);
    previewWindow.document.close();

    // 🆕 PASO 4: Agregar botones fijos después de que cargue el documento
    previewWindow.addEventListener("load", function () {
      addFixedButtonsToPreview(previewWindow, quotationData);
    });

  } catch (error) {
    console.error("Error al generar vista previa:", error);
    Swal.fire({
      icon: "error",
      title: "Error",
      text: `Error al generar la vista previa: ${error.message}`,
      confirmButtonText: "Aceptar",
    });
  }
}

/**
 * Muestra las líneas de cotización en la tabla.
 */
function displayQuotationLines(quotationLines) {
  const tableBody = document.getElementById("ApprovalAccessDistributorsbody");

  if (!tableBody) {
    console.error("No se encontró el elemento ApprovalAccessDistributorsbody");
    return;
  }

  // 🔥 DESTRUIR DataTables si existe
  if ($.fn.DataTable.isDataTable('#ApprovalAccessDistributorsTable')) {
    $('#ApprovalAccessDistributorsTable').DataTable().destroy();
    console.log("DataTable destruido antes de cargar nuevos datos");
  }

  // Limpiar tabla
  tableBody.innerHTML = "";

  // Agregar filas
  quotationLines.forEach((line, index) => {
    const row = document.createElement("tr");
    row.innerHTML = `
      <td>${line.CostingLine}</td>
      <td>${line.PartNum || "-"}</td>
      <td>${line.PartDescription || "-"}</td>
      <td>${line.Qty}</td>
      <td>${line.UOMCode || "-"}</td>
      <td>${line.UnitPrice ? formatCurrency(line.UnitPrice.toFixed(2)) : "-"}</td>
      <td>${formatCurrency(line.Amount.toFixed(2))}</td>
      <td>
      </td>
    `;
    tableBody.appendChild(row);
  });

  // Calcular y mostrar el total
  const total = quotationLines.reduce((sum, line) => sum + line.Amount, 0);
  console.log(`Total final en tabla: ${formatCurrency(total.toFixed(2))}`);

  // 🔥 REINICIALIZAR DataTables después de cargar datos
  setTimeout(() => {
    table = initializeDataTable();
    console.log("DataTable reinicializado con botones funcionales");
  }, 100);

  // Habilitar el botón de envío después de previsualizar
  const sendButton = document.querySelector(".row-actions.center button");
  if (sendButton) {
    sendButton.disabled = false;
    console.log("Botón de envío habilitado");
  }
}


// Función para inicializar DataTables
function initializeDataTable() {
    // 🔥 Verificar si ya existe una instancia y destruirla
    if ($.fn.DataTable.isDataTable('#ApprovalAccessDistributorsTable')) {
        $('#ApprovalAccessDistributorsTable').DataTable().destroy();
    }

    return $('#ApprovalAccessDistributorsTable').DataTable({
        responsive: true,
        scrollX: true,
        dom: 'Blfrtip', // B=Botones, l=length, f=filter, r=processing, t=table, i=info, p=pagination
        buttons: [
            {
                extend: 'excel',
                text: 'Exportar a Excel',
                className: 'buttons-excel',
                exportOptions: {
                    columns: ':visible:not(:last-child)', // Excluye la última columna (Acciones)
                    modifier: { page: 'all' },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $('#ApprovalAccessDistributorsTable_wrapper .dataTables_scrollHead thead tr:nth-child(1) th');
                            if (headerSelector.length === 0) {
                                headerSelector = $('#ApprovalAccessDistributorsTable thead tr:first-child th');
                            }
                            return headerSelector.eq(columnIdx).text();
                        }
                    }
                }
            },
            {
                extend: 'pdf',
                text: 'Exportar a PDF',
                className: 'buttons-pdf',
                orientation: 'landscape',
                pageSize: 'LEGAL',
                exportOptions: {
                    columns: ':visible:not(:last-child)',
                    modifier: { page: 'all' },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $('#ApprovalAccessDistributorsTable_wrapper .dataTables_scrollHead thead tr:nth-child(1) th');
                            if (headerSelector.length === 0) {
                                headerSelector = $('#ApprovalAccessDistributorsTable thead tr:first-child th');
                            }
                            return headerSelector.eq(columnIdx).text();
                        }
                    }
                },
                customize: function (doc) {
                    doc.content[1].table.widths = ['6%', '15%', '28%', '10%', '9%', '12%', '12%', '8%'];
                    doc.styles.tableHeader.fontSize = 10;
                    doc.styles.tableHeader.fillColor = '#1f71c2';
                    doc.defaultStyle.fontSize = 8;
                }
            }
        ],
        pageLength: 10,
        ordering: true,
        searching: true,
        language: {
            search: "Search:",
            lengthMenu: "Mostrar _MENU_ registros",
            zeroRecords: "No se encontraron resultados",
            info: "Showing _START_ to _END_ of _TOTAL_ entries",
            infoEmpty: "Showing 0 to 0 of 0 entries",
            infoFiltered: "(filtrado de _MAX_ registros totales)",
            paginate: {
                first: "Primero",
                last: "Último",
                next: "Next",
                previous: "Previous"
            }
        },
        lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
        columnDefs: [
            { 
                targets: -1, // Última columna (Acciones)
                orderable: false, 
                searchable: false 
            },
            {
                targets: [0, 3, 4, 5, 6, 7], // Columnas centradas
                className: 'dt-center'
            },
            {
                targets: [1, 2], // Partnum y Descripción alineados a la izquierda
                className: 'dt-left'
            }
        ],
        order: [[0, 'asc']],
        
        // 🔥 EVENTO DESPUÉS DE DIBUJAR LA TABLA
        drawCallback: function() {
            console.log("Tabla dibujada correctamente");
        }
    });
}

// Variable global para la instancia de DataTables
let table;

document.addEventListener("DOMContentLoaded", () => {
    // 🔥 Inicializar DataTables vacío al cargar la página
    table = initializeDataTable();
    console.log("DataTable inicializado en DOMContentLoaded");
    
    // 🔥 NO configurar filtros aquí, se harán después de cargar datos
    
    // Inicializar búsqueda de recepción
    initReceptionSearch();
    toggleNextSections();

    // Agregar listener al botón Previsualizar y Enviar
    setTimeout(() => {
        const allButtons = document.querySelectorAll("button");
        console.log("Buscando botones, total encontrados:", allButtons.length);

        allButtons.forEach((btn) => {
            const buttonText = btn.textContent.trim();
            console.log("Botón encontrado:", buttonText);

            if (buttonText.includes("Previsualizar")) {
                btn.addEventListener("click", handlePreviewClick);
                console.log("Listener agregado a Previsualizar");
            }
            if (buttonText.includes("Enviar Cotización") || buttonText.includes("Enviar")) {
                btn.addEventListener("click", handleSendQuotation);
                console.log("Listener agregado a Enviar Cotización");
            }
        });
    }, 1000);
});

