// Archivo: ScoreDistributorsMexico.js
// Ruta: src\static\js\Ventas\ScoreDistributorsMX\ScoreDistributorsMexico.js
// Lenguaje: JavaScript

// ===============================
// 🔹 CONSTANTES Y VARIABLES
// ===============================
const loadingIndicator = document.getElementById("loading-indicator"); // Indicador de carga
let table; // Variable para almacenar la instancia de DataTable
let selectedDistributorId = null; // ID del distribuidor seleccionado
let selectedDistributorIdForReject = null;

// ===============================
// 🔹 FUNCIONES GENÉRICAS
// ===============================

// Muestra el indicador de carga
function showLoading() {
    if (loadingIndicator) loadingIndicator.style.display = "block";
}

// Oculta el indicador de carga
function hideLoading() {
    if (loadingIndicator) loadingIndicator.style.display = "none";
}

function showGlobalLoading(message = "{{ _('Procesando solicitud...') }}") {
    // Actualizar el mensaje si es necesario
    document.querySelector("#global-loading-overlay .loading-text").textContent = message;
    
    // Mostrar el overlay
    document.getElementById("global-loading-overlay").style.display = "flex";
    
    // Bloquear el scroll
    document.body.style.overflow = "hidden";
}

function hideGlobalLoading() {
    // Ocultar el overlay
    document.getElementById("global-loading-overlay").style.display = "none";
    
    // Restaurar el scroll
    document.body.style.overflow = "auto";
}

// Modificar la función closeModal para resetear el modal al cerrarlo
function closeModal() {
    const modalElement = document.getElementById('EstandarModal');
    const modalInstance = bootstrap.Modal.getInstance(modalElement);
    if (modalInstance) {
        modalInstance.hide();
    }
    resetModal();
}

// Función para cerrar el modal de rechazo
function closeRejectModal() {
    const modalElement = document.getElementById('RejectModal');
    const modalInstance = bootstrap.Modal.getInstance(modalElement);
    if (modalInstance) {
        modalInstance.hide();
    }
    resetRejectModal();
}

// ===============================
// 🔹 FUNCIONES ESPECÍFICAS DATATABLES CON MODAL PERSONALIZADO PARA TEXTO COMPLETO
// ===============================
// Agregar estilos CSS y modal HTML
$(document).ready(function() {    
    // Agregar modal HTML solo una vez
    if (!$('#text-modal').length) {
        $('body').append(`
            <div id="text-modal" class="text-modal-overlay">
                <div class="text-modal-content">
                    <div class="text-modal-header">
                        <h5 class="text-modal-title"> ${TXT_DETAIL}</h5>
                        <button type="button" class="text-modal-close" aria-label="{{ _('Cerrar') }}">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <div class="text-modal-body" id="text-modal-body">
                        <!-- Contenido del texto completo -->
                    </div>
                </div>
            </div>
        `);
    }
});

// Función para renderizar texto con modal personalizado
function renderExpandableText(data, type) {
    if (type === 'display' && data && data.length > 25) {
        // Limpiar el texto completo
        let fullText = String(data).trim();
        
        // Generar ID único para evitar conflictos
        const uniqueId = 'expand_' + Math.random().toString(36).substr(2, 9);
        
        return `
            <div class="expandable-content">
                <span class="short-text" title="{{ _('Hacer clic en [+] para ver texto completo') }}">
                    ${data.substring(0, 25)}...
                </span>
                <span class="expand-icon" 
                      id="${uniqueId}"
                      data-full-text="${fullText.replace(/"/g, '&quot;')}"
                      onclick="showFullText('${uniqueId}')">
                    [+]
                </span>
            </div>
        `;
    }
    return data || '';
}

// Función para mostrar el texto completo en modal
function showFullText(elementId) {
    const element = document.getElementById(elementId);
    if (!element) return;
    
    const fullText = element.getAttribute('data-full-text');
    if (!fullText) return;
    
    // Mostrar el texto en el modal
    document.getElementById('text-modal-body').textContent = fullText;
    document.getElementById('text-modal').style.display = 'block';
    
    // Prevenir scroll del body
    document.body.style.overflow = 'hidden';
}

// Función para cerrar el modal
function closeTextModal() {
    document.getElementById('text-modal').style.display = 'none';
    document.body.style.overflow = 'auto';
}

// Event listeners para el modal
$(document).ready(function() {
    // Cerrar modal al hacer clic en el botón cerrar
    $(document).on('click', '.text-modal-close', function() {
        closeTextModal();
    });
    
    // Cerrar modal al hacer clic en el overlay
    $(document).on('click', '.text-modal-overlay', function(e) {
        if (e.target === this) {
            closeTextModal();
        }
    });
    
    // Cerrar modal con ESC
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape' && $('#text-modal').is(':visible')) {
            closeTextModal();
        }
    });
});

// ================================================
// FUNCIONALIDAD PARA ALTERNAR ENTRE TEXTO CON TOOLTIP EN COMPUTADORA
// ================================================
function toggleText(uniqueId) {
    const textoCorto = document.getElementById(uniqueId + '_corto');
    const textoCompleto = document.getElementById(uniqueId + '_completo');
    const boton = textoCorto.parentElement.querySelector('.ver-mas');
    
    // Verificar si el texto completo está visible
    const textoCompletoVisible = textoCompleto.style.display !== 'none' && textoCompleto.style.display !== '';
    
    if (!textoCompletoVisible) {
        // Mostrar texto completo
        textoCorto.style.display = 'none';
        textoCompleto.style.display = 'inline';
        boton.innerHTML = '[-]';
        boton.title = '{{ _("Mostrar menos") }}';
    } else {
        // Mostrar texto corto
        textoCorto.style.display = 'inline';
        textoCompleto.style.display = 'none';
        boton.innerHTML = '[+]';
        boton.title = '{{ _("Mostrar más") }}';
    }
}

// ================================================
// INICIALIZACION DE DATATABLE CON CONFIGURACIONES ESPECÍFICAS
// ================================================
function initializeDataTable() {
    return $('#ClienteScoreTable').DataTable({
        responsive: true,
        scrollX: true,
        scrollCollapse: true,
        dom: 'Bfrtip',
        buttons: [
            {
                extend: 'excel',
                text: `${TXT_BUTTON}`, 
                className: 'btn btn-success',
                exportOptions: {
                    columns: ':visible',
                    modifier: { page: 'all' },
                    format: {
                        header: function (data, columnIdx) {
                            let headerSelector = $('#ClienteScoreTable_wrapper .dataTables_scrollHead thead tr:nth-child(1) th');
                            if (headerSelector.length === 0) {
                                headerSelector = $('#ClienteScoreTable thead th');
                            }
                            return headerSelector.eq(columnIdx).text();
                        }
                    }
                }
            }
        ],
        initComplete: function() {
            // Aplicar búsqueda por columna
            this.api().columns().every(function() {
                var column = this;
                var header = $(column.header());
                
                // Solo agregar filtros a columnas con inputs
                if (header.find('input').length) {
                    var input = header.find('input');
                    
                    input.on('keyup change', function() {
                        if (column.search() !== this.value) {
                            column.search(this.value).draw();
                        }
                    });
                    
                    // Evitar que los clicks en el input propaguen al ordenamiento
                    input.on('click', function(e) {
                        e.stopPropagation();
                    });
                }
            });
        },
        pageLength: 20,
        ordering: false,
        searching: true,
        lengthMenu: [[10, 25, 50, -1], [10, 25, 50, "{{ _('Todos') }}"]],
        columnDefs: [
            { 
                targets: '_all',
                render: function(data, type, row) {
                    // 1. Primero verificar si es un número
                    if (!isNaN(parseFloat(data)) && isFinite(data)) {
                        if (type === 'display') {
                            // Mostrar el número sin decimales (.0) pero mantener decimales si existen (.5)
                            const num = parseFloat(data);
                            return num % 1 === 0 ? num.toString() : data.toString();
                        }
                        return parseFloat(data); // Para ordenamiento/filtrado
                    }
                    
                    // 2. Luego verificar si es una fecha
                    if (typeof data === 'string' && data.match(/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}(\.\d+)?$/)) {
                        if (type === 'display') {
                            const date = new Date(data);
                            return date.toLocaleDateString('es-ES', {
                                day: '2-digit',
                                month: '2-digit',
                                year: 'numeric'
                            });
                        }
                        return data;
                    }
                    
                    // 3. Finalmente usar el renderizado de texto
                    return renderExpandableText(data, type);
                }
            },
            { 
                targets: -1, 
                orderable: false, 
                searchable: false 
            }
        ]
    });
}

//-------------------- CARGA DE LOS DATOS--------------------
function loadAlternators() {
    showLoading();
    
    // Ejemplos completos de alternadores con todos los campos
    //const exampleData = [
    //    {
    //        Ordernumandline: "208/120 V",
    //        OrderNum: "Standby",
    //        Capacidad: "125",
    //        Voltaje: "",
    //        Tipo: "",
    //        Caseta: "",
    //        JobNum2: "",
    //        FechaPlaneación: "",
    //        FechaProducción: "",
    //        FechaMat: "",
    //        AvanceDeSurtimiento: "",
    //        AvisoDeTerminacion: "",
    //        MaterialFaltante: "",
    //        PartNum_M: "",
    //        Description_M: "",
    //        RefCategory_M: "",
    //        Preasignado_M: "",
    //        NoSerie_M: "",
    //        Faltante_M: "",
    //        En_PO_Altern_M: "",
    //        Estatus_M: "",
    //        Fecha_Llegada_M: "",
    //        PartNum_G: "",
    //        Description_G: "",
    //        PartClass_G: "",
    //        Preasignado_G: "",
    //        NoSerie_G: "",
    //        Faltante_G: "",
    //        En_PO_Altern_G: "",
    //        Estatus_G: "",
    //        Fecha_Llegada_G: "",
    //    },        
    //];

    fetch('/Ventas/VentasDist/ScoreDistributorsMXSQL', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({})
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Error en la respuesta de la API');
        }
        return response.json();
    })
    .then(responseData => {
        // Verifica si la respuesta tiene la propiedad 'success' y si es verdadera
        if (!responseData.success) {
            throw new Error(responseData.message || 'Error en la respuesta de la API');
        }

        // Accede a la propiedad 'data' de la respuesta
        const alternators = responseData.data;
        console.log("Datos recibidos:", alternators);

        const tableBody = document.getElementById("ClienteScorebody");
        tableBody.innerHTML = "";

        alternators.forEach(alternator => {
            const row = document.createElement("tr");
            row.innerHTML = `
                <td>${alternator.OrderNum || ''}</td>
                <td>${alternator.Tipo || ''}</td>
                <td>${alternator.Departamento || ''}</td>
                <td>${alternator.Vendedor || ''}</td>
                <td>${alternator.Name || ''}</td>
                <td>${alternator.Capacidad || ''}</td>
                <td>${alternator.OrderDate || ''}</td>
                <td>${alternator.NeedByDate || ''}</td>
                <td>${alternator.FechaPlaneación || ''}</td>
                <td>${alternator.FechaVentas || ''}</td>
                <td>${alternator.Faltante_M || ''}</td>
                <td>${alternator.Estatus_M || ''}</td>
                <td>${alternator.Fecha_Llegada_M || ''}</td>
                <td>${alternator.Faltante_G || ''}</td>
                <td>${alternator.Estatus_G || ''}</td>
                <td>${alternator.Fecha_Llegada_G || ''}</td>
            `;
            tableBody.appendChild(row);
        });

        if (table) table.destroy();
        table = initializeDataTable();
    })
    .catch(error => {
        console.error("Error al cargar alternadores:", error);
        alert("Hubo un problema al cargar los alternadores. Intenta nuevamente.");
    })
    .finally(() => hideLoading());

}


// Función para editar alternador (similar a approve() pero para alternadores)
function editAlternator(alternatorId) {
    selectedAlternatorId = alternatorId;
    
    // EJEMPLO: Datos del alternador seleccionado
    const alternatorData = {
        ID: alternatorId,
        Comments: alternatorId === "ALT-001" 
            ? "{{ _('Alternador para uso general') }}" 
            : "{{ _('Alternador de alta potencia') }}"
    };
    
    // Rellenar el modal (usando los mismos campos del original)
    document.getElementById('Comments').value = alternatorData.Comments;
    
    // Configurar select técnico (reutilizando sellerSelect)
    const techSelect = document.getElementById('sellerSelect');
    techSelect.innerHTML = `
        <option selected disabled value="">{{ _('Seleccione técnico') }}</option>
        <option value="TEC-01">{{ _('Técnico Eléctrico') }}</option>
        <option value="TEC-02">{{ _('Técnico Mecánico') }}</option>
    `;
    
    // Configurar valor técnico (reutilizando discountInput)
    document.getElementById('discountInput').value = "15";
    
    // Mostrar modal (el mismo que en el original)
    const modal = new bootstrap.Modal(document.getElementById('EstandarModal'));
    modal.show();
}

// Función para guardar cambios (similar a assignSeller())
function saveAlternatorChanges() {
    // Validación básica
    const techSelect = document.getElementById('sellerSelect');
    const techValue = document.getElementById('discountInput').value;
    
    if (!techSelect.value || isNaN(parseFloat(techValue))) {
        alert("{{ _('Complete todos los campos requeridos') }}");
        return;
    }
    
    // EJEMPLO: Mostrar datos que se enviarían
    const dataToSave = {
        alternatorId: selectedAlternatorId,
        technician: techSelect.value,
        techValue: parseFloat(techValue),
        comments: document.getElementById('Comments').value
    };
    
    console.log("{{ _('Datos a guardar:') }}", dataToSave);
    
    alert("{{ _('Cambios guardados correctamente (ejemplo)') }}");
    closeModal();
    loadAlternators();
}

// Inicialización
document.addEventListener("DOMContentLoaded", () => {
    loadAlternators();
    
    // Configurar botón de guardar
    //document.querySelector('#EstandarModal .btn-primary').onclick = saveAlternatorChanges;
});

// Abre el modal para aprobar un distribuidor y cargar los vendedores
function approve(RequestID) {
    selectedDistributorId = RequestID;
    let modal = new bootstrap.Modal(document.getElementById('EstandarModal'));
    modal.show();
    loadSellers();
    loadDistributorComments(RequestID);
}

// Función para abrir el modal de rechazo y cargar los datos del distribuidor
function reject(requestId) {
    // Guardar el ID del distribuidor seleccionado para rechazo
    selectedDistributorIdForReject = requestId;
    
    // Mostrar indicador de carga
    showLoading();
    
    // Cargar los comentarios del distribuidor
    loadDistributorComments(requestId)
        .then(data => {
            // Asignar los comentarios del distribuidor al campo correspondiente
            document.getElementById('Comments2').value = data.Comments || '';
            
            // Mostrar el modal de rechazo
            const rejectModal = new bootstrap.Modal(document.getElementById('RejectModal'));
            rejectModal.show();
        })
        .catch(error => {
            console.error("{{ _('Error al cargar comentarios:') }}", error);
            Swal.fire({
                icon: 'error',
                title: '{{ _("Error") }}',
                text: '{{ _("No se pudieron cargar los comentarios del distribuidor.") }}',
                confirmButtonColor: '#d33',
                confirmButtonText: '{{ _("Aceptar") }}'
            });
        })
        .finally(() => {
            hideLoading();
        });
}

// Carga la lista de vendedores desde el servidor y los muestra en el modal
function loadSellers() {
    // Muestra el indicador de carga si existe
    const loadingIndicator = document.getElementById('loading-indicator');
    if (loadingIndicator) {
        loadingIndicator.style.display = 'block';
    }
    
    // Obtiene el elemento select
    const sellerSelect = document.getElementById('sellerSelect');
    
    // Limpia las opciones existentes
    sellerSelect.innerHTML = '';
    
    // Agrega una opción por defecto
    const defaultOption = document.createElement('option');
    defaultOption.value = '';
    defaultOption.textContent = '{{ _("Seleccione un vendedor") }}';
    defaultOption.selected = true;
    defaultOption.disabled = true;
    sellerSelect.appendChild(defaultOption);
    
    // Hace la petición a la API
    fetch('/Ventas_Module/VentasEUA/AccessDistributors/get_sellers')
        .then(response => {
            if (!response.ok) {
                throw new Error('{{ _("Error en la respuesta del servidor:") }} ' + response.status);
            }
            return response.json();
        })
        .then(data => {
            // Para cada vendedor recibido, crea una opción en el select
            data.forEach(seller => {
                const option = document.createElement('option');
                option.value = seller.UserID;
                option.textContent = seller.Seller;
                sellerSelect.appendChild(option);
            });
        })
        .catch(error => {
            console.error('{{ _("Error al cargar los vendedores:") }}', error);
            // Muestra un mensaje de error al usuario
            alert('{{ _("Error al cargar la lista de vendedores. Por favor, intente nuevamente.") }}');
        })
        .finally(() => {
            // Oculta el indicador de carga
            if (loadingIndicator) {
                loadingIndicator.style.display = 'none';
            }
        });
}

// Función para cargar los comentarios del distribuidor
function loadDistributorComments(RequestID) {
    // Mostrar el indicador de carga
    showLoading();
    
    // Crear una promesa para permitir un uso más flexible
    return new Promise((resolve, reject) => {
        // Hacer una petición para obtener los comentarios específicos del distribuidor
        fetch(`/Ventas_Module/VentasEUA/AccessDistributors/get_comments/${RequestID}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error('{{ _("Error al obtener los comentarios") }}');
                }
                return response.json();
            })
            .then(data => {
                // Buscar el elemento para el modal de aprobación
                const approveCommentsElement = document.getElementById('Comments');
                
                // Si el elemento existe, mostrar los comentarios
                if (approveCommentsElement) {
                    approveCommentsElement.value = data.Comments || '';
                }
                
                // Buscar el elemento para el modal de rechazo
                const rejectCommentsElement = document.getElementById('Comments2');
                
                // Si el elemento existe, mostrar los comentarios
                if (rejectCommentsElement) {
                    rejectCommentsElement.value = data.Comments || '';
                }
                
                // Resolver la promesa con los datos
                resolve(data);
            })
            .catch(error => {
                console.error("{{ _('Error al cargar los comentarios del distribuidor:') }}", error);
                reject(error);
            })
            .finally(() => hideLoading());
    });
}

// Valida los campos requeridos en el modal antes de enviar el formulario
function validateForm() {
    let isValid = true;

    // Validar selección del vendedor
    let sellerSelect = document.getElementById("sellerSelect");
    if (!sellerSelect.value) {
        sellerSelect.classList.add("invalid-input");
        isValid = false;
    } else {
        sellerSelect.classList.remove("invalid-input");
    }

    // Validar el descuento
    let discountInput = document.getElementById("discountInput");
    if (discountInput.value.trim() === "" || isNaN(parseFloat(discountInput.value))) {
        discountInput.classList.add("invalid-input");
        isValid = false;
    } else {
        discountInput.classList.remove("invalid-input");
    }

    return isValid;
}

// Remueve el estado de error al corregir el input en tiempo real
//document.getElementById("sellerSelect").addEventListener("change", function () {
//    this.classList.remove("invalid-input");
//});

//document.getElementById("discountInput").addEventListener("input", function () {
//    this.classList.remove("invalid-input");
//});

// Aprueba el distribuidor, crea su cuenta de usuario y le asigna un vendedor
function assignSeller() {
    if (!validateForm()) {
        Swal.fire({
            icon: 'error',
            title: '{{ _("Campos obligatorios") }}',
            text: '{{ _("Por favor, complete todos los campos obligatorios.") }}',
            confirmButtonColor: '#d33',
            confirmButtonText: '{{ _("Aceptar") }}'
        });
        return;
    }

    // Confirmar la acción antes de proceder
    Swal.fire({
        title: '{{ _("¿Confirmar aprobación?") }}',
        text: "{{ _('Esta acción aprobará al distribuidor y creará su cuenta de usuario. ¿Desea continuar?') }}",
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#3085d6',
        cancelButtonColor: '#d33',
        confirmButtonText: '{{ _("Sí, aprobar") }}',
        cancelButtonText: '{{ _("Cancelar") }}'
    }).then((result) => {
        if (result.isConfirmed) {
            // Mostrar indicador de carga global con mensaje específico
            showGlobalLoading("{{ _('Procesando aprobación de distribuidor...') }}");
            
            let sellerId = document.getElementById("sellerSelect").value;
            let discount = document.getElementById("discountInput").value;

            fetch('/Ventas_Module/VentasEUA/AccessDistributors/assign_seller', {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    distributorId: selectedDistributorId,
                    sellerId: sellerId,
                    discount: parseFloat(discount)
                })
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(data => {
                        throw new Error(data.error || '{{ _("Error al procesar la solicitud") }}');
                    });
                }
                return response.json();
            })
            .then(data => {
                // Ocultar indicador de carga global ANTES de mostrar el mensaje de Swal
                hideGlobalLoading();
                
                Swal.fire({
                    icon: 'success',
                    title: '{{ _("Distribuidor aprobado") }}',
                    text: '{{ _("El distribuidor ha sido aprobado correctamente. Se ha creado su cuenta de usuario y se le ha asignado un vendedor.") }}',
                    confirmButtonColor: '#3085d6',
                    confirmButtonText: '{{ _("Aceptar") }}'
                }).then(() => {
                    closeModal();
                    window.location.reload();
                });
            })
            .catch(error => {
                // Ocultar indicador de carga global ANTES de mostrar el error
                hideGlobalLoading();
                
                console.error("{{ _('Error al aprobar distribuidor:') }}", error);
                Swal.fire({
                    icon: 'error',
                    title: '{{ _("Error") }}',
                    text: error.message || '{{ _("Hubo un problema al aprobar el distribuidor. Inténtelo de nuevo.") }}',
                    confirmButtonColor: '#d33',
                    confirmButtonText: '{{ _("Aceptar") }}'
                });
            });
        }
    });
}

// Función para confirmar el rechazo del distribuidor
function confirmReject() {
    // Verificar que tengamos un ID de distribuidor seleccionado
    if (!selectedDistributorIdForReject) {
        Swal.fire({
            icon: 'error',
            title: '{{ _("Error") }}',
            text: '{{ _("No se ha seleccionado un distribuidor para rechazar.") }}',
            confirmButtonColor: '#d33',
            confirmButtonText: '{{ _("Aceptar") }}'
        });
        return;
    }
    
    // Obtener el comentario de rechazo (opcional)
    const comentario = document.getElementById('Commentsresp').value.trim();
    
    // Mostrar confirmación antes de proceder
    Swal.fire({
        title: '{{ _("¿Confirmar rechazo?") }}',
        text: "{{ _('Esta acción rechazará la solicitud del distribuidor. No se podrá deshacer.') }}",
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: '{{ _("Sí, rechazar") }}',
        cancelButtonText: '{{ _("Cancelar") }}'
    }).then((result) => {
        if (result.isConfirmed) {
            // Mostrar indicador de carga global con mensaje específico
            showGlobalLoading("{{ _('Procesando rechazo de distribuidor...') }}");
            
            // Enviar solicitud al servidor
            fetch('/Ventas_Module/VentasEUA/AccessDistributors/reject', {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    distributorId: selectedDistributorIdForReject,
                    comentario: comentario || null // Enviar null si no hay comentario
                })
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(data => {
                        throw new Error(data.error || '{{ _("Error al procesar la solicitud") }}');
                    });
                }
                return response.json();
            })
            .then(data => {
                // Ocultar indicador de carga global ANTES de mostrar el mensaje de Swal
                hideGlobalLoading();
                
                // Cerrar el modal
                closeRejectModal();
                
                if (data.error_mail) {
                    // El rechazo se completó pero hubo un problema con el correo
                    Swal.fire({
                        icon: 'warning',
                        title: '{{ _("Solicitud rechazada") }}',
                        text: '{{ _("La solicitud ha sido rechazada, pero hubo un problema al enviar la notificación por correo electrónico.") }}',
                        confirmButtonColor: '#3085d6',
                        confirmButtonText: '{{ _("Aceptar") }}'
                    }).then(() => {
                        window.location.reload();
                    });
                } else {
                    // Todo se completó correctamente
                    Swal.fire({
                        icon: 'success',
                        title: '{{ _("Solicitud rechazada") }}',
                        text: '{{ _("La solicitud del distribuidor ha sido rechazada y se ha enviado una notificación.") }}',
                        confirmButtonColor: '#3085d6',
                        confirmButtonText: '{{ _("Aceptar") }}'
                    }).then(() => {
                        window.location.reload();
                    });
                }
            })
            .catch(error => {
                // Ocultar indicador de carga global ANTES de mostrar el error
                hideGlobalLoading();
                
                console.error("{{ _('Error al rechazar distribuidor:') }}", error);
                Swal.fire({
                    icon: 'error',
                    title: '{{ _("Error") }}',
                    text: error.message || '{{ _("Hubo un problema al rechazar la solicitud. Inténtelo de nuevo.") }}',
                    confirmButtonColor: '#d33',
                    confirmButtonText: '{{ _("Aceptar") }}'
                });
            });
        }
    });
}

// Resetea los campos y estilos del modal cuando se cierra
function resetModal() {
    let sellerSelect = document.getElementById("sellerSelect");
    let discountInput = document.getElementById("discountInput");
    let commentsElement = document.getElementById("Comments");

    // Restablece el select
    sellerSelect.innerHTML = `<option selected disabled value="">{{ _("Seleccione una opción") }}</option>`;

    // Restablece el input de descuento
    discountInput.value = "0.00";

    // Limpiar el campo de comentarios
    if (commentsElement) {
        commentsElement.value = "";
    }

    // Elimina posibles estilos de error
    sellerSelect.classList.remove("invalid-input");
    discountInput.classList.remove("invalid-input");

    // Resetea la variable de RequestID seleccionado
    selectedDistributorId = null;
}

// Función para limpiar el modal de rechazo
function resetRejectModal() {
    // Limpiar los campos
    document.getElementById('Comments2').value = '';
    document.getElementById('Commentsresp').value = '';
    
    // Resetear el ID del distribuidor seleccionado
    selectedDistributorIdForReject = null;
}

// ===============================
// 🔹 EVENT LISTENERS
// ===============================

// Carga los distribuidores al cargar la página y agrega validaciones al campo de descuento
document.addEventListener("DOMContentLoaded", () => {
    //loadDistributors();
    
    let discountInput = document.getElementById("discountInput");
    if (discountInput) {
        discountInput.addEventListener("blur", validateDiscount);
    }
});
//document.getElementById('EstandarModal').addEventListener('hidden.bs.modal', resetModal);

// Inicializar eventos para el modal de rechazo cuando se carga el documento
document.addEventListener("DOMContentLoaded", () => {
    // Asignar evento al botón de confirmación de rechazo
    const confirmRejectBtn = document.getElementById('confirmRejectBtn');
    if (confirmRejectBtn) {
        confirmRejectBtn.addEventListener('click', confirmReject);
    }
    
    // Evento para limpiar el modal cuando se cierra
    //document.getElementById('RejectModal').addEventListener('hidden.bs.modal', resetRejectModal);
});