// Optimized JavaScript Code
/**
 * DMBreakersOptions_catalog.js - Gestión de catálogo de controles
 * Ruta: src\static\js\Ventas\VentasEUA\DataMaster\DMBreakersOptions_catalog.js
 */

function agregarEstilosOverlay() {
    // Verificar si ya existe el estilo
    if (!document.getElementById('overlay-styles')) {
        const style = document.createElement('style');
        style.id = 'overlay-styles';
        style.innerHTML = `
            .page-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
                flex-direction: column;
            }
            .spinner-container {
                text-align: center;
                color: white;
            }
            .spinner-container .spinner-border {
                width: 3rem;
                height: 3rem;
            }
            .spinner-text {
                margin-top: 15px;
                font-size: 18px;
                font-weight: bold;
            }
        `;
        document.head.appendChild(style);
    }
}

function mostrarSpinner(mensaje = "Actualizando fichas técnicas...") {
    agregarEstilosOverlay();
    
    // Crear el overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-overlay';
    overlay.id = 'actualizacion-overlay';
    
    // Crear el spinner y el mensaje
    overlay.innerHTML = `
        <div class="spinner-container">
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <div class="spinner-text">${mensaje}</div>
        </div>
    `;
    
    // Agregar a la página
    document.body.appendChild(overlay);
    
    // Bloquear el scroll
    document.body.style.overflow = 'hidden';
}

function ocultarSpinner() {
    const overlay = document.getElementById('actualizacion-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = 'auto';
    }
}
// =====================================
// Configuración y Variables Globales
// =====================================
const CONFIG = {
    API_URL: '/ventas/data_master/BreakersOptions/listBreakersOptions',
    TABLA_ID: 'BreakersOptionsTable',
    MONEDA: {
        estilo: 'currency',
        moneda: 'MXN',
        locale: 'es-MX'
    },
    DATATABLE: {
        columnDefs: [
            { targets: 0, width: "15%", className: 'text-center' },
            { targets: 1, width: "35%", className: 'text-left' },
            { targets: [2], width: "15%", className: 'text-center' }
        ]
    }
};

// =====================================
// Gestión del Indicador de Carga
// =====================================
const LoadingManager = {
    element: document.getElementById("loading-indicator"),
    
    show() {
        if (this.element) this.element.style.display = "block";
    },
    
    hide() {
        if (this.element) this.element.style.display = "none";
    }
};

// =====================================
//  Limpiar items
// =====================================
function limpiaritems() {
    document.getElementById("Item").value = "";
    document.getElementById("Description").value = "";
    document.getElementById("TechnicalSheet").value = "";
}

// =====================================
// Actualizar Fichas Tecnicas
// =====================================
async function actualizarfichastecnicas() {
    // Mostrar el spinner antes de iniciar la petición
    mostrarSpinner();
    try {
        const response = await fetch('/ventas/data_master/BreakersOptions/actualizarBreakersOptions', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        });

        // Agregar esto para depurar
        const responseData = await response.json();
        console.log("Respuesta del servidor:", responseData);
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
        if (response.ok) {
            Swal.fire({
                title: "Actualizado!",
                text: "Los archivos se han actualizado exitosamente.",
                icon: "success"
            }).then(() => {
                location.reload(); // Recargar la página para actualizar la tabla
            });
        }
    } catch (error) {
        // Asegurarse de que el spinner se oculte incluso si hay un error
        ocultarSpinner();
        console.error("Error", error);
        Swal.fire({
            title: "Error!",
            text: "Ocurrió un error.",
            icon: "error"
        });
    }
}

// =====================================
// Gestión de la Tabla de BreakersOptions
// =====================================
class Breakers_Options {
    constructor() {
        this.table = null;
        this.initializeTable();
    }

    async initializeTable() {
        LoadingManager.show();
        try {
            const data = await this.fetchBreakersData();
            this.renderTableData(data);
            this.initializeDataTable();
            this.setupEventListeners();
        } catch (error) {
            console.error("Error al inicializar la tabla:", error);
        } finally {
            LoadingManager.hide();
        }
    }

    async fetchBreakersData() {
        try {
            const response = await fetch(CONFIG.API_URL);
            const data = await response.json();
            console.log("Datos obtenidos:", data); // Verificar los datos recibidos
            return data;
        } catch (error) {
            console.error("Error al obtener los datos:", error);
            return [];
        }
    }

    renderTableData(data) {
        const tableBody = document.getElementById("TableBody");
        if (!tableBody) {
            console.error("No se encontró el elemento TableBody");
            return;
        }
        tableBody.innerHTML = data.map(breaker => this.createTableRow(breaker)).join('');
    }

    createTableRow(breaker) {
        // Verificación de datos para depuración
        if (!breaker) {
            console.error("Se recibió un registro nulo o indefinido");
            return "";
        }
        
        console.log("Procesando registro:", breaker);
        
        // Accede a las propiedades con los nombres correctos según el SQL
        let longitud = breaker.TechnicalSheet ? breaker.TechnicalSheet.length : 0;
        let icono;
    
        if (longitud > 0) {
            //icono = `<a href="/ventas/pdf_viewerBreakers_Options/view/${encodeURIComponent(breaker.TechnicalSheet)}" target="_blank"><i class="fas fa-file-pdf fa-2x" style="color:red; cursor: pointer;"></i></a>`;
            icono = `<a href="${breaker.TechnicalSheet}" target="_blank"><i class="fas fa-file-pdf fa-2x" style="color:red; cursor: pointer;"></i></a>`;
        } else {
            icono = '<i class="fas fa-file-pdf-o"></i>';
        }
        
        return `
            <tr>
                <td>${breaker.Item_breaker_options}</td>
                <td>${breaker.description}</td>
                <td>${icono}</td>
                
            </tr>
        `;
    }

    initializeDataTable() {
        if ($.fn.DataTable.isDataTable("#" + CONFIG.TABLA_ID)) {
            $('#' + CONFIG.TABLA_ID).DataTable().destroy();
        }

        this.table = $('#' + CONFIG.TABLA_ID).DataTable({
            dom: '<"d-flex justify-content-between align-items-center mb-2" Bf>rtip',
            buttons: [{
                extend: 'excel',
                text: '<i class="fas fa-file-excel"></i> Excel',
                className: 'btn btn-success dm-btn-xs dm-custom-excel-button'
            }],
            scrollX: true,
            autoWidth: true,
            ordering: false,
            fixedHeader: true,
            searching: true,
            columnDefs: [
                { targets: 0, width: "15%" },
                { targets: 1, width: "40%" },
                { targets: 2, width: "15%" },
                {
                    targets: 0,
                    className: 'text-center',
                },
                {
                    targets: 1,
                    className: 'text-left',
                },
               
                {
                    targets: 2,
                    className: 'text-center',
                }
            ],
            language: {
                search: "Buscar:",
                lengthMenu: "Mostrar _MENU_ registros por página",
                info: "Mostrando _START_ a _END_ de _TOTAL_ registros",
                infoEmpty: "Mostrando 0 a 0 de 0 registros",
                paginate: {
                    first: "Primero",
                    last: "Último",
                    next: "Siguiente",
                    previous: "Anterior"
                }
            }
        });
    }

    setupEventListeners() {
        // Filtros de columna
        $('.column-filter').on('keyup change', (e) => {
            const columnIndex = $(e.target).data('column');
            this.table.column(columnIndex).search(e.target.value).draw();
        });

        // Edición de registros
        //$('#' + CONFIG.TABLA_ID).on('click', '.editarBreakers_Options', this.handleEdit);
        $('#' + CONFIG.TABLA_ID).on('click', '.editarBreakers_OptionsCatalog', (e) => {
            const row = e.target.closest('tr');
            if (!row) return;
        
            const cells = row.cells;
            const item = cells[0].textContent.trim();
            const description = cells[1].textContent.trim();
            const price = cells[2].textContent.trim().replace(/[^\d.-]/g, '');
        
            document.getElementById("ItemEdit").value = item;
            document.getElementById("DescriptionEdit").value = description;
            document.getElementById("PriceEdit").value = price;
        });
    }


   
}

// =====================================
// Gestión de Modales
// =====================================
class ModalManager {
    static cerrarModalEditar() {
        const modal = document.getElementById("editarBreakers_OptionsCatalog");
        const modalInstance = bootstrap.Modal.getInstance(modal);
        
        if (modalInstance) {
            modalInstance.hide();
        }

        setTimeout(() => {
            document.querySelectorAll(".modal-backdrop").forEach(backdrop => backdrop.remove());
            document.body.classList.remove("modal-open");
            document.body.style.overflow = "auto";
        }, 300);
    }

    static validarFormulario(event) {
        const campos = [
            { id: "Item", mensaje: "El campo Item es obligatorio." },
            { id: "Description", mensaje: "El campo Descripción es obligatorio." },
            { id: "Price", mensaje: "El campo Precio es obligatorio." }
        ];

        let formularioValido = true;

        campos.forEach(campo => {
            const input = document.getElementById(campo.id);
            const valor = input.value.trim();
            let mensajeError = document.getElementById(`${campo.id}-error`);

            if (!mensajeError) {
                mensajeError = document.createElement("small");
                mensajeError.id = `${campo.id}-error`;
                mensajeError.style.color = "red";
                input.parentNode.appendChild(mensajeError);
            }

            if (!valor || valor === campo.invalidValue) {
                mensajeError.textContent = campo.mensaje;
                formularioValido = false;
            } else {
                mensajeError.textContent = "";
            }
        });

        if (!formularioValido) {
            event.preventDefault();
        }
    }
}

// =====================================
// Cerrar modal editar (función global para acceder desde el HTML)
// =====================================
function cerrarModalEditar() {
    ModalManager.cerrarModalEditar();
}

// =====================================
// Inicialización cuando el DOM está listo
// =====================================
document.addEventListener("DOMContentLoaded", () => {
    // Inicializar la tabla
    const breakersTableInstance = new Breakers_Options();
    
  
    
   

    // Agregar listener para el formulario de editar
    const formEditarBreakers_Options = document.getElementById("formEditarBreakersOptions");
    if (formEditarBreakers_Options) {
        formEditarBreakers_Options.addEventListener("submit", async function(event) {
            event.preventDefault();

            const item = document.getElementById("ItemEdit").value.trim();
            const description = document.getElementById("DescriptionEdit").value.trim();
            const price = document.getElementById("PriceEdit").value.trim();

            const data = {
                item: item,
                description: description,
                price: price
            };

            try {
                const response = await fetch('/ventas/data_master/BreakersOptions/updateBreakersOptions', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });

                if (response.ok) {
                    Swal.fire({
                        title: "Actualizado!",
                        text: "El Catálogo de opciones de interruptores ha sido actualizado exitosamente.",
                        icon: "success"
                    }).then(() => {
                        location.reload(); // Recargar la página para actualizar la tabla
                    });
                } else {
                    const errorData = await response.json();
                    Swal.fire({
                        title: "Error!",
                        text: errorData.error,
                        icon: "error"
                    });
                }
            } catch (error) {
                console.error("Error al actualizar :", error);
                Swal.fire({
                    title: "Error!",
                    text: "Ocurrió un error al actualizar .",
                    icon: "error"
                });
            }
        });
    }

    // Corregir el atributo form en el botón de actualizar
    const formEditarBtn = document.querySelector("button[form='formEditarBreakers_Options']");
    if (formEditarBtn) {
        formEditarBtn.setAttribute("form", "formEditarBreakers_Options");
    }

    // Limpieza de modal
    const modalElement = document.getElementById("editarBreakers_OptionsCatalog");
    if (modalElement) {
        modalElement.addEventListener("hidden.bs.modal", ModalManager.cerrarModalEditar);
    }
});