// Optimized JavaScript Code
/**
 * DMbreakers.js - Gestión de catálogo de breakers
 * Ruta: src\static\js\Ventas\VentasEUA\DataMaster\DMbreakers.js
 */

function agregarEstilosOverlay() {
    // Verificar si ya existe el estilo
    if (!document.getElementById('overlay-styles')) {
        const style = document.createElement('style');
        style.id = 'overlay-styles';
        style.innerHTML = `
            .page-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
                flex-direction: column;
            }
            .spinner-container {
                text-align: center;
                color: white;
            }
            .spinner-container .spinner-border {
                width: 3rem;
                height: 3rem;
            }
            .spinner-text {
                margin-top: 15px;
                font-size: 18px;
                font-weight: bold;
            }
        `;
        document.head.appendChild(style);
    }
}

function mostrarSpinner(mensaje = "Actualizando fichas técnicas...") {
    agregarEstilosOverlay();
    
    // Crear el overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-overlay';
    overlay.id = 'actualizacion-overlay';
    
    // Crear el spinner y el mensaje
    overlay.innerHTML = `
        <div class="spinner-container">
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <div class="spinner-text">${mensaje}</div>
        </div>
    `;
    
    // Agregar a la página
    document.body.appendChild(overlay);
    
    // Bloquear el scroll
    document.body.style.overflow = 'hidden';
}

function ocultarSpinner() {
    const overlay = document.getElementById('actualizacion-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = 'auto';
    }
}
// =====================================
// Configuración y Variables Globales
// =====================================
const CONFIG = {
    API_URL: '/ventas/data_master/brakers/listbreakers',
    TABLA_ID: 'breakersTable',
    MONEDA: {
        estilo: 'currency',
        moneda: 'MXN',
        locale: 'es-MX'
    },
    DATATABLE: {
        columnDefs: [
            { targets: 0, width: "15%", className: 'text-center' },
            { targets: 1, width: "35%", className: 'text-left' },
            { targets: [2], width: "15%", className: 'text-center' },
            { targets: 3, width: "5%" }
        ]
    }
};

// =====================================
// Gestión del Indicador de Carga
// =====================================
const LoadingManager = {
    element: document.getElementById("loading-indicator"),
    
    show() {
        if (this.element) this.element.style.display = "block";
    },
    
    hide() {
        if (this.element) this.element.style.display = "none";
    }
};

document.addEventListener("DOMContentLoaded", () => {
    const breakersTableInstance = new BreakersTable();

    document.getElementById("TableBody").addEventListener("click", function(e) {
        if (e.target.classList.contains("ocultar-btn")) {
            breakersTableInstance.handleDelete(e);
        }
    });
});

    //Funcion para limpiar items
    function limpiaritems(){
        document.getElementById("Item").value = "";
        document.getElementById("Description").value = "";
        document.getElementById("Tipo").value = "";
        document.getElementById("TechnicalSheet").value = "";
        location.reload(); // Recargar la página para actualizar la tabla
    }


  // Función para descargar el PDF
  function downloadPDF(filename) {
    const link = document.createElement('a');
    link.href = `https://descargas.sycelephant.com/Ventas/PDF_EUA/DMITEMS/BREAKER_CATALOG/${filename}`;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}
// =====================================
// Actualizar Fichas Tecnicas
// =====================================

async function actualizarfichastecnicas() {
      // Mostrar el spinner antes de iniciar la petición
    mostrarSpinner();
    try {
        const response = await fetch('/ventas/data_master/brakers/actualizarbreaker', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        });
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
        if (response.ok) {
            Swal.fire({
                title: "Actualizado!",
                text: "Los archivos se han actualizado exitosamente.",
                icon: "success"
            }).then(() => {
                location.reload(); // Recargar la página para actualizar la tabla
            });
        }
       
        // Puedes agregar más lógica aquí para manejar la respuesta si es necesario
    } catch (error) {
        // Asegurarse de que el spinner se oculte incluso si hay un error
        
        console.error("Error", error);
        Swal.fire({
            title: "Error!",
            text: "Ocurrió un error.",
            icon: "error"
        });
    }
}

// =====================================
// Gestión de la Tabla de Breakers
// =====================================
class BreakersTable {


    constructor() {
        this.table = null;
        this.initializeTable();
    }

    async initializeTable() {
        LoadingManager.show();
        try {
            const data = await this.fetchBreakersData();
            this.renderTableData(data);
            this.initializeDataTable();
            this.setupEventListeners();
        } catch (error) {
            console.error("Error al inicializar la tabla:", error);
        } finally {
            LoadingManager.hide();
        }
    }

    async fetchBreakersData() {
        const response = await fetch(CONFIG.API_URL);
        const data = await response.json();
        console.log(data); // Verifica que los datos de precios estén presentes
        return data;
    }
    renderTableData(data) {
        const tableBody = document.getElementById("TableBody");
        tableBody.innerHTML = data.map(breaker => this.createTableRow(breaker)).join('');
    }

    createTableRow(breaker) {
        let longitud = breaker.TechnicalSheet ? breaker.TechnicalSheet.length : 0;
        let icono;
    
        if (longitud > 0) {
           // icono = `<a href="/ventas/pdf_viewer/view/${encodeURIComponent(breaker.TechnicalSheet)}" target="_blank"><i class="fas fa-file-pdf fa-2x" style="color:red; cursor: pointer;"></i></a>`;
            icono = `<a href="${(breaker.TechnicalSheet)}" target="_blank"><i class="fas fa-file-pdf fa-2x" style="color:red; cursor: pointer;"></i></a>`;
        
        } else {
            icono = '<i class="fas fa-file-pdf-o"></i>';
        }
        return `
            <tr>
                <td>${breaker.Item_breaker}</td>
                <td>${breaker.description}</td>
                <td>${breaker.type}</td>
                <td> ${icono} </td>
            </tr>
        `;
    }


  
    initializeDataTable() {
        if ($.fn.DataTable.isDataTable("#" + CONFIG.TABLA_ID)) {
            $('#' + CONFIG.TABLA_ID).DataTable().destroy();
        }

        this.table = $('#' + CONFIG.TABLA_ID).DataTable({
            dom: '<"d-flex justify-content-between align-items-center mb-2" Bf>rtip',
            buttons: [{
                extend: 'excel',
                text: '<i class="fas fa-file-excel"></i> Excel',
                className: 'btn btn-success dm-btn-xs dm-custom-excel-button'
            }],
            scrollX: true,
            autoWidth: true,
            ordering: false,
            fixedHeader: true,
            searching: true,
            columnDefs: [
                { targets: 0, width: "15%" },
                { targets: 1, width: "40%" },
                { targets: 2, width: "15%" },
                { targets: 3, width: "30%" },
                {
                    targets: 0,
                    className: 'text-center',
                },
                {
                    targets: 1,
                    className: 'text-left',
                },
                {
                    targets: 2,
                    className: 'text-center',
                },
                
                {
                    targets: 3,
                    className: 'text-center',
                },
            ],
            language: {
                search: "Buscar:",
                lengthMenu: "Mostrar _MENU_ registros por página",
                info: "Mostrando _START_ a _END_ de _TOTAL_ registros",
                infoEmpty: "Mostrando 0 a 0 de 0 registros",
                paginate: {
                    first: "Primero",
                    last: "Último",
                    next: "Siguiente",
                    previous: "Anterior"
                }
            }
        });
    }

    setupEventListeners() {
        // Filtros de columna
        $('.column-filter').on('keyup change', (e) => {
            const columnIndex = $(e.target).data('column');
            this.table.column(columnIndex).search(e.target.value).draw();
        });

        // Edición de registros
        $('#' + CONFIG.TABLA_ID).on('click', '.editarbreakers', this.handleEdit);

        // Eliminación de registros
        $('#TableBody').on('click', '.ocultar-btn', this.handleDelete.bind(this));
    }

    handleEdit(e) {
        const row = e.target.closest('tr');
        if (!row) return;

        const [item, description, price, tipo] = Array.from(row.cells)
            .map((cell, index) => {
                const value = cell.textContent.trim();
                return index === 2 ? value.replace(/[^\d.-]/g, '') : value;
            });

        document.getElementById("ItemEdit").value = item;
        document.getElementById("DescriptionEdit").value = description;
        document.getElementById("PriceEdit").value = price;
        document.getElementById("TipoEdit").value = tipo;
    }

    async handleDelete(e) {
        const row = $(e.target).closest('tr');
        const item = row.find('td').eq(0).text().trim();
        
        const result = await Swal.fire({
            title: "¿Estás seguro de eliminar este item?",
            icon: "warning",
            showCancelButton: true,
            confirmButtonColor: "#3085d6",
            cancelButtonColor: "#d33",
            confirmButtonText: "Sí, eliminar!"
        });

        if (result.isConfirmed) {
            try {
                const response = await fetch('/ventas/data_master/brakers/deletebreaker', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ item: item })
                });

                if (response.ok) {
                    row.fadeOut(500, () => { 
                        this.table.row(row).remove().draw();
                        Swal.fire({
                            title: "Eliminado!",
                            text: "El item ha sido eliminado.",
                            icon: "success"
                        });
                    });
                } else {
                    const errorData = await response.json();
                    Swal.fire({
                        title: "Error!",
                        text: errorData.error,
                        icon: "error"
                    });
                }
            } catch (error) {
                console.error("Error al eliminar el breaker:", error);
                Swal.fire({
                    title: "Error!",
                    text: "Ocurrió un error al eliminar el breaker.",
                    icon: "error"
                });
            }
        }
    }



}

// =====================================
// Gestión de Modales
// =====================================
class ModalManager {
    static cerrarModalEditar() {
        const modal = document.getElementById("editarbreakersCatalog");
        const modalInstance = bootstrap.Modal.getInstance(modal);
        
        if (modalInstance) {
            modalInstance.hide();
        }

        setTimeout(() => {
            document.querySelectorAll(".modal-backdrop").forEach(backdrop => backdrop.remove());
            document.body.classList.remove("modal-open");
            document.body.style.overflow = "auto";
        }, 300);
    }

    static validarFormulario(event) {
        const campos = [
            { id: "Item", mensaje: "El campo Item es obligatorio." },
            { id: "Description", mensaje: "El campo Descripción es obligatorio." },
            { id: "Price", mensaje: "El campo Precio es obligatorio." },
            { id: "Tipo", mensaje: "Debes seleccionar un tipo.", invalidValue: "Elige una opción" }
        ];

        let formularioValido = true;

        campos.forEach(campo => {
            const input = document.getElementById(campo.id);
            const valor = input.value.trim();
            let mensajeError = document.getElementById(`${campo.id}-error`);

            if (!mensajeError) {
                mensajeError = document.createElement("small");
                mensajeError.id = `${campo.id}-error`;
                mensajeError.style.color = "red";
                input.parentNode.appendChild(mensajeError);
            }

            if (!valor || valor === campo.invalidValue) {
                mensajeError.textContent = campo.mensaje;
                formularioValido = false;
            } else {
                mensajeError.textContent = "";
            }
        });

        if (!formularioValido) {
            event.preventDefault();
        }
    }
}

// =====================================
//  Guardar
// =====================================

document.getElementById("formAgregarBreaker").addEventListener("submit", async function(event) {
    event.preventDefault();
   
    const item = document.getElementById("Item").value.trim();
    const description = document.getElementById("Description").value.trim();
    const price = document.getElementById("Price").value.trim();
    const tipo = document.getElementById("Tipo").value.trim();
    const technicalSheet = document.getElementById("TechnicalSheet").files;
    const formData = new FormData();
    
    formData.append('item', item);
    formData.append('description', description);
    formData.append('price', price);
    formData.append('type', tipo);

    // Validar y añadir el archivo PDF si existe
    if (technicalSheet.length > 0) {
        if (!technicalSheet[0].name.toLowerCase().endsWith('.pdf')) {
            Swal.fire({
                title: "Error!",
                text: "Por favor selecciona un archivo PDF.",
                icon: "error"
            });
            return;
        }
        formData.append('TechnicalSheet', technicalSheet[0]);
    }
    
    // Mostrar el spinner antes de iniciar la petición
    mostrarSpinner();
    
    try {
        const response = await fetch('/ventas/data_master/brakers/addbreaker', {
            method: 'POST',
            body: formData
        });
        
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
        
        if (response.ok) {
            Swal.fire({
                title: "Guardado!",
                text: "El breaker ha sido guardado exitosamente.",
                icon: "success"
            }).then(() => {
                // Limpiar todas las cajas de texto
                document.getElementById("Item").value = "";
                document.getElementById("Description").value = "";
                document.getElementById("Price").value = "";
                document.getElementById("Tipo").value = "";
                document.getElementById("TechnicalSheet").value = "";
                location.reload(); // Recargar la página para actualizar la tabla
            });
        } else {
            const errorData = await response.json();
            Swal.fire({
                title: "Error!",
                text: errorData.error,
                icon: "error"
            });
        }
    } catch (error) {
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
        console.error("Error al guardar el breaker:", error);
        Swal.fire({
            title: "Error!",
            text: "Ocurrió un error al guardar el breaker.",
            icon: "error"
        });
    }
});// =====================================
// Actualizar
// =====================================
document.getElementById("formEditarbreakers").addEventListener("submit", async function(event) {
    event.preventDefault();

    const item = document.getElementById("ItemEdit").value.trim();
    const description = document.getElementById("DescriptionEdit").value.trim();
    const price = document.getElementById("PriceEdit").value.trim();
    const tipo = document.getElementById("TipoEdit").value.trim();

    const data = {
        item: item,
        description: description,
        price: price,
        type: tipo
    };

    try {
        const response = await fetch('/ventas/data_master/brakers/updatebreaker', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(data)
        });

        if (response.ok) {
            Swal.fire({
                title: "Actualizado!",
                text: "El breaker ha sido actualizado exitosamente.",
                icon: "success"
            }).then(() => {
                location.reload(); // Recargar la página para actualizar la tabla
            });
        } else {
            const errorData = await response.json();
            Swal.fire({
                title: "Error!",
                text: errorData.error,
                icon: "error"
            });
        }
    } catch (error) {
        console.error("Error al actualizar el breaker:", error);
        Swal.fire({
            title: "Error!",
            text: "Ocurrió un error al actualizar el breaker.",
            icon: "error"
        });
    }
});
// =====================================
// Inicialización
// =====================================
document.addEventListener("DOMContentLoaded", () => {
    new BreakersTable();
    // Limpieza de modal
    document.getElementById("editarbreakersCatalog").addEventListener("hidden.bs.modal", ModalManager.cerrarModalEditar);
});
