// Optimized JavaScript Code
/**
 * Test.js - Gestión de catálogo de controles
 * Ruta: src\static\js\Ventas\VentasEUA\DataMaster\Test.js
 */

function agregarEstilosOverlay() {
    // Verificar si ya existe el estilo
    if (!document.getElementById('overlay-styles')) {
        const style = document.createElement('style');
        style.id = 'overlay-styles';
        style.innerHTML = `
            .page-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
                flex-direction: column;
            }
            .spinner-container {
                text-align: center;
                color: white;
            }
            .spinner-container .spinner-border {
                width: 3rem;
                height: 3rem;
            }
            .spinner-text {
                margin-top: 15px;
                font-size: 18px;
                font-weight: bold;
            }
        `;
        document.head.appendChild(style);
    }
}

function mostrarSpinner(mensaje = "Actualizando fichas técnicas...") {
    agregarEstilosOverlay();
    
    // Crear el overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-overlay';
    overlay.id = 'actualizacion-overlay';
    
    // Crear el spinner y el mensaje
    overlay.innerHTML = `
        <div class="spinner-container">
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <div class="spinner-text">${mensaje}</div>
        </div>
    `;
    
    // Agregar a la página
    document.body.appendChild(overlay);
    
    // Bloquear el scroll
    document.body.style.overflow = 'hidden';
}

function ocultarSpinner() {
    const overlay = document.getElementById('actualizacion-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = 'auto';
    }
}
// =====================================
// Configuración y Variables Globales
// =====================================
const CONFIG = {
    API_URL: '/ventas/data_master/PrjWarTestConfig/listTestConfigs',
    TABLA_ID: 'TestTable',
    MONEDA: {
        estilo: 'currency',
        moneda: 'MXN',
        locale: 'es-MX'
    },
    DATATABLE: {
        columnDefs: [
            { targets: 0, width: "10%", className: 'text-center' },
            { targets: 1, width: "10%", className: 'text-center' },
            { targets: 2, width: "10%", className: 'text-center' },
            { targets: 3, width: "10%", className: 'text-center' },
            { targets: 4, width: "10%", className: 'text-center' },
            { targets: 5, width: "10%", className: 'text-center' }            
        ]
    }
};

// =====================================
// Gestión del Indicador de Carga
// =====================================
const LoadingManager = {
    element: document.getElementById("loading-indicator"),
    
    show() {
        if (this.element) this.element.style.display = "block";
    },
    
    hide() {
        if (this.element) this.element.style.display = "none";
    }
};

// =====================================
//  Limpiar items
// =====================================
function limpiaritems() {
    document.getElementById("ConfigID").value = "";
    document.getElementById("CertificationID").value = "";
    document.getElementById("Fueltype").value = "";
    document.getElementById("Item_Enclosure_type").value = "";
    document.getElementById("Item_Enclosure_Mat").value = "";
    document.getElementById("KWOutPut").value = "";
    document.getElementById("Cost").value = "";
    document.getElementById("Evaluator").value = "";
    document.getElementById("TechnicalSheet").value = "";
}

// =====================================
// Actualizar Fichas Tecnicas
// =====================================
async function actualizarfichastecnicas() {
    // Mostrar el spinner antes de iniciar la petición
    mostrarSpinner();
    try {
        const response = await fetch('/ventas/data_master/Enclosure/actualizarEnclosure', {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        });

        // Agregar esto para depurar
        const responseData = await response.json();
        console.log("Respuesta del servidor:", responseData);
        // Ocultar el spinner una vez que hay respuesta
        ocultarSpinner();
        if (response.ok) {
            Swal.fire({
                title: "Actualizado!",
                text: "Los archivos se han actualizado exitosamente.",
                icon: "success"
            }).then(() => {
                location.reload(); // Recargar la página para actualizar la tabla
            });
        }
    } catch (error) {
        // Asegurarse de que el spinner se oculte incluso si hay un error
        ocultarSpinner();
        console.error("Error", error);
        Swal.fire({
            title: "Error!",
            text: "Ocurrió un error.",
            icon: "error"
        });
    }
}

// =====================================
// Gestión de la Tabla de Enclosure
// =====================================
class Enclosure_Table {
    constructor() {
        this.table = null;
        this.initializeTable();
    }

    async initializeTable() {
        LoadingManager.show();
        try {
            const data = await this.fetchEnclosureData();
            this.renderTableData(data);
            this.initializeDataTable();
            this.setupEventListeners();
        } catch (error) {
            console.error("Error al inicializar la tabla:", error);
        } finally {
            LoadingManager.hide();
        }
    }

    async fetchEnclosureData() {
        try {
            const response = await fetch(CONFIG.API_URL);
            const data = await response.json();
            console.log("Datos obtenidos:", data); // Verificar los datos recibidos
            return data;
        } catch (error) {
            console.error("Error al obtener los datos:", error);
            return [];
        }
    }

    renderTableData(data) {
        const tableBody = document.getElementById("TableBody");
        if (!tableBody) {
            console.error("No se encontró el elemento TableBody");
            return;
        }
        tableBody.innerHTML = data.map(enclosure => this.createTableRow(enclosure)).join('');
    }

    createTableRow(enclosure) {
        // Verificación de datos para depuración
        if (!enclosure) {
            console.error("Se recibió un registro nulo o indefinido");
            return "";
        }
        
        console.log("Procesando registro:", enclosure);
        
        // Accede a las propiedades con los nombres correctos según el SQL
        let longitud = enclosure.TechnicalSheet ? enclosure.TechnicalSheet.length : 0;
        let icono;
    
        if (longitud > 0) {
            icono = `<a href="${enclosure.TechnicalSheet}" target="_blank"><i class="fas fa-file-pdf fa-2x" style="color:red; cursor: pointer;"></i></a>`;
        } else {
            icono = '<i class="fas fa-file-pdf-o"></i>';
        }
        
        return `
            <tr>
                <td>${enclosure.ConfigID}</td>
                <td>${enclosure.CertificationID}</td>
                <td>${enclosure.Fueltype}</td>
                <td>${enclosure.Item_Enclosure_type}</td>
                <td>${enclosure.Item_Enclosure_Mat}</td>
                <td>${enclosure.KWOutPut}</td>
                <td>${enclosure.Cost}</td>
                <td>${enclosure.Evaluator}</td>
                <td>${icono}</td>
                <td>
                    <div class="btn-group">
                        <button type="button" class="btn btn-primary dm-btn-small editarEnclosureCatalog" 
                                data-bs-toggle="modal" data-bs-target="#editarEnclosureCatalog">
                            <i class="fas fa-edit"></i>
                            <span class="dm-btn-text">Editar</span>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    }

    initializeDataTable() {
        if ($.fn.DataTable.isDataTable("#" + CONFIG.TABLA_ID)) {
            $('#' + CONFIG.TABLA_ID).DataTable().destroy();
        }

        this.table = $('#' + CONFIG.TABLA_ID).DataTable({
            dom: '<"d-flex justify-content-between align-items-center mb-2" Bf>rtip',
            buttons: [{
                extend: 'excel',
                text: '<i class="fas fa-file-excel"></i> Excel',
                className: 'btn btn-success dm-btn-xs dm-custom-excel-button'
            }],
            scrollX: true,
            autoWidth: true,
            ordering: false,
            fixedHeader: true,
            searching: true,
            columnDefs: [
                { targets: 0, width: "10%" },
                { targets: 1, width: "10%" },
                { targets: 2, width: "10%" },
                { targets: 3, width: "10%" },
                { targets: 4, width: "10%" },
                { targets: 5, width: "10%" },
                { targets: 6, width: "10%" },
                { targets: 7, width: "10%" },
                { targets: 8, width: "10%" },
                {
                    targets: [0, 2, 3, 4, 5],
                    className: 'text-center',
                },
                {
                    targets: [1, 6, 7],
                    className: 'text-center',
                },
                {
                    targets: 6,
                    className: 'text-center',
                    render: function(data, type, row) {
                        if (type === 'display' || type === 'filter') {
                            // Verificar si el dato es un número válido
                            const number = parseFloat(data);
                            if (!isNaN(number)) {
                                return new Intl.NumberFormat('es-MX', {
                                    style: 'currency',
                                    currency: 'MXN'
                                }).format(number);
                            } else {
                                return data; // Devolver el dato original si no es un número válido
                            }
                        }
                        return data;
                    }
                },
                {
                    targets: 8,
                    className: 'text-center',
                },
            ],
            language: {
                search: "Buscar:",
                lengthMenu: "Mostrar _MENU_ registros por página",
                info: "Mostrando _START_ a _END_ de _TOTAL_ registros",
                infoEmpty: "Mostrando 0 a 0 de 0 registros",
                paginate: {
                    first: "Primero",
                    last: "Último",
                    next: "Siguiente",
                    previous: "Anterior"
                }
            }
        });
    }

    setupEventListeners() {
        // Filtros de columna
        $('.column-filter').on('keyup change', (e) => {
            const columnIndex = $(e.target).data('column');
            this.table.column(columnIndex).search(e.target.value).draw();
        });

        // Edición de registros
        $('#' + CONFIG.TABLA_ID).on('click', '.editarEnclosureCatalog', (e) => {
            const row = e.target.closest('tr');
            if (!row) return;
        
            const cells = row.cells;
            const configID = cells[0].textContent.trim();
            const certificationID = cells[1].textContent.trim();
            const fueltype = cells[2].textContent.trim();
            const item_Enclosure_type = cells[3].textContent.trim();
            const item_Enclosure_Mat = cells[4].textContent.trim();
            const kwoutput = cells[5].textContent.trim();
            const cost = cells[6].textContent.trim().replace(/[^\d.-]/g, '');
            const evaluator = cells[7].textContent.trim();
        
            document.getElementById("ConfigID").value = configID;
            document.getElementById("CertificationID").value = certificationID;
            document.getElementById("Fueltype").value = fueltype;
            document.getElementById("Item_Enclosure_type").value = item_Enclosure_type;
            document.getElementById("Item_Enclosure_Mat").value = item_Enclosure_Mat;
            document.getElementById("KWOutput").value = kwoutput;
            document.getElementById("Cost").value = cost;
            document.getElementById("Evaluator").value = evaluator;
        });
    }
}

// =====================================
// Gestión de Modales
// =====================================
class ModalManager {
    static cerrarModalEditar() {
        const modal = document.getElementById("editarEnclosureCatalog");
        const modalInstance = bootstrap.Modal.getInstance(modal);
        
        if (modalInstance) {
            modalInstance.hide();
        }

        setTimeout(() => {
            document.querySelectorAll(".modal-backdrop").forEach(backdrop => backdrop.remove());
            document.body.classList.remove("modal-open");
            document.body.style.overflow = "auto";
        }, 300);
    }

    static validarFormulario(event) {
        const campos = [
            { id: "ConfigID", mensaje: "El campo ConfigID es obligatorio." },
            { id: "CertificationID", mensaje: "El campo CertificationID es obligatorio." },
            { id: "Fueltype", mensaje: "El campo Fueltype es obligatorio." },
            { id: "Item_Enclosure_type", mensaje: "El campo Item_Enclosure_type es obligatorio." },
            { id: "Item_Enclosure_Mat", mensaje: "El campo Item_Enclosure_Mat es obligatorio." },
            { id: "KWOutput", mensaje: "El campo KWOutput es obligatorio." },
            { id: "Cost", mensaje: "El campo Costo es obligatorio." },
            { id: "Evaluator", mensaje: "El campo Evaluator es obligatorio." },
        ];

        let formularioValido = true;

        campos.forEach(campo => {
            const input = document.getElementById(campo.id);
            const valor = input.value.trim();
            let mensajeError = document.getElementById(`${campo.id}-error`);

            if (!mensajeError) {
                mensajeError = document.createElement("small");
                mensajeError.id = `${campo.id}-error`;
                mensajeError.style.color = "red";
                input.parentNode.appendChild(mensajeError);
            }

            if (!valor || valor === campo.invalidValue) {
                mensajeError.textContent = campo.mensaje;
                formularioValido = false;
            } else {
                mensajeError.textContent = "";
            }
        });

        if (!formularioValido) {
            event.preventDefault();
        }
    }
}

// =====================================
// Cerrar modal editar (función global para acceder desde el HTML)
// =====================================
function cerrarModalEditar() {
    ModalManager.cerrarModalEditar();
}

// =====================================
// Inicialización cuando el DOM está listo
// =====================================
document.addEventListener("DOMContentLoaded", () => {
    // Inicializar la tabla
    const enclosureTableInstance = new Enclosure_Table();
    
    // Agregar listener para el formulario de editar
    const formEditarEnclosure = document.getElementById("formEditarEnclosure");
    if (formEditarEnclosure) {
        formEditarEnclosure.addEventListener("submit", async function(event) {
            event.preventDefault();

            const configID = document.getElementById("configID").value.trim();
            const certificationID = document.getElementById("certificationID").value.trim();
            const fueltype = document.getElementById("fueltype").value.trim();
            const item_Enclosure_type = document.getElementById("Item_Enclosure_type").value.trim();
            const item_Enclosure_Mat = document.getElementById("item_Enclosure_Mat").value.trim();
            const kwoutput = document.getElementById("kwoutput").value.trim();
            const cost = document.getElementById("cost").value.trim();
            const Evaluator = document.getElementById("Evaluator").value.trim();

            const data = {
                item: configID,
                certificationID: certificationID,
                fueltype: fueltype,
                item_Enclosure_type: item_Enclosure_type,
                item_Enclosure_Mat: item_Enclosure_Mat,
                kwoutput: kwoutput,
                price: cost,
                Evaluator: Evaluator
            };

            try {
                const response = await fetch('/ventas/data_master/Enclosure/updateEnclosure', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });

                if (response.ok) {
                    Swal.fire({
                        title: "Actualizado!",
                        text: "El Catálogo de componentes adicionales de casetas ha sido actualizado exitosamente.",
                        icon: "success"
                    }).then(() => {
                        location.reload(); // Recargar la página para actualizar la tabla
                    });
                } else {
                    const errorData = await response.json();
                    Swal.fire({
                        title: "Error!",
                        text: errorData.error,
                        icon: "error"
                    });
                }
            } catch (error) {
                console.error("Error al actualizar caseta:", error);
                Swal.fire({
                    title: "Error!",
                    text: "Ocurrió un error al actualizar la caseta.",
                    icon: "error"
                });
            }
        });
    }

    // Corregir el atributo form en el botón de actualizar
    const formEditarBtn = document.querySelector("button[form='formEditarEnclosure']");
    if (formEditarBtn) {
        formEditarBtn.setAttribute("form", "formEditarEnclosure");
    }

    // Limpieza de modal
    const modalElement = document.getElementById("editarEnclosureCatalog");
    if (modalElement) {
        modalElement.addEventListener("hidden.bs.modal", ModalManager.cerrarModalEditar);
    }
});