// Archivo: VEUcotizador.js
// Ruta: src/static/js/Ventas/VentasEUA/VEUcotizador.js
// Lenguaje: JavaScript

/* Organizacion: el JS esta organizado por secciones, cada una de una accion diferente. y cada seccion 
se organiza así: 
al principio de cada seccion primero se declara constantes de toda la seccion despues si no existen 
subsecciones se ordena así:
primero se colocan los Listeners, despues las Faunciones y al final las inicializaciones.  
Si existen subsecciones, entonces se escribe el titulo de la subseccion y despues se respeta el orden:
primero se colocan los Listeners, despues las Faunciones y al final las inicializaciones. */

document.addEventListener('DOMContentLoaded', function () {
    /****************************************************
     * SECCIÓN 1: NAVEGACIÓN ENTRE PASOS
     ****************************************************/
    // Constantes
    const steps = document.querySelectorAll('.step');
    const formSteps = document.querySelectorAll('.form-step');
    const nextBtns = document.querySelectorAll('.next-btn');
    const prevBtns = document.querySelectorAll('.prev-btn');
    let currentStep = 0;

    // Event Listeners
    nextBtns.forEach((btn) =>
        btn.addEventListener('click', () => {
            // Validar campos del paso actual antes de avanzar
            if (validarCamposPaso(currentStep)) {
                if (currentStep < steps.length - 1) {
                    currentStep++;
                    updateSteps();
                }
            }
        }),
    );

    prevBtns.forEach((btn) =>
        btn.addEventListener('click', () => {
            if (currentStep > 0) {
                currentStep--;
                updateSteps();
            }
        }),
    );

    // Funciones
    function updateSteps() {
        steps.forEach((step, index) => {
            step.classList.toggle('active', index === currentStep);
        });

        formSteps.forEach((formStep, index) => {
            formStep.classList.toggle('active', index === currentStep);
        });

        // Opción 1: Scroll al principio del formulario
        document.querySelector('.multi-step-container').scrollIntoView({
            behavior: 'smooth',
            block: 'start',
        });
    }

    /****************************************************
     * SECCIÓN 2: GESTIÓN DE VISIBILIDAD DE SECCIONES
     ****************************************************/
    // Constantes
    const qtyBreakersSelect = document.getElementById('QtyBreakers');
    const seccionesInterruptores = document.querySelectorAll('.section-interruptores');
    const BrandControlSelect = document.getElementById('Controlbrand');
    const seccionesControl = document.querySelectorAll('.section-ctrol');
    const EnclosureSelect = document.getElementById('Enclosure');
    const seccionesAutomaticLouvers = document.querySelectorAll('.section-acutomaticlouvers');


    // Mapeos de Opciones
    const opcionesInterruptores = {
        1: [0],
        2: [0, 1],
        3: [0, 1, 2],
    };

    const opcionesControl = {
        'DEEP SEA': [0],
        COMAP: [1],
    };

    const opcionesLouvers = {
        WPack1: [0],
        WPack2: [0],
    };

    // Event Listeners
    qtyBreakersSelect.addEventListener('change', () =>
        actualizarSecciones(qtyBreakersSelect, seccionesInterruptores, opcionesInterruptores),
    );
    BrandControlSelect.addEventListener('change', () =>
        actualizarSecciones(BrandControlSelect, seccionesControl, opcionesControl),
    );
    EnclosureSelect.addEventListener('change', () =>
        actualizarSecciones(EnclosureSelect, seccionesAutomaticLouvers, opcionesLouvers),
    );

    // Funciones
    /**
     * Muestra las secciones de acuerdo a la seleccion de select
     */
    function actualizarSecciones(selectElement, secciones, opciones) {
        secciones.forEach((seccion) => {
            seccion.style.display = 'none';
        });

        const seleccion = selectElement.value;
        opciones[seleccion]?.forEach((index) => {
            secciones[index].style.display = 'block';
        });
    }

    // Inicialización
    actualizarSecciones(qtyBreakersSelect, seccionesInterruptores, opcionesInterruptores);
    actualizarSecciones(BrandControlSelect, seccionesControl, opcionesControl);
    actualizarSecciones(EnclosureSelect, seccionesAutomaticLouvers, opcionesLouvers);

    /****************************************************
     * SECCIÓN 3: GESTIÓN DE SELECTBOX - INFORMACIÓN GENERAL
     ****************************************************/
    // Constantes
    const tipoCombustibleSelect = document.getElementById('tipoCombustible');
    const capacidadKWSelect = document.getElementById('capacidadKW');
    const marcaMotorSelect = document.getElementById('marcaMotor');
    const marcaGeneradorSelect = document.getElementById('marcaGenerador');
    const voltajeSelect = document.getElementById('voltaje');
    const certifiedSelect = document.getElementById('certified');
    const cantidadPlantasInput = document.getElementById('cantidadPlantas');

    /*-------------------- Tipo de Combustible --------------------*/
    // Event Listeners
    document.addEventListener('DOMContentLoaded', cargarTiposCombustible);

    // Funciones
    async function cargarTiposCombustible() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/fuel-types');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();
            tipoCombustibleSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.Fueltype, tipo.Fueltype);
                option.dataset.id = tipo.id; // Guardamos el ID en el dataset del option
                tipoCombustibleSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            tipoCombustibleSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarTiposCombustible();

    /*-------------------- Capacidad KW --------------------*/
    // Event Listeners
    tipoCombustibleSelect.addEventListener('change', (e) => cargarCapacidades(e.target.value));

    // Funciones
    async function cargarCapacidades(fuelType) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/capacidades/${fuelType}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const capacidades = await response.json();
            capacidadKWSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            capacidades.forEach((cap) => {
                capacidadKWSelect.appendChild(new Option(cap.KW, cap.KW));
            });
        } catch (error) {
            console.error('Error:', error);
            capacidadKWSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Marca del Motor --------------------*/
    // Event Listeners
    capacidadKWSelect.addEventListener('change', (e) => {
        const fuelType = tipoCombustibleSelect.value;
        const kw = e.target.value;
        cargarMarcasMotor(fuelType, kw);
    });

    // Funciones
    async function cargarMarcasMotor(fuelType, kw) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/engine-brands/${fuelType}/${kw}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const marcas = await response.json();
            marcaMotorSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            marcas.forEach((marca) => {
                const option = new Option(marca.brand, marca.brand);
                option.dataset.id = marca.id;
                marcaMotorSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            marcaMotorSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Marca del Generador --------------------*/
    // Event Listeners
    marcaMotorSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const engineBrandId = selectedOption.dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarMarcasAlternador(fuelType, kw, engineBrandId);
    });

    // Funciones
    async function cargarMarcasAlternador(fuelType, kw, engineBrandId) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/alternator-brands/${fuelType}/${kw}/${engineBrandId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const marcas = await response.json();
            marcaGeneradorSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            marcas.forEach((marca) => {
                const option = new Option(marca.brand, marca.brand);
                option.dataset.id = marca.id;
                marcaGeneradorSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            marcaGeneradorSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Voltaje --------------------*/
    // Event Listeners
    marcaGeneradorSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const alternatorBrandId = selectedOption.dataset.id;
        const engineBrandId = marcaMotorSelect.selectedOptions[0].dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarVoltajes(fuelType, kw, engineBrandId, alternatorBrandId);
    });

    // Funciones
    async function cargarVoltajes(fuelType, kw, engineBrandId, alternatorBrandId) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/voltages/${fuelType}/${kw}/${engineBrandId}/${alternatorBrandId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const voltajes = await response.json();
            voltajeSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            voltajes.forEach((voltaje) => {
                const option = new Option(voltaje.voltage, voltaje.voltage);
                option.dataset.id = voltaje.id;
                voltajeSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            voltajeSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Certificaciones --------------------*/
    // Event Listeners
    voltajeSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const voltageId = selectedOption.dataset.id;
        const alternatorBrandId = marcaGeneradorSelect.selectedOptions[0].dataset.id;
        const engineBrandId = marcaMotorSelect.selectedOptions[0].dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarCertificaciones(fuelType, kw, engineBrandId, alternatorBrandId, voltageId);
    });

    // Funciones
    async function cargarCertificaciones(
        fuelType,
        kw,
        engineBrandId,
        alternatorBrandId,
        voltageId,
    ) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/certifications/${fuelType}/${kw}/${engineBrandId}/${alternatorBrandId}/${voltageId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const certificaciones = await response.json();
            certifiedSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            certificaciones.forEach((cert) => {
                const option = new Option(cert.certification, cert.certification);
                option.dataset.id = cert.id;
                certifiedSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            certifiedSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Cantidad de Plantas --------------------*/

    // Event Listeners
    cantidadPlantasInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    cantidadPlantasInput.addEventListener('blur', () => {
        const value = cantidadPlantasInput.value;

        // Si está vacío o es menor que 1, establecer a 1
        if (!value || parseInt(value) < 1) {
            cantidadPlantasInput.value = 1;
        }
    });

    // Prevenir caracteres no deseados
    cantidadPlantasInput.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /****************************************************
     * SECCIÓN 4: GESTIÓN DE SELECTBOX - INTERRUPTORES
     ****************************************************/

    /*-------------------- Modelo de interruptores --------------------*/
    // Funciones
    function inicializarInterruptores() {
        async function cargarOpcionesInterruptor(numeroInterruptor) {
            try {
                const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/breaker-options');
                if (!response.ok) throw new Error('Error en la respuesta del servidor');

                const opciones = await response.json();
                const select = document.getElementById(`interruptor${numeroInterruptor}`);

                select.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
                opciones.forEach((opcion) => {
                    const option = new Option(opcion.description, opcion.id);
                    option.dataset.type = opcion.type;
                    select.appendChild(option);
                });
            } catch (error) {
                console.error(`Error loading options ${numeroInterruptor}:`, error);
            }
        }
        qtyBreakersSelect.addEventListener('change', (e) => {
            actualizarSecciones(qtyBreakersSelect, seccionesInterruptores, opcionesInterruptores);

            const cantidad = parseInt(e.target.value);
            for (let i = 1; i <= cantidad; i++) {
                cargarOpcionesInterruptor(i);
            }
        });
    }

    // Inicialización
    inicializarInterruptores();

    /*-------------------- Tipos de protección --------------------*/
    // Funciones
    function inicializarTiposProteccion() {
        const interruptores = ['interruptor1', 'interruptor2', 'interruptor3'].map((id) =>
            document.getElementById(id),
        );

        interruptores.forEach((interruptor, index) => {
            if (interruptor) {
                interruptor.addEventListener('change', (e) => {
                    const breakerType = e.target.selectedOptions[0]?.dataset.type;
                    if (breakerType) {
                        cargarTiposProteccion(
                            breakerType,
                            `lsigDisparo${index === 0 ? '' : index + 1}`,
                        );
                    }
                });
            }
        });
    }

    async function cargarTiposProteccion(breakerType, protectionSelectId) {
        const protectionSelect = document.getElementById(protectionSelectId);
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/protection-types/${breakerType}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const protections = await response.json();
            protectionSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            protections.forEach((protection) => {
                const option = new Option(protection.description, protection.id);
                protectionSelect.appendChild(option);
            });
        } catch (error) {
            console.error(`Error cargando tipos de protección para ${protectionSelectId}:`, error);
            protectionSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    inicializarTiposProteccion();

    /****************************************************
     * SECCIÓN 5: GESTIÓN DE SELECTBOX - CONTROL
     ****************************************************/
    // Constantes
    const controlModelSelect = document.getElementById('ControlModel');
    const voltOperSelect = document.getElementById('VotlOper');
    const batteryChargerSelect = document.getElementById('BateryCharger');
    const DSE2157 = document.getElementById('DSE2157');
    const qtyDSE2157 = document.getElementById('QtyDSE2157');
    const DSE2548 = document.getElementById('DSE2548');
    const qtyDSE2548 = document.getElementById('QtyDSE2548');
    const ComapDSE2157 = document.getElementById('ComapDSE2157');
    const qtyComapDSE2157 = document.getElementById('QtyComapDSE2157');
    const ComapDSE2548 = document.getElementById('ComapDSE2548');
    const qtyComapDSE2548 = document.getElementById('QtyComapDSE2548');

    // Caseta gestion select box
    select_louvers_out = document.getElementById('louvers_out');
    select_louvers_in = document.getElementById('louvers_in');

    /*-------------------- Marcas de Control --------------------*/
    // Event Listeners
    certifiedSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const certificationId = selectedOption.dataset.id;
        const voltageId = voltajeSelect.selectedOptions[0].dataset.id;
        const alternatorBrandId = marcaGeneradorSelect.selectedOptions[0].dataset.id;
        const engineBrandId = marcaMotorSelect.selectedOptions[0].dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarMarcasControl(
            fuelType,
            kw,
            engineBrandId,
            alternatorBrandId,
            voltageId,
            certificationId,
        );
    });

    // Funciones
    async function cargarMarcasControl(
        fuelType,
        kw,
        engineBrandId,
        alternatorBrandId,
        voltageId,
        certificationId,
    ) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/control-brands/${fuelType}/${kw}/${engineBrandId}/${alternatorBrandId}/${voltageId}/${certificationId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const marcasControl = await response.json();
            BrandControlSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            marcasControl.forEach((marca) => {
                const option = new Option(marca.brand, marca.brand);
                option.dataset.id = marca.id;
                BrandControlSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            BrandControlSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Marca y Modelo del Control --------------------*/
    // Event Listeners
    BrandControlSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const brandId = selectedOption.dataset.id;
        cargarModelosControl(brandId);
    });

    // Funciones
    async function cargarModelosControl(brandId) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/control-models/${brandId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const modelos = await response.json();
            controlModelSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            modelos.forEach((modelo) => {
                const option = new Option(modelo.description, modelo.item);
                option.dataset.id = modelo.item;
                controlModelSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            controlModelSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Cantidad de DSE2157 Output module (8 dry contacts) --------------------*/
    // Event Listeners
    DSE2157.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyDSE2157.disabled = false;
            qtyDSE2157.value = 1;
        } else {
            qtyDSE2157.disabled = true;
            qtyDSE2157.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyDSE2157.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyDSE2157.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyDSE2157.disabled) {
            const value = qtyDSE2157.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyDSE2157.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyDSE2157.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Cantidad de DSE2548 Remote annunciator --------------------*/
    // Event Listeners
    DSE2548.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyDSE2548.disabled = false;
            qtyDSE2548.value = 1;
        } else {
            qtyDSE2548.disabled = true;
            qtyDSE2548.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyDSE2548.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyDSE2548.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyDSE2548.disabled) {
            const value = qtyDSE2548.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyDSE2548.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyDSE2548.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*    


    /*-------------------- Cantidad de Equivalente en COMAP de DSE2548 Remote annunciator --------------------*/
    // Event Listeners
    ComapDSE2157.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyComapDSE2157.disabled = false;
            qtyComapDSE2157.value = 1;
        } else {
            qtyComapDSE2157.disabled = true;
            qtyComapDSE2157.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyComapDSE2157.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyComapDSE2157.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyComapDSE2157.disabled) {
            const value = qtyComapDSE2157.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyComapDSE2157.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyComapDSE2157.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /* 


    /*-------------------- Cantidad de Equivalente en COMAP de DSE2548 Remote annunciator --------------------*/
    // Event Listeners
    ComapDSE2548.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyComapDSE2548.disabled = false;
            qtyComapDSE2548.value = 1;
        } else {
            qtyComapDSE2548.disabled = true;
            qtyComapDSE2548.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyComapDSE2548.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyComapDSE2548.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyComapDSE2548.disabled) {
            const value = qtyComapDSE2548.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyComapDSE2548.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyComapDSE2548.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /* 


    
    /*-------------------- Voltajes de Precalentador --------------------*/
    // Funciones
    async function cargarVoltajesOperacion() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/preheater-voltages');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const voltajes = await response.json();
            voltOperSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;

            voltajes.forEach((voltaje) => {
                const option = new Option(voltaje.description, voltaje.item);
                if (voltaje.item === 'PreHeater-127') {
                    option.selected = true;
                }
                voltOperSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            voltOperSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarVoltajesOperacion();

    /*-------------------- Cargadores de Batería --------------------*/
    // Funciones
    async function cargarCargadoresBateria() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/battery-chargers');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const cargadores = await response.json();
            batteryChargerSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            cargadores.forEach((cargador) => {
                const option = new Option(cargador.description, cargador.item);
                if (cargador.item === 'DSE-9470') {
                    option.selected = true;
                }
                batteryChargerSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            batteryChargerSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarCargadoresBateria();

    /****************************************************
     * SECCIÓN 6: GESTIÓN DE SELECTBOX - CASETA
     ****************************************************/
    // Constantes
    const enclosureTypeSelect = document.getElementById('Enclosure');
    const enclosureMaterialSelect = document.getElementById('EnclosureType');
    const lightACSelect = document.getElementById('lightAC');
    const qtyLightACInput = document.getElementById('QtylightAC');
    const lightDCSelect = document.getElementById('lightDC');
    const qtyLightDCInput = document.getElementById('QtylightDC');
    const spaceHeaterSelect = document.getElementById('SpaceHeater');
    const qtySpaceHeaterInput = document.getElementById('QtySpaceHeater');
    const loadCenterSelect = document.getElementById('LoadCenter');
    const GFCI120V = document.getElementById('GFCI120V');
    const qtyGFCI120V = document.getElementById('QtyGFCI120V');
    const Receptable120V = document.getElementById('Receptable120V');
    const qtyReceptable120V = document.getElementById('QtyReceptable120V');
    const select_OpeningEnclousure =document.getElementById('OpeningEnclousure');
    const select_StopButtonEnclosure = document.getElementById('StopButtonEnclosure');
    const input_StopButtonEnclosure_Qty = document.getElementById('StopButtonEnclosure_Qty');

    /*-------------------- Tipos de Caseta --------------------*/
    // Event Listeners
    certifiedSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const certificationId = selectedOption.dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarTiposCaseta(fuelType, kw, certificationId);
    });

    // Funciones
    async function cargarTiposCaseta(fuelType, kw, certificationId) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-types/${fuelType}/${kw}/${certificationId}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();
            enclosureTypeSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.description, tipo.item);
                option.dataset.id = tipo.item;
                enclosureTypeSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            enclosureTypeSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // V1 start Cargar Stop Button Enclosure ------------------------------------
    if (select_StopButtonEnclosure) {
        select_StopButtonEnclosure.addEventListener('focus', function handler() {
            load_select_StopButtonEnclosure();
        });
        
        select_StopButtonEnclosure.addEventListener('change', function() {
            const value = select_StopButtonEnclosure.value;
            if (
                value !== '' &&
                value !== 'No' &&
                value !== 'Select an option' &&
                value !== 'Seleccione una opción'
            ) {
                input_StopButtonEnclosure_Qty.disabled = false;
                input_StopButtonEnclosure_Qty.value = 1; // Establecer valor por defecto
            } else {
                input_StopButtonEnclosure_Qty.disabled = true;
                input_StopButtonEnclosure_Qty.value = 0; // Reiniciar valor
            }
        });
    }

    async function load_select_StopButtonEnclosure() {
        try {
            const response = await fetch(`/Ventas/Ventas_EUA/Cotizador_EUA/StopButtonEnclosure`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();

            select_StopButtonEnclosure.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.stop_decription, tipo.stop_id);
                option.dataset.id = tipo.stop_id;
                select_StopButtonEnclosure.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            select_StopButtonEnclosure.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }
    // V1 end Cargar Stop Button Enclosure ------------------------------------


    // V1 start Cargar Louvers in ------------------------------------
    if (select_louvers_in) {
        select_louvers_in.addEventListener('focus', function handler() {
            load_louvers_in();
        });
    }

    async function load_louvers_in() {
        try {
            const response = await fetch(`/Ventas/Ventas_EUA/Cotizador_EUA/load_louvers_in`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();

            select_louvers_in.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.louvers_in_decription, tipo.louvers_in_id);
                option.dataset.id = tipo.louvers_in_id;
                select_louvers_in.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            select_louvers_in.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }
    // V1 end Cargar Louvers In ------------------------------------


    // V1 start Cargar Louvers Out ------------------------------------
    if (select_louvers_out) {
        select_louvers_out.addEventListener('focus', function handler() {
            load_louvers_out();
        });
    }

    async function load_louvers_out() {
        try {
            const response = await fetch(`/Ventas/Ventas_EUA/Cotizador_EUA/load_louvers_out`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();

            select_louvers_out.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.louvers_out_decription, tipo.louvers_out_id);
                option.dataset.id = tipo.louvers_out_id;
                select_louvers_out.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            select_louvers_out.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }
     // V1 start Cargar Louvers out ------------------------------------


    /*-------------------- Materiales de Caseta --------------------*/
    // Event Listeners
    enclosureTypeSelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const enclosureType = selectedOption.dataset.id;
        const certificationId = certifiedSelect.selectedOptions[0].dataset.id;
        const fuelType = tipoCombustibleSelect.value;
        const kw = capacidadKWSelect.value;
        cargarMaterialesCaseta(fuelType, kw, certificationId, enclosureType);
    });

    // Funciones
    async function cargarMaterialesCaseta(fuelType, kw, certificationId, enclosureType) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-materials/${fuelType}/${kw}/${certificationId}/${enclosureType}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const materiales = await response.json();
            enclosureMaterialSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            materiales.forEach((material) => {
                const option = new Option(material.description, material.item);
                option.dataset.id = material.item;
                enclosureMaterialSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            enclosureMaterialSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Cantidad de Toma de corriente tipo GFCI de 120V --------------------*/
    // Event Listeners
    GFCI120V.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyGFCI120V.disabled = false;
            qtyGFCI120V.value = 1;
        } else {
            qtyGFCI120V.disabled = true;
            qtyGFCI120V.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyGFCI120V.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyGFCI120V.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyGFCI120V.disabled) {
            const value = qtyGFCI120V.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyGFCI120V.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyGFCI120V.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Toma de corriente de 120V --------------------*/
    // Event Listeners
    Receptable120V.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyReceptable120V.disabled = false;
            qtyReceptable120V.value = 1;
        } else {
            qtyReceptable120V.disabled = true;
            qtyReceptable120V.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyReceptable120V.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyReceptable120V.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyReceptable120V.disabled) {
            const value = qtyReceptable120V.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyReceptable120V.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyReceptable120V.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Luces AC --------------------*/
    // Funciones
    async function cargarLucesAC() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-lights-ac');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const luces = await response.json();
            lightACSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            luces.forEach((luz) => {
                const option = new Option(luz.description, luz.item);
                lightACSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar luces AC:', error);
            lightACSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarLucesAC();

    /*-------------------- cantidad de Luces AC --------------------*/
    // Event Listeners
    lightACSelect.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyLightACInput.disabled = false;
            qtyLightACInput.value = 1;
        } else {
            qtyLightACInput.disabled = true;
            qtyLightACInput.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyLightACInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyLightACInput.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyLightACInput.disabled) {
            const value = qtyLightACInput.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyLightACInput.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyLightACInput.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Luces DC --------------------*/
    // Funciones
    async function cargarLucesDC() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-lights-dc');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const luces = await response.json();
            lightDCSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            luces.forEach((luz) => {
                const option = new Option(luz.description, luz.item);
                lightDCSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar luces DC:', error);
            lightDCSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarLucesDC();

    /*-------------------- cantidad de Luces DC --------------------*/
    // Event Listeners
    lightDCSelect.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyLightDCInput.disabled = false;
            qtyLightDCInput.value = 1;
        } else {
            qtyLightDCInput.disabled = true;
            qtyLightDCInput.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyLightDCInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyLightDCInput.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyLightDCInput.disabled) {
            const value = qtyLightDCInput.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyLightDCInput.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyLightDCInput.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Calentadores de Espacio --------------------*/
    // Funciones
    async function cargarCalentadoresEspacio() {
        try {
            const response = await fetch(
                '/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-space-heaters',
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const calentadores = await response.json();
            spaceHeaterSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            calentadores.forEach((calentador) => {
                const option = new Option(calentador.description, calentador.item);
                spaceHeaterSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar calentadores de espacio:', error);
            spaceHeaterSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarCalentadoresEspacio();

    /*-------------------- cantidad de Calentadores de Espacio --------------------*/
    // Event Listeners
    spaceHeaterSelect.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtySpaceHeaterInput.disabled = false;
            qtySpaceHeaterInput.value = 1;
        } else {
            qtySpaceHeaterInput.disabled = true;
            qtySpaceHeaterInput.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtySpaceHeaterInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtySpaceHeaterInput.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtySpaceHeaterInput.disabled) {
            const value = qtySpaceHeaterInput.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtySpaceHeaterInput.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtySpaceHeaterInput.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Centros de Carga --------------------*/
    // Funciones
    async function cargarCentrosCarga() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-load-centers');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const centros = await response.json();
            loadCenterSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            centros.forEach((centro) => {
                const option = new Option(centro.description, centro.item);
                loadCenterSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar centros de carga:', error);
            loadCenterSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    (async function load_select_OpeningEnclousure() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/enclosure-load-centers');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const centros = await response.json();
            loadCenterSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            centros.forEach((centro) => {
                const option = new Option(centro.description, centro.item);
                loadCenterSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar centros de carga:', error);
            loadCenterSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    })();

    // Inicialización
    cargarCentrosCarga();

    /****************************************************
     * SECCIÓN 7: GESTIÓN DE SELECTBOX - ACCESORIOS ADICIONALES
     ****************************************************/

    // Constantes
    const eStopSelect = document.getElementById('EStop');
    const qtyEStopInput = document.getElementById('QtyEStop');
    const springIsolatorSelect = document.getElementById('SpringIsolator');

    /*-------------------- Paro de emergencia remoto --------------------*/
    // Funciones
    async function cargarOpcionesEStop() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/additional-estop');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const opciones = await response.json();
            eStopSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            opciones.forEach((opcion) => {
                const option = new Option(opcion.description, opcion.item);
                eStopSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar opciones de paro de emergencia:', error);
            eStopSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarOpcionesEStop();

    /*-------------------- Cantidad de Paros de Emergencia --------------------*/
    // Event Listeners
    eStopSelect.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        if (
            selectedValue !== '' &&
            selectedValue !== 'No' &&
            selectedValue !== 'Select an option' &&
            selectedValue !== 'Seleccione una opción'
        ) {
            qtyEStopInput.disabled = false;
            qtyEStopInput.value = 1;
        } else {
            qtyEStopInput.disabled = true;
            qtyEStopInput.value = 0;
        }
    });

    // Event Listeners para el input de cantidad
    qtyEStopInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir que el campo esté vacío temporalmente
        if (value === '') return;

        // Convertir a número y validar
        const numValue = parseInt(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Si es decimal, convertir a entero
        if (value % 1 !== 0) {
            e.target.value = Math.floor(numValue);
        }
    });

    // Validar cuando el usuario termina de editar
    qtyEStopInput.addEventListener('blur', () => {
        // Solo validar si el input está habilitado
        if (!qtyEStopInput.disabled) {
            const value = qtyEStopInput.value;

            // Si está vacío o es menor que 1, establecer a 1
            if (!value || parseInt(value) < 1) {
                qtyEStopInput.value = 1;
            }
        }
    });

    // Prevenir caracteres no deseados
    qtyEStopInput.addEventListener('keydown', (e) => {
        // Permitir: backspace, delete, tab, escape, enter, números
        if (
            e.key === 'Backspace' ||
            e.key === 'Delete' ||
            e.key === 'Tab' ||
            e.key === 'Escape' ||
            e.key === 'Enter' ||
            (e.key >= '0' && e.key <= '9')
        ) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /*-------------------- Resorte de Aislador --------------------*/
    async function cargarOpcionesSpringIsolator() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/additional-spring');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const opciones = await response.json();
            springIsolatorSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            opciones.forEach((opcion) => {
                const option = new Option(opcion.description, opcion.item);
                springIsolatorSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar opciones de resorte de aislador:', error);
            springIsolatorSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización
    cargarOpcionesSpringIsolator();

    /****************************************************
     * SECCIÓN 8: GESTIÓN DE SELECTBOX - CASETAS
     ****************************************************/

    // Constantes
    const tankCapacitySelect = document.getElementById('TankCapacity');
    const tankTypeSelect = document.getElementById('TankType');
    const spillContainmentSelect = document.getElementById('SpillContainment');

    /*-------------------- Capacidad del Tanque --------------------*/
    // Event Listeners
    capacidadKWSelect.addEventListener('change', (e) => {
        const kw = e.target.value;
        cargarCapacidadesTanque(kw);
    });

    // Funciones
    async function cargarCapacidadesTanque(kw) {
        try {
            const response = await fetch(`/Ventas/Ventas_EUA/Cotizador_EUA/tank-capacity/${kw}`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const capacidades = await response.json();
            tankCapacitySelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            capacidades.forEach((capacidad) => {
                const option = new Option(capacidad.description, capacidad.item);
                option.dataset.id = capacidad.item;
                tankCapacitySelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            tankCapacitySelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Tipo de Tanque --------------------*/
    // Event Listeners
    tankCapacitySelect.addEventListener('change', (e) => {
        const selectedOption = e.target.selectedOptions[0];
        const tankAutonomy = selectedOption.dataset.id;
        const kw = capacidadKWSelect.value;
        cargarTiposTanque(kw, tankAutonomy);
    });

    // Funciones
    async function cargarTiposTanque(kw, tankAutonomy) {
        try {
            const response = await fetch(
                `/Ventas/Ventas_EUA/Cotizador_EUA/tank-type/${kw}/${tankAutonomy}`,
            );
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const tipos = await response.json();
            tankTypeSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            tipos.forEach((tipo) => {
                const option = new Option(tipo.description, tipo.item);
                option.dataset.id = tipo.item;
                tankTypeSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            tankTypeSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Contención de Derrames --------------------*/
    // Funciones
    async function cargarOpcionesContencionDerrames() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/spill-containment');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const opciones = await response.json();
            spillContainmentSelect.innerHTML = `<option selected value="">${TEXT_SELECT_OPTION}</option>`;
            opciones.forEach((opcion) => {
                const option = new Option(opcion.description, opcion.item);
                spillContainmentSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar opciones de contención de derrames:', error);
            spillContainmentSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    // Inicialización

    cargarOpcionesContencionDerrames();

    /****************************************************
     * SECCIÓN 9: GESTIÓN DE SELECTBOX - ULTIMOS PASOS
     ****************************************************/

    /*-------------------- Información de Usuario y Perfil --------------------*/
    // Constantes
    const projectInput = document.getElementById('Project');
    const customerCompanyInput = document.getElementById('CustomerCompany');
    const customerAddressInput = document.getElementById('CustomerAddress');
    const customerContactInput = document.getElementById('CustomerContact');
    const customerEmailInput = document.getElementById('CustomerEmail');
    const customerTelInput = document.getElementById('CustomerTel');
    const sellerNameInput = document.getElementById('SellerName');
    const sellerEmailInput = document.getElementById('SellerEmail');
    const sellerPhoneInput = document.getElementById('SellerPhone');
    const discountInput = document.getElementById('Discount');

    // Variables globales para la sección
    let profileInfo = null;

    // Inicialización
    cargarInformacionPerfil();

    // Funciones
    async function cargarInformacionPerfil() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/get-profile-info');
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(
                    `Error en la respuesta del servidor: ${errorData.error || response.statusText}`,
                );
            }

            profileInfo = await response.json();
            console.log('Información de perfil cargada:', profileInfo);

            // Aplicar configuraciones según el tipo de usuario
            configurarFormularioSegunPerfil(profileInfo);
        } catch (error) {
            console.error('Error al cargar información de perfil:', error);
            // Mostrar mensaje al usuario con SweetAlert2
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'The profile information could not be loaded. Please try again.',
                confirmButtonText: 'Accept',
            });
        }
    }

    function configurarFormularioSegunPerfil(profileInfo) {
        if (!profileInfo) return;

        const requestType = profileInfo.RequestTypeID;

        // Configuración según tipo de solicitud
        if (requestType === 'EMPLEADO') {
            // EMPLEADO DE IGSA

            // Desbloquear campos para que el usuario los complete
            customerCompanyInput.disabled = false;
            customerAddressInput.disabled = false;
            customerContactInput.disabled = false;
            customerEmailInput.disabled = false;
            customerTelInput.disabled = false;
            discountInput.disabled = false;

            discountInput.value = '0.00';

            // Agregar validaciones para email y teléfono
            customerEmailInput.addEventListener('input', validarCorreoElectronico);
            customerTelInput.addEventListener('input', validarNumeroTelefonico);

            // Autocompletar información del vendedor (el empleado mismo)
            if (profileInfo.FirstName && profileInfo.LastName) {
                let nombreCompleto = `${profileInfo.FirstName || ''} ${profileInfo.MiddleName || ''} ${profileInfo.LastName || ''} ${profileInfo.SecondLastName || ''}`;
                // Limpiar espacios múltiples
                nombreCompleto = nombreCompleto.replace(/\s+/g, ' ').trim();
                sellerNameInput.value = nombreCompleto;
            }

            if (profileInfo.Email) {
                sellerEmailInput.value = profileInfo.Email;
            }

            if (profileInfo.ContactPhone) {
                sellerPhoneInput.value = profileInfo.ContactPhone;
            }

            // Deshabilitar campos del vendedor
            sellerNameInput.disabled = true;
            sellerEmailInput.disabled = true;
            sellerPhoneInput.disabled = true;
        } else if (requestType === 'DISEUA' || requestType === 'DISCAN') {
            // DISTRIBUIDOR EN ESTADOS UNIDOS O CANADÁ

            // Autocompletar información del cliente (distribuidor)
            if (profileInfo.BusinessName) {
                customerCompanyInput.value = profileInfo.BusinessName;
            }

            if (profileInfo.FiscalAddress) {
                customerAddressInput.value = profileInfo.FiscalAddress;
            }

            if (profileInfo.PrimaryContact) {
                customerContactInput.value = profileInfo.PrimaryContact;
            }

            if (profileInfo.Email) {
                customerEmailInput.value = profileInfo.Email;
            }

            if (profileInfo.ContactPhone) {
                customerTelInput.value = profileInfo.ContactPhone;
            }

            if (profileInfo.Discount) {
                discountInput.value = parseFloat(profileInfo.Discount).toFixed(2);
            }

            // Deshabilitar campos del cliente
            customerCompanyInput.disabled = true;
            customerAddressInput.disabled = true;
            customerContactInput.disabled = true;
            customerEmailInput.disabled = true;
            customerTelInput.disabled = true;
            discountInput.disabled = true;

            // Autocompletar información del vendedor
            if (profileInfo.SellerInfo) {
                if (profileInfo.SellerInfo.FirstName && profileInfo.SellerInfo.LastName) {
                    let nombreCompleto = `${profileInfo.SellerInfo.FirstName || ''} ${profileInfo.SellerInfo.MiddleName || ''} ${profileInfo.SellerInfo.LastName || ''} ${profileInfo.SellerInfo.SecondLastName || ''}`;
                    // Limpiar espacios múltiples
                    nombreCompleto = nombreCompleto.replace(/\s+/g, ' ').trim();
                    sellerNameInput.value = nombreCompleto;
                }

                if (profileInfo.SellerInfo.Email) {
                    sellerEmailInput.value = profileInfo.SellerInfo.Email;
                }

                if (profileInfo.SellerInfo.ContactPhone) {
                    sellerPhoneInput.value = profileInfo.SellerInfo.ContactPhone;
                }
            }

            // Deshabilitar campos del vendedor
            sellerNameInput.disabled = true;
            sellerEmailInput.disabled = true;
            sellerPhoneInput.disabled = true;
        }
    }

    // Funciones de validación
    function validarCorreoElectronico(e) {
        const email = e.target.value;
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

        if (email && !emailRegex.test(email)) {
            e.target.classList.add('is-invalid');
        } else {
            e.target.classList.remove('is-invalid');
        }
    }

    function validarNumeroTelefonico(e) {
        const telefono = e.target.value;
        // Permitir dígitos, espacios, paréntesis, guiones y signos + (formato internacional)
        const telefonoRegex = /^[0-9+\-\s()]{4,20}$/;

        if (telefono && !telefonoRegex.test(telefono)) {
            e.target.classList.add('is-invalid');
        } else {
            e.target.classList.remove('is-invalid');
        }
    }

    /*------------------ DESTINOS --------------------*/

    // Constantes

    const countrySelect = document.getElementById('Country');
    const statesSelect = document.getElementById('States');

    // Inicialización

    getCountrys();

    // Event Listeners

    // Funciones

    async function getCountrys() {
        fetch('/static/data/Utilities/countries_states_mx_us_ca.json')
            .then((response) => response.json())
            .then((data) => {
                // Se filtra méxico para que no se muestre.
                const countries = data.countries.filter((country) => country.isoCode !== 'MX');

                console.warn(data);

                //LLenar el select de countries
                countries.forEach((country) => {
                    const option = document.createElement('option');
                    option.value = country.isoCode;
                    option.text = country.name;
                    countrySelect.appendChild(option);
                });

                //Escuchar cambio de país y llenar los datos
                countrySelect.addEventListener('change', function () {
                    //Limpiar estados previos
                    statesSelect.innerHTML = `
                            <option value="">${TEXT_SELECT_STATE}</option>
                        `;

                    const selectedCountry = countries.find((c) => c.isoCode === this.value);

                    if (selectedCountry) {
                        selectedCountry.states.forEach((state) => {
                            const option = document.createElement('option');
                            option.value = state.name;
                            option.text = state.name;
                            statesSelect.appendChild(option);
                        });
                    }
                });

                //Disparar el evento para cargar los estados del primer país
                countrySelect.dispatchEvent(new Event('change'));
            });
    }

    /*-------------------- Garantías --------------------*/
    // Constantes
    const warrantySelect = document.getElementById('Garanty');

    // Inicialización
    cargarOpcionesGarantia();

    // Event Listeners
    warrantySelect.addEventListener('change', () => {
        // Aquí puedes agregar lógica adicional cuando cambie la selección de garantía
    });

    // Funciones

    async function cargarOpcionesGarantia() {
        try {
            const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/warranty-options');
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const opciones = await response.json();
            warrantySelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            opciones.forEach((opcion) => {
                const option = new Option(opcion.description, opcion.item);
                warrantySelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar opciones de garantía:', error);
            warrantySelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Pruebas --------------------*/
    // Constantes
    const testingSelect = document.getElementById('Teasting');

    // Inicialización
    //  No es necesario inicializar

    // Event Listeners
    capacidadKWSelect.addEventListener('change', (e) => {
        const kw = e.target.value;
        cargarTiposPruebas(kw);
    });

    // Funciones
    async function cargarTiposPruebas(kw) {
        try {
            const response = await fetch(`/Ventas/Ventas_EUA/Cotizador_EUA/testing-options/${kw}`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor');

            const pruebas = await response.json();
            testingSelect.innerHTML = `<option selected disabled value="">${TEXT_SELECT_OPTION}</option>`;
            pruebas.forEach((prueba) => {
                const option = new Option(prueba.description, prueba.item);
                option.dataset.id = prueba.item;
                testingSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error:', error);
            testingSelect.innerHTML = `<option disabled selected>${TEXT_ERROR_LOADING}</option>`;
        }
    }

    /*-------------------- Descuento --------------------*/

    // Event Listener para el input de descuento
    discountInput.addEventListener('input', (e) => {
        const value = e.target.value;

        // Permitir campo vacío temporalmente
        if (value === '') return;

        // Convertir a número flotante
        const numValue = parseFloat(value);

        // Si no es un número válido, limpiar el campo
        if (isNaN(numValue)) {
            e.target.value = '';
            return;
        }

        // Limitar valor entre 0.00 y 30.00
        if (numValue > 30) {
            e.target.value = 30;
        } else if (numValue < 0) {
            e.target.value = 0;
        }
    });

    // Validar al salir del campo
    discountInput.addEventListener('blur', () => {
        if (!discountInput.disabled) {
            const value = discountInput.value;

            // Si está vacío, establecer en 0
            if (value === '' || isNaN(parseFloat(value)) || parseFloat(value) < 0) {
                discountInput.value = '0.00';
            } else if (parseFloat(value) > 30) {
                discountInput.value = '30.00';
            } else {
                // Redondear a 2 decimales
                discountInput.value = parseFloat(value).toFixed(2);
            }
        }
    });

    // Prevenir caracteres no deseados
    discountInput.addEventListener('keydown', (e) => {
        // Permitir: números, punto decimal, backspace, delete, tab, escape, enter
        const allowedKeys = ['Backspace', 'Delete', 'Tab', 'Escape', 'Enter', '.'];

        // Permitir números del teclado
        if ((e.key >= '0' && e.key <= '9') || allowedKeys.includes(e.key)) {
            return;
        }

        // Bloquear cualquier otra entrada
        e.preventDefault();
    });

    /****************************************************
     * SECCIÓN 10: RECOPILAR INFORMACION DE LOS PASOS
     ****************************************************/

    // Constantes
    const formCotizacion = document.querySelector('form');

    // Event Listeners
    if (formCotizacion) {
        formCotizacion.addEventListener('submit', handleFormSubmit);
    }

    // Ejecutar cuando el DOM esté listo
    document.addEventListener('DOMContentLoaded', function () {
        // Si estamos en la página del template base, cargar los datos
        if (window.location.pathname.includes('VEUcotizadorbase.html')) {
            const quoteData = loadQuoteData();
            console.log(quoteData);
            if (quoteData) {
                // Los datos estarán disponibles en el template a través de
                // la variable 'data' que pasamos en el contexto de Jinja
            }
        }
    });

    // Funciones

    function mostrarOverlayCarga(mostrar = true) {
        let overlay = document.querySelector('.loading-overlay');
        const body = document.body;

        if (mostrar) {
            if (!overlay) {
                overlay = document.createElement('div');
                overlay.className = 'loading-overlay';
                overlay.innerHTML = `
                    <div class="spinner"></div>
                    <p>Processing quote...</p>
                    <p>Please wait, this may take a few moments.</p>
                `;
                document.body.appendChild(overlay);
            }
            overlay.style.display = 'flex';
            body.classList.add('loading-active');

            document.querySelectorAll('button').forEach((button) => {
                button.disabled = true;
            });
        } else {
            if (overlay) {
                overlay.style.display = 'none';
                body.classList.remove('loading-active');

                document.querySelectorAll('button').forEach((button) => {
                    button.disabled = false;
                });
            }
        }
    }

    async function handleFormSubmit(e) {
        e.preventDefault();

        if (!validarCamposRequeridos()) {
            return;
        }

        try {
            // Mostrar overlay antes de comenzar el proceso
            mostrarOverlayCarga(true);

            const datosCotizacion = recolectarDatosCotizacion();

        const response = await fetch('/Ventas/Ventas_EUA/Cotizador_EUA/save-quote', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(datosCotizacion),
            });

            // Procesar la respuesta
            const result = await response.json();

            // Importante: Quitamos el overlay ANTES de mostrar el Swal
            mostrarOverlayCarga(false);

            if (!response.ok) {
                throw new Error(result.error || 'Error al guardar la cotización');
            }

            if (result.success) {
                const quoteId = result.quote_id;
                console.log(quoteId);

                // Agregar logs para debug
                console.log('Resultado completo:', result);
                console.log('Data:', result.data);

                await Swal.fire({
                    icon: 'success',
                    title: 'Quote saved!',
                    text: `Quote #${quoteId} successfully saved`,
                    showCancelButton: true,
                    confirmButtonText: 'See Quote',
                    cancelButtonText: 'New Quote',
                }).then((swalResult) => {
                    // Cambié el nombre de la variable para evitar confusión
                    if (swalResult.isConfirmed) {
                        const quoteId = result.quote_id;
                        console.log(
                            'Redirigiendo a:',
                            `/Ventas/VentasEUA/VEUcotizadorbase?id=${quoteId}`,
                        );

                        // Enviar el ID de la cotización al servidor mediante un formulario oculto
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = '/Ventas/VentasEUA/VEUcotizadorbase';

                        const input = document.createElement('input');
                        input.type = 'hidden';
                        input.name = 'quote_id';
                        input.value = quoteId;
                        form.appendChild(input);

                        document.body.appendChild(form);
                        form.submit();
                    } else {
                        window.location.href = '/Ventas/Ventas_EUA/Cotizador_EUA';
                    }
                });
            }
        } catch (error) {
            mostrarOverlayCarga(false);

            console.error('Error:', error);

            // Mostrar error al usuario
            await Swal.fire({
                icon: 'error',
                title: 'Error',
                text: `Error saving quote: ${error.message}`,
                confirmButtonText: 'OK',
            });
        }
    }

    // Función auxiliar para cargar los datos en el template base
    function loadQuoteData() {
        const urlParams = new URLSearchParams(window.location.search);
        const quoteData = urlParams.get('data');

        if (quoteData) {
            try {
                const data = JSON.parse(quoteData);
                // Aquí puedes agregar lógica adicional para manipular los datos
                // antes de que el template los muestre
                return data;
            } catch (error) {
                console.error('Error parsing quote data:', error);
                return null;
            }
        }
        return null;
    }

    // Funciones
    function recolectarDatosCotizacion() {
        // Objeto final con todos los campos:
        const data = {};
        const valoresFijos = {
            Item_Sunt_t1: 'STB',
            Item_Sunt_t1: 'STB',
            Item_Sunt_w1: 'STW',
            Item_GFI1: 'GFI',
            Item_Auxiliary_c1: 'STB-1',
            Item_Breaker_L1: 'LB',
            Item_Motorized_B1: 'MB',
            Item_Sunt_t2: 'STB',
            Item_Sunt_w2: 'STW',
            Item_GFI2: 'GFI',
            Item_Auxiliary_c2: 'STB-1',
            Item_Breaker_L2: 'LB',
            Item_Motorized_B2: 'MB',
            Item_Sunt_t3: 'STB',
            Item_Sunt_w3: 'STW',
            Item_GFI3: 'GFI',
            Item_Auxiliary_c3: 'STB-1',
            Item_Breaker_L3: 'LB',
            Item_Motorized_B3: 'MB',
            Item_CtrlPanelHeater: 'CPH',
            Item_RempteDisplMod: 'RDM',
            Item_Antena: 'GSM_GSP',
            Item_DSE890: 'DSE-890',
            Item_DSE2157: 'DSE-2157',
            Item_DSE2548: 'DSE-2548',
            Item_CtrlPanelHeaterCMP: 'CPH_C',
            Item_RempteDisplModCMP: 'RDM_C',
            Item_AntenaCMP: 'GSM_GSP_C',
            Item_ComapDSE890: 'CMP-890',
            Item_ComapDSE2157: 'CMP-2157',
            Item_ComapDSE2548: 'CMP-2548',
            Item_GFCI120V: '120-REC',
            Item_Receptable120V: '240-REC',
            Item_PrewireAccsEncl: 'PW-LC',
            Item_autolouvers: 'AUT-LOV',
            Item_Receptautolouvers: 'LOV-REC',
            Item_BandHeater: 'ASH',
            Item_PMGUpdate: 'AVR-321',
            Item_DigitalRegulator: 'AVR-DM110',
            Item_VoltageRheostat: 'VAR',
            Item_ThermalWrap: 'BTW',
            Item_BaseHeater: 'BPH',
            Item_BatterySwitch: 'BDS',
            Item_BatteryHolder: 'BTR',
            Item_OilHeater: 'OPH',
            Item_OilLevelSwitch: 'OLS-M129',
            Item_CommonAlarmRelay: 'RCA-10A',
            Item_FunctionRelay: 'RRR-10A',
            Item_Coolant: 'CLN-GS',
            Item_Oil: 'OIL-GS',
            Item_DisconnectSwitches: 'HD-SWT',
            Item_RemoteEStop: 'RES-PHC',
            Item_BreakTraySwitch: 'FBS',
            Item_FuelLevelSensor: 'FLS',
            Item_OverfillValve: 'OVP',
            Item_DualFuelSwitch: 'DFS',
            Item_HighFuelSwitch: 'HFLS',
            Item_RemoteAlarmP: 'RAP',
            Item_VentPipe: '12FE',
            Item_FlammableL: 'SCL',
            Item_NFPAIdentificat: 'SNFPA',
            Item_NoSmoking: 'SNS',
            Item_TankNumber: 'STN',
            Item_StopButtonEnclosure: 'STB',
            Item_OpeningEnclousure: 'OPE-ENC',
            Item_louvers_in : 'LOV-IN',
            Item_louvers_out: 'LOV-OUT',
            
        };

        /* ──────────────────────────────────────────────────────────────────────────
                1. INFORMACIÓN GENERAL
            ────────────────────────────────────────────────────────────────────────── */
        data.PlateRating = document.getElementById('potencia')?.value || '';
        {
            const select = document.getElementById('tipoCombustible');
            data.FuelTypeIDBack = select?.selectedOptions[0]?.dataset.id || '';
            data.FuelTypeFront = select?.value || '';
        }
        data.KWOutput = parseInt(document.getElementById('capacidadKW')?.value || 0);
        {
            const select = document.getElementById('marcaMotor');
            data.EngineBrandID = select?.selectedOptions[0]?.dataset.id || '';
            data.EngineBrand = select?.value || '';
        }
        {
            const select = document.getElementById('marcaGenerador');
            data.AlternatorBrandID = select?.selectedOptions[0]?.dataset.id || '';
            data.AlternatorBrand = select?.value || '';
        }
        {
            const select = document.getElementById('voltaje');
            data.VoltageID = parseInt(select?.selectedOptions[0]?.dataset.id) || 0;
            data.Voltage_Front = select?.value || '';
        }
        {
            const select = document.getElementById('certified');
            data.CertificationID = select?.selectedOptions[0]?.dataset.id || '';
            data.Certification = select?.value || '';
        }
        data.QtyGenset = parseInt(document.getElementById('cantidadPlantas')?.value || 0);
        // Cantidad de Interruptores (en el CSV aparece como dos columnas: QtyBreakers y FrontQtyBreakers)
        {
            const select = document.getElementById('QtyBreakers');
            data.QtyBreakers = select?.value || '';
            data.FrontQtyBreakers = select?.options[select.selectedIndex]?.text || '';
        }

        /* ──────────────────────────────────────────────────────────────────────────
                2. INTERRUPTORES
            ────────────────────────────────────────────────────────────────────────── */

        // ── Interruptor 1 ──
        {
            const select = document.getElementById('interruptor1');
            data.Item_breaker1 = select?.value || '';
            data.Breaker1 = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('preparacionDerivacion');
            data.Item_Sunt_t1 = valoresFijos['Item_Sunt_t1'];
            data.Shunt_trip1 = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('cableadoDerivacion');
            data.Item_Sunt_w1 = valoresFijos['Item_Sunt_w1'];
            data.Shunt_trip_wiring1 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('gfiFalla');
            data.Item_GFI1 = valoresFijos['Item_GFI1'];
            data.GFI1 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('lsigDisparo');
            data.Item_Proyection_t1 = select?.value || '';
            data.Protection_type1 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('contactosAuxiliares');
            data.Item_Auxiliary_c1 = valoresFijos['Item_Auxiliary_c1'];
            data.Auxiliary_contacts1 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerLock');
            data.Item_Breaker_L1 = valoresFijos['Item_Breaker_L1'];
            data.Breaker_Lock1 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerMoto');
            data.Item_Motorized_B1 = valoresFijos['Item_Motorized_B1'];
            data.Motorized_Breaker1 = select?.options[select.selectedIndex]?.text || '';
        }

        // ── Interruptor 2 ──
        {
            const select = document.getElementById('interruptor2');
            data.Item_breaker2 = select?.value || '';
            data.Breaker2 = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('preparacionDerivacion2');
            data.Item_Sunt_t2 = valoresFijos['Item_Sunt_t2'];
            data.Shunt_trip2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('cableadoDerivacion2');
            data.Item_Sunt_w2 = valoresFijos['Item_Sunt_w2'];
            data.Shunt_trip_wiring2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('gfiFalla2');
            data.Item_GFI2 = valoresFijos['Item_GFI2'];
            data.GFI2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('lsigDisparo2');
            data.Item_Proyection_t2 = select?.value || '';
            data.Protection_type2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('contactosAuxiliares2');
            data.Item_Auxiliary_c2 = valoresFijos['Item_Auxiliary_c2'];
            data.Auxiliary_contacts2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerLock2');
            data.Item_Breaker_L2 = valoresFijos['Item_Breaker_L2'];
            data.Breaker_Lock2 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerMoto2');
            data.Item_Motorized_B2 = valoresFijos['Item_Motorized_B2'];
            data.Motorized_Breaker2 = select?.options[select.selectedIndex]?.text || '';
        }

        // ── Interruptor 3 ──
        {
            const select = document.getElementById('interruptor3');
            data.Item_breaker3 = select?.value || '';
            data.Breaker3 = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('preparacionDerivacion3');
            data.Item_Sunt_t3 = valoresFijos['Item_Sunt_t3'];
            data.Shunt_trip3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('cableadoDerivacion3');
            data.Item_Sunt_w3 = valoresFijos['Item_Sunt_w3'];
            data.Shunt_trip_wiring3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('gfiFalla3');
            data.Item_GFI3 = valoresFijos['Item_GFI3'];
            data.GFI3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('lsigDisparo3');
            data.Item_Proyection_t3 = select?.value || '';
            data.Protection_type3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('contactosAuxiliares3');
            data.Item_Auxiliary_c3 = valoresFijos['Item_Auxiliary_c3'];
            data.Auxiliary_contacts3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerLock3');
            data.Item_Breaker_L3 = valoresFijos['Item_Breaker_L3'];
            data.Breaker_Lock3 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('breakerMoto3');
            data.Item_Motorized_B3 = valoresFijos['Item_Motorized_B3'];
            data.Motorized_Breaker3 = select?.options[select.selectedIndex]?.text || '';
        }

        /* ──────────────────────────────────────────────────────────────────────────
                3. CONTROL (Incluye Panel, Módulos DSE/COMAP, Cargadores, etc.)
            ────────────────────────────────────────────────────────────────────────── */

        {
            const select = document.getElementById('Controlbrand');
            data.ControllerBrandID = select?.selectedOptions[0]?.dataset.id || '';
            data.Controlbrand = select?.value || '';
        }
        {
            const select = document.getElementById('ControlModel');
            data.Item_controller = select?.value || '';
            data.ControlModel = select?.options[select.selectedIndex]?.text || '';
        }

        // Módulos  DEEPSEA
        {
            const select = document.getElementById('CtrlPanelHeater');
            data.Item_CtrlPanelHeater = valoresFijos['Item_CtrlPanelHeater'];
            data.CtrlPanelHeater = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('RemoteDisplayModule');
            data.Item_RempteDisplMod = valoresFijos['Item_RempteDisplMod'];
            data.RemoteDisplayModule = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('Antena');
            data.Item_Antena = valoresFijos['Item_Antena'];
            data.Antena = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('DSE890');
            data.Item_DSE890 = valoresFijos['Item_DSE890'];
            data.DSE890 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('DSE2157');
            data.Item_DSE2157 = valoresFijos['Item_DSE2157'];
            data.DSE2157 = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyDSE2157 = parseInt(document.getElementById('QtyDSE2157')?.value || 0);
        {
            const select = document.getElementById('DSE2548');
            data.Item_DSE2548 = valoresFijos['Item_DSE2548'];
            data.DSE2548 = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyDSE2548 = parseInt(document.getElementById('QtyDSE2548')?.value || 0);

        // Módulos COMAP equivalentes

        {
            const select = document.getElementById('CtrlPanelHeaterCMP');
            data.Item_CtrlPanelHeaterCMP = valoresFijos['Item_CtrlPanelHeaterCMP'];
            data.CtrlPanelHeaterCMP = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('RemoteDisplayModuleCMP');
            data.Item_RempteDisplModCMP = valoresFijos['Item_RempteDisplModCMP'];
            data.RemoteDisplayModuleCMP = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('AntenaCMP');
            data.Item_AntenaCMP = valoresFijos['Item_AntenaCMP'];
            data.AntenaCMP = select?.options[select.selectedIndex]?.text || '';
        }

        {
            const select = document.getElementById('ComapDSE890');
            data.Item_ComapDSE890 = valoresFijos['Item_ComapDSE890'];
            data.ComapDSE890 = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('ComapDSE2157');
            data.Item_ComapDSE2157 = valoresFijos['Item_ComapDSE2157'];
            data.ComapDSE2157 = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyComapDSE2157 = parseInt(document.getElementById('QtyComapDSE2157')?.value || 0);
        {
            const select = document.getElementById('ComapDSE2548');
            data.Item_ComapDSE2548 = valoresFijos['Item_ComapDSE2548'];
            data.ComapDSE2548 = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyComapDSE2548 = parseInt(document.getElementById('QtyComapDSE2548')?.value || 0);
        // Voltaje de Operación y Cargador de Batería
        {
            const select = document.getElementById('VotlOper');
            data.Item_CtrlVotlOper = select?.value || '';
            data.CtrlVotlOper = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('BateryCharger');
            data.Item_BateryCharger = select?.value || '';
            data.BateryCharger = select?.options[select.selectedIndex]?.text || '';
        }

        /* ──────────────────────────────────────────────────────────────────────────
                4. CASETA
            ────────────────────────────────────────────────────────────────────────── */

        {
            const select = document.getElementById('Enclosure');
            data.Item_Enclosuree_type = select?.value || '';
            data.Enclosure = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('EnclosureType');
            data.Item_Eclousure_Mat = select?.value || '';
            data.EnclosureType = select?.options[select.selectedIndex]?.text || '';
        }
        // GFCI 120V
        {
            const select = document.getElementById('GFCI120V');
            data.Item_GFCI120V = valoresFijos['Item_GFCI120V'];
            data.GFCI120V = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyGFCI120V = parseInt(document.getElementById('QtyGFCI120V')?.value || 0);
        // Receptacle 120V
        {
            const select = document.getElementById('Receptable120V');
            data.Item_Receptable120V = valoresFijos['Item_Receptable120V'];
            data.Receptable120V = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyReceptable120V = parseInt(document.getElementById('QtyReceptable120V')?.value || 0);
        // Luces AC
        {
            const select = document.getElementById('lightAC');
            data.Item_Enclosure_ligthAC = select?.value || '';

            data.lightAC = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtylightAC = parseInt(document.getElementById('QtylightAC')?.value || 0);
        // Luces DC
        {
            const select = document.getElementById('lightDC');
            data.Item_Enclosure_ligthDC = select?.value || '';
            data.lightDC = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtylightDC = parseInt(document.getElementById('QtylightDC')?.value || 0);
        // Calentador de Espacio
        {
            const select = document.getElementById('SpaceHeater');
            data.Item_Enclosure_SHeater = select?.value || '';
            data.SpaceHeater = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtySpaceHeater = parseInt(document.getElementById('QtySpaceHeater')?.value || 0);
        // LoadCenter
        {
            const select = document.getElementById('LoadCenter');
            data.Item_Enclosure_LCenter = select?.value || '';
            data.LoadCenter = select?.options[select.selectedIndex]?.text || '';
        }
        // Pre-cablear Accesorios
        {
            const select = document.getElementById('PrewireAccsEnclousure');
            data.Item_PrewireAccsEncl = valoresFijos['Item_PrewireAccsEncl'];
            data.PrewireAccsEnclousure = select?.options[select.selectedIndex]?.text || '';
        }
        // Persianas automáticas
        {
            const select = document.getElementById('autolouvers');
            data.Item_autolouvers = valoresFijos['Item_autolouvers'];
            data.autolouvers = select?.options[select.selectedIndex]?.text || '';
        }
        // Contacto de falla persianas
        {
            const select = document.getElementById('Receptautolouvers');
            data.Item_Receptautolouvers = valoresFijos['Item_Receptautolouvers'];
            data.Receptautolouvers = select?.options[select.selectedIndex]?.text || '';
        }
        {
            data.Item_OpeningEnclousure = valoresFijos['Item_OpeningEnclousure']; // Apertura interna
            data.OpeningEnclousure = select_OpeningEnclousure.value;
        }
        {
            data.Item_StopButtonEnclosure = valoresFijos['Item_StopButtonEnclosure']; // Boton de paro
            data.StopButtonEnclosure = select_StopButtonEnclosure.value;
            data.StopButtonEnclosure_Qty = input_StopButtonEnclosure_Qty.value;
        }
        {
            data.Item_louvers_in = valoresFijos['Item_louvers_in']; // lovers in
            data.louvers_in = select_louvers_in.value
        }
        {
            data.Item_louvers_out = valoresFijos['Item_louvers_out']; // lovers in
            data.louvers_out = select_louvers_out.value
        }
        {
            data.Item_StopButtonEnclosure = valoresFijos['Item_StopButtonEnclosure']; // Boton de paro
            data.StopButtonEnclosure = select_StopButtonEnclosure.value;
            data.StopButtonEnclosure_Qty = input_StopButtonEnclosure_Qty.value;
        }
        

        /* ──────────────────────────────────────────────────────────────────────────
                5. ACCESORIOS
            ────────────────────────────────────────────────────────────────────────── */

        {
            const select = document.getElementById('BandHeater');
            data.Item_BandHeater = valoresFijos['Item_BandHeater'];
            data.BandHeater = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('PMGUpdate');
            data.Item_PMGUpdate = valoresFijos['Item_PMGUpdate'];
            data.PMGUpdate = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('DigitalRegulator');
            data.Item_DigitalRegulator = valoresFijos['Item_DigitalRegulator'];
            data.DigitalRegulator = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('VoltageRheostat');
            data.Item_VoltageRheostat = valoresFijos['Item_VoltageRheostat'];
            data.VoltageRheostat = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('ThermalWrap');
            data.Item_ThermalWrap = valoresFijos['Item_ThermalWrap'];
            data.ThermalWrap = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('BaseHeater');
            data.Item_BaseHeater = valoresFijos['Item_BaseHeater'];
            data.BaseHeater = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('BatterySwitch');
            data.Item_BatterySwitch = valoresFijos['Item_BatterySwitch'];
            data.BatterySwitch = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('BatteryHolder');
            data.Item_BatteryHolder = valoresFijos['Item_BatteryHolder'];
            data.BatteryHolder = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('OilHeater');
            data.Item_OilHeater = valoresFijos['Item_OilHeater'];
            data.OilHeater = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('OilLevelSwitch');
            data.Item_OilLevelSwitch = valoresFijos['Item_OilLevelSwitch'];
            data.OilLevelSwitch = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('CommonAlarmRelay');
            data.Item_CommonAlarmRelay = valoresFijos['Item_CommonAlarmRelay'];
            data.CommonAlarmRelay = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('FunctionRelay');
            data.Item_FunctionRelay = valoresFijos['Item_FunctionRelay'];
            data.FunctionRelay = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('Coolant');
            data.Item_Coolant = valoresFijos['Item_Coolant'];
            data.Coolant = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('Oil');
            data.Item_Oil = valoresFijos['Item_Oil'];
            data.Oil = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('DisconnectSwitches');
            data.Item_DisconnectSwitches = valoresFijos['Item_DisconnectSwitches'];
            data.DisconnectSwitches = select?.options[select.selectedIndex]?.text || '';
        }

        // Paro de emergencia remoto
        {
            const select = document.getElementById('EStop');
            data.Item_AddAccesory_Estop = select?.value || '';
            data.EStop = select?.options[select.selectedIndex]?.text || '';
        }
        data.QtyEStop = parseInt(document.getElementById('QtyEStop')?.value || 0);
        {
            const select = document.getElementById('RemoteEStop');
            data.Item_RemoteEStop = valoresFijos['Item_RemoteEStop'];
            data.RemoteEStop = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('SpringIsolator');
            data.Item_AddAccesory_Spring = select?.value || '';
            data.SpringIsolator = select?.options[select.selectedIndex]?.text || '';
        }

        /* ──────────────────────────────────────────────────────────────────────────
                6. TANQUE
            ────────────────────────────────────────────────────────────────────────── */

        {
            const select = document.getElementById('TankCapacity');
            data.Item_Tank_Autonomy = select?.value || '';
            data.TankCapacity = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('TankType');
            data.Item_Tank_type = select?.value || '';
            data.TankType = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('BreakTraySwitch');
            data.Item_BreakTraySwitch = valoresFijos['Item_BreakTraySwitch'];
            data.BreakTraySwitch = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('FuelLevelSensor');
            data.Item_FuelLevelSensor = valoresFijos['Item_FuelLevelSensor'];
            data.FuelLevelSensor = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('SpillContainment');
            data.Item_Tank_SpillCont = select?.value || '';
            data.SpillContainment = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('OverfillValve');
            data.Item_OverfillValve = valoresFijos['Item_OverfillValve'];
            data.OverfillValve = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('DualFuelSwitches');
            data.Item_DualFuelSwitch = valoresFijos['Item_DualFuelSwitch'];
            data.DualFuelSwitches = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('HighFuelSwitch');
            data.Item_HighFuelSwitch = valoresFijos['Item_HighFuelSwitch'];
            data.HighFuelSwitch = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('RemoteAlarmPanel');
            data.Item_RemoteAlarmP = valoresFijos['Item_RemoteAlarmP'];
            data.RemoteAlarmPanel = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('VentPipe');
            data.Item_VentPipe = valoresFijos['Item_VentPipe'];
            data.VentPipe = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('FlammableLiquids');
            data.Item_FlammableL = valoresFijos['Item_FlammableL'];
            data.FlammableLiquids = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('NFPAIdentification');
            data.Item_NFPAIdentificat = valoresFijos['Item_NFPAIdentificat'];
            data.NFPAIdentification = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('NoSmoking');
            data.Item_NoSmoking = valoresFijos['Item_NoSmoking'];
            data.NoSmoking = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('TankNumber');
            data.Item_TankNumber = valoresFijos['Item_TankNumber'];
            data.TankNumber = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('FluidContainment');
            data.Item_FluidContainment = valoresFijos['Item_FluidContainment'];
            data.FluidContainment = select?.options[select.selectedIndex]?.text || '';
        }

        /* ──────────────────────────────────────────────────────────────────────────
                7. PROYECTO
            ────────────────────────────────────────────────────────────────────────── */

        data.Project = document.getElementById('Project')?.value || '';
        {
            const select = document.getElementById('Garanty');
            data.Item_PrjWar_Warranty = select?.value || '';
            data.Garanty = select?.options[select.selectedIndex]?.text || '';
        }
        {
            const select = document.getElementById('Teasting');
            data.Item_PrjWar_Testing = select?.value || '';
            data.Teasting = select?.options[select.selectedIndex]?.text || '';
        }

        data.Discount = document.getElementById('Discount')?.value || '';

        data.ReqCustomer = document.getElementById('ReqCustomer')?.value || '';

        data.CustomerCompany = document.getElementById('CustomerCompany')?.value || '';

        data.CustomerAddress = document.getElementById('CustomerAddress')?.value || '';

        data.CustomerContact = document.getElementById('CustomerContact')?.value || '';

        data.CustomerEmail = document.getElementById('CustomerEmail')?.value || '';

        data.CustomerTel = document.getElementById('CustomerTel')?.value || '';

        data.SellerName = document.getElementById('SellerName')?.value || '';

        data.SellerEmail = document.getElementById('SellerEmail')?.value || '';

        data.SellerPhone = document.getElementById('SellerPhone')?.value || '';
        data.RoutePDF = '';

        //Nuevos valores

        data.Destiny_Country = document.getElementById('Country')?.value || '';
        data.Destiny_State = document.getElementById('States')?.value || '';

        return data;
    }

    /****************************************************
     * SECCIÓN 11: VALIDACIONES
     ****************************************************/

    // Función para validar campos del paso actual
    function validarCamposPaso(pasoActual) {
        const formStepActual = formSteps[pasoActual];
        const camposRequeridos = formStepActual.querySelectorAll('.required');
        let esValido = true;
        let primerCampoInvalido = null;

        // Limpiar validaciones previas
        document.querySelectorAll('.campo-invalido').forEach((campo) => {
            campo.classList.remove('campo-invalido');
        });

        camposRequeridos.forEach((campo) => {
            const input = document.getElementById(campo.getAttribute('for'));
            if (!input || !input.value || input.value.trim() === '') {
                esValido = false;
                campo.classList.add('campo-invalido');
                if (!primerCampoInvalido) primerCampoInvalido = campo;
            }
        });

        if (!esValido) {
            if (primerCampoInvalido) {
                primerCampoInvalido.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }

            // Usar SweetAlert2 en lugar de alert
            Swal.fire({
                icon: 'warning',
                title: '<p>Required fields</p>',
                text: '<p>Please wait, this may take a few moments.</p>',
                confirmButtonText: 'Understood',
            });
        }

        return esValido;
    }

    // Función para validar todos los campos (usado en el submit final)
    function validarCamposRequeridos() {
        let esValido = true;

        // Validar cada paso
        for (let i = 0; i < formSteps.length; i++) {
            const formStep = formSteps[i];
            const camposRequeridos = formStep.querySelectorAll('.required');

            camposRequeridos.forEach((campo) => {
                const input = document.getElementById(campo.getAttribute('for'));
                if (!input || !input.value || input.value.trim() === '') {
                    esValido = false;
                    campo.classList.add('campo-invalido');
                    // Si encontramos un campo inválido, cambiar al paso correspondiente
                    if (currentStep !== i) {
                        currentStep = i;
                        updateSteps();
                    }
                    // Salir del bucle después de encontrar el primer paso con campos inválidos
                    return;
                }
            });

            if (!esValido) break; // Salir si encontramos campos inválidos
        }

        if (!esValido) {
            Swal.fire({
                icon: 'warning',
                title: 'Required fields',
                text: 'Please complete all required fields before submitting the quote.',
                confirmButtonText: 'Understood',
            });
        }

        return esValido;
    }

    /****************************************************
     * SECCIÓN 11: INICIALIZACION GRAL
     ****************************************************/
    // Inicializar
    updateSteps();
}); // Fin del DOMContentLoaded
