/*
Solicitud de alta
Archivo: solicitud_de_alta.js
Ruta: src\static\js\solicitud_de_alta.js
Lenguaje: Java Script
*/
function agregarEstilosOverlay() {
    // Verificar si ya existe el estilo
    if (!document.getElementById('overlay-styles')) {
        const style = document.createElement('style');
        style.id = 'overlay-styles';
        style.innerHTML = `
            .page-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 9999;
                display: flex;
                justify-content: center;
                align-items: center;
                flex-direction: column;
            }
            .spinner-container {
                text-align: center;
                color: white;
            }
            .spinner-container .spinner-border {
                width: 3rem;
                height: 3rem;
            }
            .spinner-text {
                margin-top: 15px;
                font-size: 18px;
                font-weight: bold;
            }
        `;
        document.head.appendChild(style);
    }
}

function mostrarSpinner(mensaje = "Actualizando fichas técnicas...") {
    agregarEstilosOverlay();
    
    // Crear el overlay
    const overlay = document.createElement('div');
    overlay.className = 'page-overlay';
    overlay.id = 'actualizacion-overlay';
    
    // Crear el spinner y el mensaje
    overlay.innerHTML = `
        <div class="spinner-container">
            <div class="spinner-border text-light" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <div class="spinner-text">${mensaje}</div>
        </div>
    `;
    
    // Agregar a la página
    document.body.appendChild(overlay);
    
    // Bloquear el scroll
    document.body.style.overflow = 'hidden';
}

function ocultarSpinner() {
    const overlay = document.getElementById('actualizacion-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = 'auto';
    }
}
document.addEventListener('DOMContentLoaded', function() {
    // Selectores para la gestión de visibilidad de secciones
    const RequestTypeSelect = document.getElementById("RequestType");
    const seccionFormulario = document.querySelectorAll(".seccionFormulario");
    
    // Mapeos de Opciones para tipo de registro
    const opcionderegistro = {
        "DISCAN": [1,3],
        "DISEUA": [1,3],
        "DISMÉX": [2,3],
        "EMPLEADO": [0,3]
    };

    // Función para desactivar/activar required en los formularios
    function toggleRequiredFields() {
        // Primero, guardar el estado original de los campos required si no se ha hecho
        document.querySelectorAll('input[required], select[required], textarea[required]').forEach(element => {
            if (!element.hasAttribute('data-required')) {
                element.setAttribute('data-required', 'true');
            }
        });

        // Quitar required de todos los campos
        document.querySelectorAll('[data-required="true"]').forEach(element => {
            element.removeAttribute('required');
        });

        // Activar required solo en el formulario visible
        const selectedType = RequestTypeSelect.value;
        if (selectedType && opcionderegistro[selectedType]) {
            opcionderegistro[selectedType].forEach(index => {
                if (index !== 3) { // No aplicar a la sección del botón
                    const section = seccionFormulario[index];
                    if (section) {
                        section.querySelectorAll('[data-required="true"]').forEach(element => {
                            element.setAttribute('required', '');
                        });
                    }
                }
            });
        }
    }

    // Event Listener para tipo de registro
    if (RequestTypeSelect) {
        RequestTypeSelect.addEventListener('change', () => {
            actualizarSecciones(RequestTypeSelect, seccionFormulario, opcionderegistro);
            toggleRequiredFields();
            // Marcar campos requeridos después de actualizar las secciones
            const selectedType = RequestTypeSelect.value;
            let formId;
            switch(selectedType) {
                case 'EMPLEADO':
                    formId = 'Colaboradores-form';
                    break;
                case 'DISEUA':
                case 'DISCAN':
                    formId = 'Distribuidores_Estados_Unidos-form';
                    break;
                case 'DISMÉX':
                    formId = 'Distribuidores_Mexico-form';
                    break;
            }
            if (formId) {
                markRequiredFields(formId);
            }
        });
        
        // Inicialización
        actualizarSecciones(RequestTypeSelect, seccionFormulario, opcionderegistro);
        toggleRequiredFields();
    }

    // Marcar campos requeridos inicialmente para todos los formularios
    ['Colaboradores-form', 'Distribuidores_Estados_Unidos-form', 'Distribuidores_Mexico-form'].forEach(formId => {
        markRequiredFields(formId);
    });

    // Función para actualizar secciones
    function actualizarSecciones(selectElement, secciones, opciones) {
        // Ocultar todas las secciones primero
        secciones.forEach(seccion => {
            seccion.style.display = 'none';
        });

        const seleccion = selectElement.value;
        // Mostrar sección correspondiente si hay una selección
        if (seleccion && opciones[seleccion]) {
            opciones[seleccion].forEach(index => {
                if (secciones[index]) {
                    secciones[index].style.display = 'block';
                }
            });
        }
    }

    // Selectores para la gestión de compañía/división/departamento
    const companySelect = document.getElementById('EMPCompanyID');
    const divisionSelect = document.getElementById('EMPDivisionID');
    const departmentSelect = document.getElementById('EMPDepartamentID');
    
    // Indicadores de carga
    const loadingDivision = document.getElementById('loading-indicator-division');
    const loadingDepartment = document.getElementById('loading-indicator-department');

    // Event Listeners para compañía y división
    if (companySelect) {
        companySelect.addEventListener('change', handleCompanyChange);
        console.log('Listener agregado a companySelect');
    }
    if (divisionSelect) {
        divisionSelect.addEventListener('change', handleDivisionChange);
        console.log('Listener agregado a divisionSelect');
    }

    // Manejador de cambio de compañía
    async function handleCompanyChange(event) {
        console.log('handleCompanyChange iniciado');
        const companyId = event.target.value;
        console.log('CompanyID seleccionado:', companyId);
        
        if (!companyId) {
            console.log('No se seleccionó ninguna compañía');
            return;
        }

        resetSelect(divisionSelect);
        resetSelect(departmentSelect);

        try {
            if (loadingDivision) loadingDivision.style.display = 'block';
            mostrarSpinner();
            console.log('Haciendo fetch a:', `/get_divisions/${companyId}`);
            const response = await fetch(`/get_divisions/${companyId}`);
            
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || `Error al obtener divisiones (${response.status})`);
            }

            const data = await response.json();
            console.log('Datos de divisiones recibidos:', data);
            populateSelect(divisionSelect, data);
            ocultarSpinner();
        } catch (error) {
            console.error('Error en handleCompanyChange:', error);
            if (divisionSelect) {
                divisionSelect.innerHTML = `<option value="" disabled selected>Error: ${error.message}</option>`;
            }
        } finally {
            if (loadingDivision) loadingDivision.style.display = 'none';
        }
    }

    // Manejador de cambio de división
    async function handleDivisionChange(event) {
        console.log('handleDivisionChange iniciado');
        const divisionId = event.target.value;
        const companyId = companySelect.value;
        console.log('DivisionID:', divisionId, 'CompanyID:', companyId);

        if (!divisionId || !companyId) {
            console.log('Falta CompanyID o DivisionID');
            return;
        }

        resetSelect(departmentSelect);

        try {
            if (loadingDepartment) loadingDepartment.style.display = 'block';
            
            console.log('Haciendo fetch a:', `/get_departments/${companyId}/${divisionId}`);
            const response = await fetch(`/get_departments/${companyId}/${divisionId}`);
            
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || `Error al obtener departamentos (${response.status})`);
            }

            const data = await response.json();
            console.log('Datos de departamentos recibidos:', data);
            populateSelect(departmentSelect, data);
        } catch (error) {
            console.error('Error en handleDivisionChange:', error);
            if (departmentSelect) {
                departmentSelect.innerHTML = `<option value="" disabled selected>Error: ${error.message}</option>`;
            }
        } finally {
            if (loadingDepartment) loadingDepartment.style.display = 'none';
        }
    }

    // Función auxiliar para resetear un select
    function resetSelect(selectElement) {
        if (selectElement) {
            selectElement.innerHTML = '<option value="" disabled selected>Selecciona una opción</option>';
        }
    }

    // Función auxiliar para poblar un select
    function populateSelect(selectElement, items) {
        if (!selectElement) return;
        
        selectElement.innerHTML = '<option value="" disabled selected>Selecciona una opción</option>';
        
        if (Array.isArray(items)) {
            items.forEach(item => {
                const option = document.createElement('option');
                if (item.id) {
                    option.value = item.id;
                    option.textContent = item.name;
                } else {
                    option.value = item.DivisionID || item.DepartamentID || item.CompanyID;
                    option.textContent = item.Division || item.Departament || item.Company;
                }
                selectElement.appendChild(option);
            });
        } else {
            console.error('Los datos recibidos no son un array:', items);
        }
    }

    // Añadir estilo CSS para los asteriscos rojos
    const style = document.createElement('style');
    style.textContent = `
        label.required-field::after {
            content: "*";
            color: red !important;
            margin-left: 3px;
            display: inline-block;
        }
        label.filled-field::after {
            display: none !important;
        }
        .error-input {
            border-color: red !important;
        }
        .error-message {
            color: red;
            font-size: 0.8em;
            margin-top: 5px;
        }
    `;
    document.head.appendChild(style);

    // Marcar campos requeridos con asterisco rojo
    function markRequiredFields(formId) {
        const form = document.getElementById(formId);
        if (!form) return;

        const requiredInputs = form.querySelectorAll('[required]');
        requiredInputs.forEach(input => {
            const label = input.previousElementSibling;
            if (label && label.tagName === 'LABEL') {
                label.classList.add('required-field');
                
                input.addEventListener('input', function() {
                    validateFieldAndUpdateAsterisk(input, label);
                });

                validateFieldAndUpdateAsterisk(input, label);
            }
        });
    }

    // Validar campo y actualizar asterisco
    function validateFieldAndUpdateAsterisk(input, label) {
        if (validateInput(input)) {
            label.classList.add('filled-field');
        } else {
            label.classList.remove('filled-field');
        }
    }

    // Función para validar teléfono
    function validatePhoneNumber(phone) {
        // Eliminar espacios, guiones, paréntesis y puntos
        var cleanPhone = phone.replace(/[\s\-\(\)\.]/g, '');
        
        // Verificar que solo contenga dígitos y que tenga al menos 4 dígitos
        return /^\d{4,}$/.test(cleanPhone);
    }

    // Función para validar email
    function validateEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    // Validar formulario según el tipo seleccionado
    function validateForm(event) {
        event.preventDefault();
        
        const requestType = RequestTypeSelect.value;
        if (!requestType) {
            Swal.fire({
                title: 'Error',
                text: 'Por favor seleccione un tipo de usuario',
                icon: 'error'
            });
            return;
        }

        let formId;
        let phoneField;
        let emailField;
        switch(requestType) {
            case 'EMPLEADO':
                formId = 'Colaboradores-form';
                phoneField = 'EMPContactPhone';
                emailField = 'EMPCorporateEmail';
                break;
            case 'DISEUA':
            case 'DISCAN':
                formId = 'Distribuidores_Estados_Unidos-form';
                phoneField = 'EU_ContactPhone';
                emailField = 'EU_ContactEmail';
                break;
            case 'DISMÉX':
                formId = 'Distribuidores_Mexico-form';
                phoneField = 'MX_ContactPhone';
                emailField = 'MX_ContactEmail';
                break;
        }

        clearErrors(formId);

        const form = document.getElementById(formId);
        if (!form) return;

        let isValid = true;
        
        // Validar campos requeridos
        form.querySelectorAll('[required]').forEach(input => {
            if (!validateInput(input)) {
                isValid = false;
                showError(input, 'Este campo es requerido');
            }
        });

        // Validar específicamente el teléfono
        const phoneInput = document.getElementById(phoneField);
        if (phoneInput && !validatePhoneNumber(phoneInput.value)) {
            isValid = false;
            showError(phoneInput, 'Número de teléfono no válido. Debe contener al menos 10 dígitos.');
            Swal.fire({
                title: 'Error',
                text: 'Número de teléfono no válido. Debe contener al menos 10 dígitos.',
                icon: 'error'
            });
            return;
        }

        // Validar específicamente el email
        const emailInput = document.getElementById(emailField);
        if (emailInput && !validateEmail(emailInput.value)) {
            isValid = false;
            showError(emailInput, 'Formato de correo electrónico no válido.');
            Swal.fire({
                title: 'Error',
                text: 'Formato de correo electrónico no válido.',
                icon: 'error'
            });
            return;
        }

        // Validar formato de RFC para México
        if (requestType === 'DISMÉX') {
            const rfcInput = document.getElementById('MX_RFC');
            if (rfcInput) {
                const rfcPattern = /^[A-ZÑ&]{3,4}[0-9]{6}[A-V1-9][A-Z1-9][0-9A]$/;
                if (!rfcPattern.test(rfcInput.value)) {
                    isValid = false;
                    showError(rfcInput, 'Formato de RFC no válido.');
                    Swal.fire({
                        title: 'Error',
                        text: 'Formato de RFC no válido.',
                        icon: 'error'
                    });
                    return;
                }
            }
        }

        // Validar términos y condiciones
        const termsMapping = {
            'EMPLEADO': 'EMPTerms',
            'DISEUA': 'EU_Terms',
            'DISCAN': 'EU_Terms',
            'DISMÉX': 'MX_Terms'
        };

        const termsCheckbox = document.getElementById(termsMapping[requestType]);
        if (termsCheckbox && !termsCheckbox.checked) {
            isValid = false;
            Swal.fire({
                title: 'Error',
                text: 'Debe aceptar los términos y condiciones',
                icon: 'error'
            });
            return;
        }

        if (isValid) {
            submitForm(formId);
        }
    }

    // Validar un input específico
    function validateInput(input) {
        if (input.type === 'email') {
            return validateEmail(input.value);
        } else if (input.type === 'tel' || input.id.includes('ContactPhone')) {
            return validatePhoneNumber(input.value);
        } else if (input.tagName === 'SELECT') {
            return input.value !== '' && input.value !== 'null' && input.value !== 'undefined';
        } else if (input.type === 'checkbox') {
            return input.checked;
        }
        return input.value.trim() !== '';
    }

    // Mostrar mensaje de error
    function showError(input, message) {
        input.classList.add('error-input');
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.textContent = message;
        input.parentNode.appendChild(errorDiv);
    }

    // Limpiar errores
    function clearErrors(formId) {
        const form = document.getElementById(formId);
        if (!form) return;

        const errorInputs = form.querySelectorAll('.error-input');
        const errorMessages = form.querySelectorAll('.error-message');
        
        errorInputs.forEach(input => input.classList.remove('error-input'));
        errorMessages.forEach(msg => msg.remove());
    }

    // Agregar validación en tiempo real para teléfonos
    document.querySelectorAll('input[id$="ContactPhone"]').forEach(function(input) {
        input.addEventListener('input', function() {
            const phoneValue = this.value;
            const isValid = validatePhoneNumber(phoneValue);
            
            if (isValid) {
                this.classList.remove('error-input');
                // Eliminar mensaje de error si existe
                const errorMsg = this.parentNode.querySelector('.error-message');
                if (errorMsg) errorMsg.remove();
            } else {
                this.classList.add('error-input');
                // Añadir mensaje de error si no existe
                if (!this.parentNode.querySelector('.error-message')) {
                    const errorDiv = document.createElement('div');
                    errorDiv.className = 'error-message';
                    errorDiv.textContent = 'Formato de teléfono inválido. Debe contener al menos 4 dígitos.';
                    this.parentNode.appendChild(errorDiv);
                }
            }
        });
    });

    // Agregar validación en tiempo real para email
    document.querySelectorAll('input[type="email"]').forEach(function(input) {
        input.addEventListener('input', function() {
            const emailValue = this.value;
            const isValid = validateEmail(emailValue);
            
            if (isValid) {
                this.classList.remove('error-input');
                // Eliminar mensaje de error si existe
                const errorMsg = this.parentNode.querySelector('.error-message');
                if (errorMsg) errorMsg.remove();
            } else {
                this.classList.add('error-input');
                // Añadir mensaje de error si no existe
                if (!this.parentNode.querySelector('.error-message')) {
                    const errorDiv = document.createElement('div');
                    errorDiv.className = 'error-message';
                    errorDiv.textContent = 'Formato de correo electrónico inválido.';
                    this.parentNode.appendChild(errorDiv);
                }
            }
        });
    });

    // Agregar validación en tiempo real para RFC (México)
    const rfcInput = document.getElementById('MX_RFC');
    if (rfcInput) {
        rfcInput.addEventListener('input', function() {
            const rfcValue = this.value;
            const rfcPattern = /^[A-ZÑ&]{3,4}[0-9]{6}[A-V1-9][A-Z1-9][0-9A]$/;
            const isValid = rfcPattern.test(rfcValue);
            
            if (isValid) {
                this.classList.remove('error-input');
                // Eliminar mensaje de error si existe
                const errorMsg = this.parentNode.querySelector('.error-message');
                if (errorMsg) errorMsg.remove();
            } else {
                this.classList.add('error-input');
                // Añadir mensaje de error si no existe
                if (!this.parentNode.querySelector('.error-message')) {
                    const errorDiv = document.createElement('div');
                    errorDiv.className = 'error-message';
                    errorDiv.textContent = 'Formato de RFC inválido.';
                    this.parentNode.appendChild(errorDiv);
                }
            }
        });
    }


    function submitForm(formId) {
        const requestType = RequestTypeSelect.value;
        console.log('Tipo de solicitud:', requestType);
    
        const formData = new FormData();
        formData.append('RequestType', requestType);
    
        const form = document.getElementById(formId);
        const inputs = form.querySelectorAll('input, select, textarea');
        
        console.log('Inputs del formulario:', inputs);
    
        inputs.forEach(input => {
            if (input.type === 'checkbox') {
                formData.append(input.name, input.checked ? '1' : '0');
            } else {
                formData.append(input.name, input.value.trim());
            }
        });
    
        // Mostrar el spinner
        Swal.fire({
            title: 'Procesando',
            text: 'Por favor espere...',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
    
        // Envío usando fetch
        fetch('/submit-form', {
            method: 'POST',
            body: formData
        })
        .then(async response => {
            console.log('Respuesta recibida:', response);
            
            // Verificamos si la respuesta fue exitosa
            if (!response.ok) {
                // Intentamos obtener el mensaje de error
                let errorMessage;
                try {
                    // Intentamos parsear como JSON primero
                    const errorData = await response.json();
                    errorMessage = errorData.message || `Error en la solicitud (${response.status})`;
                } catch (e) {
                    // Si no es JSON, intentamos obtener el texto
                    try {
                        errorMessage = await response.text();
                    } catch (textError) {
                        // Si todo falla, usamos un mensaje genérico
                        errorMessage = `Error del servidor (código: ${response.status})`;
                    }
                }
                throw new Error(errorMessage);
            }
            
            // Si la respuesta es exitosa, intentamos parsear el JSON
            try {
                return await response.json();
            } catch (jsonError) {
                // Si hay un error al parsear el JSON, devolvemos un objeto simple
                return { 
                    success: true, 
                    message: 'Solicitud procesada, pero hubo un error al procesar la respuesta del servidor.' 
                };
            }
        })
        .then(data => {
            console.log('Datos de respuesta:', data);
    
            if (data.success) {
                Swal.fire({
                    title: 'Éxito',
                    text: data.message || 'Solicitud enviada correctamente',
                    icon: 'success',
                    confirmButtonText: 'Aceptar'
                }).then((result) => {
                    if (result.isConfirmed) {
                        form.reset();
                        // Redirección después de 1 segundo
                        setTimeout(() => {
                            window.location.assign("/");
                        }, 1000);
                    }
                });
            } else {
                // Si data.success es false pero llegamos aquí, es porque el servidor devolvió un JSON válido con success=false
                throw new Error(data.message || 'Error al procesar la solicitud');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                title: 'Error',
                text: error.message || 'Error al enviar el formulario',
                icon: 'error',
                confirmButtonText: 'Aceptar'
            });
        });
    }
    
    // Agregar validación al envío del formulario
    document.querySelector('form').addEventListener('submit', validateForm);
});